<?php

namespace App\Console\Commands;

use App\Services\Ai\PoultryAiService;
use App\Services\Notify\TelegramNotifier;
use Illuminate\Console\Command;

class PoultryRunAi extends Command
{
    protected $signature = 'poultry:run-ai {--house=} {--period=daily}';

    protected $description = 'Generate AI report for poultry houses.';

    public function handle(PoultryAiService $service, TelegramNotifier $notifier): int
    {
        $period = $this->option('period') ?? 'daily';
        if (!in_array($period, ['daily', 'weekly'], true)) {
            $this->error('Invalid period. Use daily or weekly.');
            return self::FAILURE;
        }

        $query = \App\Models\House::query();
        if ($this->option('house')) {
            $query->where('id', $this->option('house'));
        }

        $count = 0;
        foreach ($query->get() as $house) {
            $report = $service->generateReport($house, $period);
            $chatId = $house->telegram_chat_id ?: env('TELEGRAM_DEFAULT_CHAT_ID');
            if ($chatId) {
                $label = $period === 'weekly' ? 'Weekly Digest' : 'Daily Summary';
                $notifier->sendMessage($chatId, implode("\n", [
                    'PoultryOps AI ' . $label . ' - ' . $house->name,
                    trim($report->report_text ?? $report->content ?? ''),
                ]));
            }
            $this->info("Generated report {$report->id} for house {$house->name}.");
            $count++;
        }

        if ($count === 0) {
            $this->warn('No houses found.');
        }

        return self::SUCCESS;
    }
}
