<?php

namespace App\Domain\AI;

use Illuminate\Support\Facades\Http;

class AiOrchestrator
{
    public function route(string $message, array $userContext): array
    {
        $key = config('services.openai.key');
        if (!$key) {
            return [
                'reply' => 'AI is not configured. Use slash commands for now.',
                'tool_calls' => [],
            ];
        }

        $tools = [
            'assets.search', 'assets.create', 'assets.update',
            'workorders.list', 'workorders.create', 'workorders.assign', 'workorders.accept', 'workorders.update', 'workorders.pr', 'workorders.approve', 'workorders.add_parts', 'workorders.close', 'workorders.rcfa',
            'parts.search', 'stock.balance', 'meter.create',
            'purchase_requests.create', 'purchase_requests.add_line', 'purchase_requests.submit', 'purchase_requests.approve', 'purchase_requests.reject', 'purchase_requests.edit',
            'purchase_orders.convert', 'purchase_orders.send', 'grn.create',
            'data.list', 'data.get', 'data.export',
        ];

        $system = "You are an AI router for a fleet management assistant. " .
            "Return ONLY JSON with keys: reply, tool_calls. " .
            "tool_calls is an array of {tool, args}. " .
            "Never request delete or destroy operations. " .
            "Available tools: " . implode(', ', $tools) . ". " .
            "Prefer data.list for lists or summaries, data.get for a specific record, and data.export for exports. " .
            "If the user asks for KPI summaries, use data.list with module=kpi. " .
            "If the user asks for assets by status (ex: active), use data.list with module=assets and status=active. " .
            "If the user asks to export, use data.export with format=pdf or format=xlsx. " .
            "If the user asks for a specific number of items (ex: 2 assets), set limit=2. " .
            "For data.list/get/export, use module=assets|workorders|maintenance|inventory|fuel|tyres|telemetry|alerts|kpi|reports|tenants|users|purchase_requests|purchase_orders|vendors|parts|supply. " .
            "Args must use these keys: " .
            "assets.search {tag|q}, assets.create {tag,name,category,meter_type,location}, " .
            "assets.update {tag,name,category,location,meter_type,status,lifecycle_status}, " .
            "workorders.list {asset,status}, workorders.create {asset,title,description,priority,ref,estimated_cost}, " .
            "workorders.assign {wo,user,priority,planned_start,due_at}, workorders.accept {wo}, " .
            "workorders.update {wo,progress,notes,planned_start,due_at,status}, workorders.pr {wo,parts}, workorders.approve {wo}, workorders.add_parts {wo,sku,qty,unit_cost}, " .
            "workorders.close {wo,notes,cause,downtime}, workorders.rcfa {wo,notes}, " .
            "purchase_requests.create {department,cost_center,priority,needed_by,reason}, " .
            "purchase_requests.add_line {pr,sku,description,qty,est_unit_cost,preferred_vendor}, " .
            "purchase_requests.submit {pr}, purchase_requests.approve {pr,notes}, purchase_requests.reject {pr,reason}, " .
            "purchase_requests.edit {pr,line,qty,est_unit_cost,description}, " .
            "purchase_orders.convert {pr,vendor,delivery,terms}, purchase_orders.send {po}, " .
            "grn.create {po,reference,items}, parts.search {q}, stock.balance {sku}, " .
            "meter.create {asset,reading,meter_type}.";

        $payload = [
            'model' => config('services.openai.model'),
            'messages' => [
                ['role' => 'system', 'content' => $system],
                ['role' => 'user', 'content' => $message],
            ],
            'temperature' => 0.2,
            'max_tokens' => 400,
        ];

        $verify = config('services.openai.verify_ssl', true);
        $verify = filter_var($verify, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $verify = $verify ?? true;

        $request = Http::withOptions(['verify' => $verify])
            ->withToken($key)
            ->timeout((int) config('services.openai.timeout', 20));

        if (config('services.openai.organization')) {
            $request = $request->withHeaders([
                'OpenAI-Organization' => config('services.openai.organization'),
            ]);
        }
        if (config('services.openai.project')) {
            $request = $request->withHeaders([
                'OpenAI-Project' => config('services.openai.project'),
            ]);
        }

        try {
            $response = $request->post(config('services.openai.endpoint'), $payload);
        } catch (\Throwable $e) {
            return [
                'reply' => 'AI routing failed. Use slash commands.',
                'tool_calls' => [],
            ];
        }

        if ($response->failed()) {
            return [
                'reply' => 'AI routing failed. Use slash commands.',
                'tool_calls' => [],
            ];
        }

        $content = data_get($response->json(), 'choices.0.message.content', '');
        $decoded = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded)) {
            return [
                'reply' => trim((string) $content),
                'tool_calls' => [],
            ];
        }

        return [
            'reply' => (string) ($decoded['reply'] ?? ''),
            'tool_calls' => $decoded['tool_calls'] ?? [],
        ];
    }
}
