<?php

namespace App\Domain\AI;

use App\Models\Asset;
use App\Models\AuditLog;
use App\Models\DowntimeLog;
use App\Models\GoodsReceipt;
use App\Models\GoodsReceiptItem;
use App\Models\InventoryItem;
use App\Models\MeterReading;
use App\Models\Part;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use App\Models\PurchaseRequestItem;
use App\Models\Site;
use App\Models\User;
use App\Models\Vendor;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Models\WorkOrderDocument;
use App\Models\WorkOrderNote;
use App\Models\WorkOrderPart;
use App\Services\ModuleDataService;
use App\Domain\Telegram\TelegramNotifier;
use App\Services\TelegramExportService;
use App\Models\Tenant;
use App\Models\TelegramLink;
use Illuminate\Support\Str;
use RuntimeException;
use Carbon\Carbon;
use Illuminate\Database\QueryException;

class ToolExecutor
{
    public function __construct(
        private ModuleDataService $moduleData,
        private TelegramNotifier $notifier,
        private TelegramExportService $exportService
    ) {
    }

    public function execute(string $toolName, array $args, array $userContext, string $channel): array
    {
        if ($channel === 'telegram' && $this->isDeniedTool($toolName)) {
            throw new RuntimeException('Deletion is not allowed on Telegram; use the web app.');
        }

        $user = $userContext['user'] ?? null;
        if (!$user instanceof User) {
            throw new RuntimeException('User context missing.');
        }

        $tenantId = $user->tenant_id;
        $roleNames = array_map('strtolower', $userContext['role_names'] ?? []);

        $action = $this->actionForTool($toolName);
        if (in_array($action, ['create', 'update', 'close'], true) && in_array('viewer', $roleNames, true)) {
            throw new RuntimeException('Your role is read-only.');
        }

        try {
            $result = match ($toolName) {
                'assets.search' => $this->assetsSearch($tenantId, $args),
                'assets.create' => $this->assetsCreate($tenantId, $args),
                'assets.update' => $this->assetsUpdate($tenantId, $args),
                'workorders.list' => $this->workOrdersList($tenantId, $args),
                'workorders.create' => $this->workOrdersCreate($tenantId, $args, $user),
                'workorders.assign' => $this->workOrdersAssign($tenantId, $args, $user),
                'workorders.accept' => $this->workOrdersAccept($tenantId, $args, $user),
                'workorders.update' => $this->workOrdersUpdate($tenantId, $args, $user),
                'workorders.pr' => $this->workOrdersCreatePr($tenantId, $args, $user),
                'workorders.approve' => $this->workOrdersApprove($tenantId, $args, $user),
                'workorders.reject' => $this->workOrdersReject($tenantId, $args, $user),
                'workorders.add_parts' => $this->workOrdersAddParts($tenantId, $args, $user),
                'workorders.rcfa' => $this->workOrdersRcfa($tenantId, $args, $user),
                'workorders.close' => $this->workOrdersClose($tenantId, $args),
                'parts.search' => $this->partsSearch($tenantId, $args),
                'stock.balance' => $this->stockBalance($tenantId, $args),
                'meter.create' => $this->meterCreate($tenantId, $args),
                'purchase_requests.create' => $this->purchaseRequestsCreate($tenantId, $args, $user),
                'purchase_requests.add_line' => $this->purchaseRequestsAddLine($tenantId, $args, $user),
                'purchase_requests.submit' => $this->purchaseRequestsSubmit($tenantId, $args, $user),
                'purchase_requests.approve' => $this->purchaseRequestsApprove($tenantId, $args, $user),
                'purchase_requests.reject' => $this->purchaseRequestsReject($tenantId, $args, $user),
                'purchase_requests.edit' => $this->purchaseRequestsEdit($tenantId, $args, $user),
                'purchase_orders.convert' => $this->purchaseOrdersConvert($tenantId, $args, $user),
                'purchase_orders.send' => $this->purchaseOrdersSend($tenantId, $args, $user),
                'grn.create' => $this->goodsReceiptCreate($tenantId, $args, $user),
                'data.list' => $this->moduleList($tenantId, $args),
                'data.get' => $this->moduleGet($tenantId, $args),
                'data.export' => $this->moduleExport($tenantId, $args),
                default => ['message' => 'Unknown tool.'],
            };
        } catch (QueryException $e) {
            $result = ['message' => $this->friendlyDatabaseError($e)];
        }

        $this->logTool($user, $toolName, $args, $result, $channel);
        return $result;
    }

    private function assetsSearch(int $tenantId, array $args): array
    {
        $tag = $args['tag'] ?? null;
        $query = $args['q'] ?? null;

        $builder = Asset::where('tenant_id', $tenantId);
        if ($tag) {
            $builder->where('asset_tag', $tag);
        } elseif ($query) {
            [$filters, $search] = $this->parseSearchFilters($query);
            if (!empty($filters['status'])) {
                $builder->where('status', $filters['status']);
            }
            if (!empty($filters['category'])) {
                $builder->where('category', 'like', '%' . $filters['category'] . '%');
            }
            if (!empty($filters['location'])) {
                $builder->where('location', 'like', '%' . $filters['location'] . '%');
            }
            if ($search) {
                $builder->where(function ($q) use ($search) {
                    $q->where('name', 'like', '%' . $search . '%')
                        ->orWhere('asset_tag', 'like', '%' . $search . '%');
                });
            }
        }

        $assets = $builder->limit(5)->get();

        if ($assets->isEmpty()) {
            return $this->assetNotFoundResponse($tenantId, $tag ?? $query);
        }

        $lines = $assets->map(fn ($asset) => "{$asset->asset_tag} - {$asset->name} ({$asset->location})");
        return ['message' => "Assets:\n" . $lines->implode("\n")];
    }

    private function assetsCreate(int $tenantId, array $args): array
    {
        $tag = $args['tag'] ?? $args['asset_tag'] ?? null;
        $name = $args['name'] ?? null;

        if (!$tag || !$name) {
            return ['message' => 'Missing tag or name.'];
        }

        $asset = Asset::create([
            'tenant_id' => $tenantId,
            'asset_tag' => $tag,
            'name' => $name,
            'category' => $args['category'] ?? null,
            'meter_type' => $args['meter_type'] ?? null,
            'location' => $args['location'] ?? null,
            'status' => 'active',
            'lifecycle_status' => 'active',
            'source' => 'telegram',
        ]);

        return ['message' => 'Created asset ' . $this->assetLabel($asset) . '.'];
    }

    private function assetsUpdate(int $tenantId, array $args): array
    {
        $tag = $args['tag'] ?? $args['asset_tag'] ?? null;
        if (!$tag) {
            return ['message' => 'Missing tag for update.'];
        }

        $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $tag)->first();
        if (!$asset) {
            return $this->assetNotFoundResponse($tenantId, (string) $tag);
        }

        $updates = array_intersect_key($args, array_flip([
            'name', 'category', 'location', 'meter_type', 'status', 'lifecycle_status',
        ]));

        if (!$updates) {
            return ['message' => 'No fields to update.'];
        }

        $asset->fill($updates)->save();
        return ['message' => 'Updated asset ' . $this->assetLabel($asset) . '.'];
    }

    private function workOrdersList(int $tenantId, array $args): array
    {
        $assetTag = $args['asset'] ?? $args['asset_tag'] ?? null;
        $assetId = $args['asset_id'] ?? null;
        $status = $args['status'] ?? null;
        $builder = WorkOrder::where('tenant_id', $tenantId)->with('asset');

        if ($assetId) {
            $asset = is_numeric($assetId)
                ? Asset::where('tenant_id', $tenantId)->where('id', $assetId)->first()
                : Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetId)->first();
            if (!$asset) {
                return $this->assetNotFoundResponse($tenantId, (string) $assetId);
            }
            $builder->where('asset_id', $asset->id);
        } elseif ($assetTag) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetTag)->first();
            if (!$asset) {
                return $this->assetNotFoundResponse($tenantId, (string) $assetTag);
            }
            $builder->where('asset_id', $asset->id);
        }

        if ($status) {
            $builder->where('status', $status);
        }

        $orders = $builder->limit(5)->get();
        if ($orders->isEmpty()) {
            return ['message' => 'No work orders found.'];
        }

        $lines = $orders->map(function ($order) {
            $ref = $order->reference_code ?? ('WO-' . $order->id);
            $asset = $this->assetLabel($order->asset);
            return "{$ref} - {$asset} ({$order->status})";
        });

        return ['message' => "Work Orders:\n" . $lines->implode("\n")];
    }

    private function workOrdersCreate(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['reporter', 'operator', 'planner', 'dispatcher', 'maintenance manager', 'engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to create work orders.'];
        }

        $assetTag = $args['asset'] ?? $args['asset_tag'] ?? null;
        $assetId = $args['asset_id'] ?? null;
        $title = $args['title'] ?? null;
        $description = $args['description'] ?? $args['notes'] ?? null;
        $description = $description ?: $title;

        if ((!$assetTag && !$assetId) || !$description) {
            return ['message' => 'Missing asset or title.'];
        }

        if ($assetId) {
            $asset = is_numeric($assetId)
                ? Asset::where('tenant_id', $tenantId)->where('id', $assetId)->first()
                : Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetId)->first();
        } else {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetTag)->first();
        }
        if (!$asset) {
            return $this->assetNotFoundResponse($tenantId, (string) ($assetTag ?? $assetId));
        }

        $estimatedCost = $this->numericValue($args['estimated_cost'] ?? $args['est_cost'] ?? null);
        $priority = $args['priority'] ?? 'medium';

        $order = WorkOrder::create([
            'tenant_id' => $tenantId,
            'asset_id' => $asset->id,
            'reference_code' => $args['ref'] ?? strtoupper('WO-' . Str::random(6)),
            'source' => 'telegram',
            'status' => 'pending_approval',
            'priority' => $priority,
            'description' => $description,
            'reported_by' => $user->id,
            'estimated_cost' => $estimatedCost,
            'requires_approval' => true,
            'approval_status' => 'pending',
            'last_reminded_at' => now(),
        ]);

        $parts = $this->normalizeWorkOrderParts($args);
        $pr = null;
        if (!empty($parts)) {
            $pr = $this->createPrForWorkOrder($tenantId, $order, $user, $parts);
        }

        $this->notifyWoStatus($order, "WO {$order->reference_code} created (pending Engineering approval).");

        $message = "WO created: {$order->reference_code}. Reply APPROVE {$order->reference_code} or REJECT {$order->reference_code} reason=\"...\".";
        if ($pr) {
            $message .= " PR {$pr->request_code} created for required parts.";
        }

        return array_filter([
            'message' => $message,
            'work_order_ref' => $order->reference_code,
            'file_path' => null,
            'filename' => null,
        ]);
    }

    private function workOrdersAssign(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['planner', 'dispatcher', 'maintenance manager', 'engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to assign work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        if ($order->approval_status === 'pending' && !$this->hasAnyRole($user, ['maintenance manager', 'admin'])) {
            return ['message' => "WO {$order->reference_code} requires manager approval. Reply APPROVE {$order->reference_code}."];
        }

        $assigneeRef = $args['user'] ?? $args['assignee'] ?? null;
        if (!$assigneeRef) {
            return ['message' => 'Provide user= for assignment.'];
        }

        $assignee = $this->resolveUserReference($tenantId, $assigneeRef);
        if (!$assignee) {
            return ['message' => 'Assignee not found.'];
        }

        $priority = $args['priority'] ?? $order->priority;
        $plannedStart = $this->dateTimeValue($args['planned_start'] ?? $args['start'] ?? null);
        $dueAt = $this->dateTimeValue($args['due'] ?? $args['due_at'] ?? null);

        $order->update([
            'assigned_to' => $assignee->id,
            'priority' => $priority,
            'planned_start' => $plannedStart,
            'due_at' => $dueAt,
            'status' => $order->status === 'open' ? 'scheduled' : $order->status,
            'last_reminded_at' => now(),
        ]);

        $message = "{$order->reference_code} assigned to {$assignee->name}. Priority: " . ucfirst($priority) . '.';
        if ($dueAt) {
            $message .= ' Due: ' . Carbon::parse($dueAt)->toDateString() . '.';
        }
        $message .= ' Use the buttons below to accept, schedule, create a PR, or close.';

        $this->notifyWoStatus($order, "WO {$order->reference_code} assigned to {$assignee->name}.");

        return ['message' => $message];
    }

    private function workOrdersAccept(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['technician', 'artisan', 'maintenance', 'admin'])) {
            return ['message' => 'You do not have permission to accept work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        if ($order->assigned_to && $order->assigned_to !== $user->id) {
            if (!$this->hasAnyRole($user, ['admin', 'maintenance manager', 'engineering manager', 'planner'])) {
                return ['message' => 'This work order is assigned to another user.'];
            }
        }

        if ($order->approval_status === 'pending') {
            return ['message' => "WO {$order->reference_code} awaiting approval. Reply APPROVE {$order->reference_code} (manager only)."];
        }

        $order->update([
            'status' => 'in_progress',
            'accepted_at' => now(),
            'started_at' => $order->started_at ?: now(),
            'last_reminded_at' => now(),
        ]);

        WorkOrderNote::create([
            'tenant_id' => $tenantId,
            'work_order_id' => $order->id,
            'created_by' => $user->id,
            'type' => 'status',
            'note' => 'Accepted by technician.',
        ]);

        $this->notifyWoStatus($order, "WO {$order->reference_code} accepted and in progress.");

        return ['message' => "Work order {$order->reference_code} is now in progress."];
    }

    private function workOrdersUpdate(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['technician', 'artisan', 'maintenance', 'engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to update work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $progress = $this->numericValue($args['progress'] ?? $args['progress_percent'] ?? null);
        $note = $args['note'] ?? $args['notes'] ?? null;
        $status = $args['status'] ?? null;
        $plannedStart = $this->dateTimeValue($args['planned_start'] ?? $args['start'] ?? null);
        $dueAt = $this->dateTimeValue($args['due_at'] ?? $args['due'] ?? null);

        $updates = [];
        if ($progress !== null) {
            $updates['progress_percent'] = $progress;
            if (in_array($order->status, ['open', 'scheduled'], true)) {
                $updates['status'] = 'in_progress';
            }
        }

        if ($status) {
            if ($this->hasAnyRole($user, ['engineering manager', 'maintenance manager', 'admin'])) {
                $updates['status'] = $status;
            } elseif ($status === 'scheduled' && $order->assigned_to === $user->id) {
                $updates['status'] = $status;
            }
        }

        if (($plannedStart || $dueAt) && ($order->assigned_to === $user->id || $this->hasAnyRole($user, ['planner', 'maintenance manager', 'engineering manager', 'admin']))) {
            if ($plannedStart) {
                $updates['planned_start'] = $plannedStart;
            }
            if ($dueAt) {
                $updates['due_at'] = $dueAt;
            }
            if (in_array($order->status, ['open', 'scheduled'], true)) {
                $updates['status'] = $updates['status'] ?? 'scheduled';
            }
        }

        if (($updates['status'] ?? null) === 'scheduled' && empty($updates['planned_start'])) {
            $updates['planned_start'] = $order->planned_start ?? now();
        }

        if ($this->hasAnyRole($user, ['engineering manager', 'maintenance manager', 'admin'])) {
            if (!empty($args['description'])) {
                $updates['description'] = $args['description'];
            }
            if (!empty($args['priority'])) {
                $updates['priority'] = $args['priority'];
            }
            if (!empty($args['estimated_cost'])) {
                $updates['estimated_cost'] = $this->numericValue($args['estimated_cost']);
            }
        }

        if ($updates || $note || $progress !== null) {
            $updates['last_reminded_at'] = now();
        }

        if ($updates) {
            $order->update($updates);
        }

        if ($note || $progress !== null) {
            WorkOrderNote::create([
                'tenant_id' => $tenantId,
                'work_order_id' => $order->id,
                'created_by' => $user->id,
                'type' => $progress !== null ? 'progress' : 'comment',
                'note' => $note,
                'progress_percent' => $progress,
            ]);
        }

        if (!empty($updates['status'])) {
            $this->notifyWoStatus($order, "WO {$order->reference_code} status updated to {$updates['status']}.");
        }

        if ($note || $progress !== null) {
            $message = "WO {$order->reference_code} update by {$user->name}.";
            if ($progress !== null) {
                $message .= " Progress {$progress}%.";
            }
            if ($note) {
                $message .= " Notes: {$note}";
            }
            $this->notifyWoParticipants($order, $message, $user);
        }

        return ['message' => "Work order {$order->reference_code} updated."];
    }

    private function workOrdersCreatePr(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['technician', 'artisan', 'maintenance', 'planner', 'maintenance manager', 'engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to create PRs for work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $parts = $this->normalizeWorkOrderParts($args);
        if (empty($parts)) {
            $pr = $this->createDraftPrForWorkOrder($tenantId, $order, $user);
            if (!$pr) {
                return ['message' => 'Unable to create a draft PR for this work order.'];
            }

            $this->notifyWoParticipants($order, "Draft PR {$pr->request_code} created for WO {$order->reference_code} by {$user->name}.", $user);

            return ['message' => "Draft PR {$pr->request_code} created for WO {$order->reference_code}. Add lines with /pr add or /pr edit, then /pr submit."];
        }

        $pr = $this->createPrForWorkOrder($tenantId, $order, $user, $parts);
        if (!$pr) {
            return ['message' => 'Unable to create PR for this work order.'];
        }

        $this->notifyWoParticipants($order, "PR {$pr->request_code} created for WO {$order->reference_code} by {$user->name}.", $user);

        return ['message' => "PR {$pr->request_code} created for WO {$order->reference_code}."];
    }

    private function workOrdersApprove(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to approve work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $order->update([
            'requires_approval' => true,
            'approval_status' => 'approved',
            'approved_by' => $user->id,
            'approved_at' => now(),
            'status' => $order->status === 'pending_approval' ? 'open' : $order->status,
            'last_reminded_at' => now(),
        ]);

        $this->notifyWoStatus($order, "WO {$order->reference_code} approved by Engineering Manager.");
        return array_filter([
            'message' => "Work order {$order->reference_code} approved.",
            'file_path' => null,
            'filename' => null,
        ]);
    }

    private function workOrdersReject(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['engineering manager', 'admin'])) {
            return ['message' => 'You do not have permission to reject work orders.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $reason = $args['reason'] ?? $args['notes'] ?? null;
        $order->update([
            'approval_status' => 'rejected',
            'status' => 'rejected',
            'last_reminded_at' => now(),
        ]);

        if ($reason) {
            WorkOrderNote::create([
                'tenant_id' => $tenantId,
                'work_order_id' => $order->id,
                'created_by' => $user->id,
                'type' => 'comment',
                'note' => $reason,
            ]);
        }

        $this->notifyWoStatus($order, "WO {$order->reference_code} rejected. Reason: " . ($reason ?: 'Not provided.'));

        return ['message' => "Work order {$order->reference_code} rejected."];
    }

    private function workOrdersAddParts(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['planner', 'dispatcher', 'maintenance manager', 'admin'])) {
            return ['message' => 'You do not have permission to add parts.'];
        }

        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $sku = $args['sku'] ?? null;
        if (!$sku) {
            return ['message' => 'Provide sku= for the part.'];
        }

        $part = Part::where('tenant_id', $tenantId)->where('sku', $sku)->first();
        if (!$part) {
            return ['message' => 'Part not found.'];
        }

        $qty = $this->numericValue($args['qty'] ?? $args['quantity'] ?? 1) ?? 1;
        $unitCost = $this->numericValue($args['unit_cost'] ?? $args['cost'] ?? null);
        $totalCost = $unitCost !== null ? $unitCost * $qty : null;

        WorkOrderPart::create([
            'tenant_id' => $tenantId,
            'work_order_id' => $order->id,
            'part_id' => $part->id,
            'quantity' => $qty,
            'unit_cost' => $unitCost,
            'total_cost' => $totalCost,
        ]);

        return ['message' => "Added part {$part->sku} x{$qty} to {$order->reference_code}."];
    }

    private function workOrdersRcfa(int $tenantId, array $args, User $user): array
    {
        $order = $this->resolveWorkOrder($tenantId, $args);
        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $args['wo'] ?? $args['ref'] ?? null);
        }

        $note = $args['note'] ?? $args['notes'] ?? null;
        if (!$note) {
            return ['message' => 'Provide notes= for RCFA.'];
        }

        WorkOrderNote::create([
            'tenant_id' => $tenantId,
            'work_order_id' => $order->id,
            'created_by' => $user->id,
            'type' => 'rcfa',
            'note' => $note,
        ]);

        return ['message' => "RCFA captured for {$order->reference_code}."];
    }

    private function workOrdersClose(int $tenantId, array $args): array
    {
        $reference = $args['wo'] ?? $args['ref'] ?? null;
        if (!$reference) {
            return ['message' => 'Missing work order reference.'];
        }

        $order = WorkOrder::where('tenant_id', $tenantId)
            ->where('reference_code', $reference)
            ->first();

        if (!$order) {
            return $this->workOrderNotFoundResponse($tenantId, $reference);
        }

        $hasEvidence = WorkOrderDocument::where('tenant_id', $tenantId)
            ->where('work_order_id', $order->id)
            ->exists();

        if (!$hasEvidence) {
            return ['message' => 'Evidence required. Upload at least one photo or job card before closing.'];
        }

        $resolution = $args['notes'] ?? null;
        $cause = $args['cause'] ?? null;
        $downtime = $this->numericValue($args['downtime'] ?? null);

        $order->update([
            'status' => 'completed',
            'resolution' => $resolution,
            'cause' => $cause,
            'completed_at' => now(),
            'progress_percent' => $order->progress_percent ?? 100,
            'last_reminded_at' => now(),
        ]);

        if ($resolution || $cause) {
            WorkOrderNote::create([
                'tenant_id' => $tenantId,
                'work_order_id' => $order->id,
                'created_by' => $args['closed_by'] ?? null,
                'type' => 'closure',
                'note' => $resolution ?: $cause,
            ]);
        }

        if ($downtime !== null && $downtime > 0) {
            DowntimeLog::create([
                'tenant_id' => $tenantId,
                'asset_id' => $order->asset_id,
                'work_order_id' => $order->id,
                'started_at' => now()->subMinutes((int) ($downtime * 60)),
                'ended_at' => now(),
                'duration_minutes' => (int) ($downtime * 60),
                'notes' => 'Logged via Telegram close.',
            ]);
        }

        $partsCost = WorkOrderPart::where('tenant_id', $tenantId)
            ->where('work_order_id', $order->id)
            ->sum('total_cost');

        $laborCost = WorkOrderCost::where('tenant_id', $tenantId)
            ->where('work_order_id', $order->id)
            ->sum('total_cost');

        $totalCost = (float) $partsCost + (float) $laborCost;
        if ($totalCost > 0) {
            $order->update(['total_cost' => $totalCost]);
        }

        $message = "WO {$reference} closed. Cost summary: Parts $" . number_format((float) $partsCost, 2) .
            ", Labour $" . number_format((float) $laborCost, 2) . ".";
        if ($downtime !== null) {
            $message .= " Downtime {$downtime}h.";
        }

        if ($cause) {
            $repeat = WorkOrder::where('tenant_id', $tenantId)
                ->where('asset_id', $order->asset_id)
                ->where('cause', $cause)
                ->where('id', '!=', $order->id)
                ->where('completed_at', '>=', now()->subDays(30))
                ->exists();
            if ($repeat) {
                $message .= " Repeat failure detected. Reply RCFA {$reference} notes=\"...\".";
            }
        }

        $this->notifyWoStatus($order, "WO {$reference} closed.");

        return ['message' => $message];
    }

    private function purchaseRequestsCreate(int $tenantId, array $args, User $user): array
    {
        $code = $this->nextReference($tenantId, 'PR', 'purchase_requests', 'request_code');
        $autoSubmit = $this->boolValue($args['auto_submit'] ?? $args['submit'] ?? null);

        $request = PurchaseRequest::create([
            'tenant_id' => $tenantId,
            'request_code' => $code,
            'title' => $args['title'] ?? ($args['reason'] ?? 'Telegram PR'),
            'department' => $args['department'] ?? $args['dept'] ?? null,
            'cost_center' => $args['cost_center'] ?? null,
            'status' => $autoSubmit ? 'submitted' : 'draft',
            'priority' => $args['priority'] ?? 'medium',
            'requested_by' => $user->id,
            'needed_by' => $this->dateValue($args['needed_by'] ?? null),
            'submitted_at' => $autoSubmit ? now() : null,
            'currency' => $args['currency'] ?? 'USD',
            'notes' => $args['reason'] ?? $args['notes'] ?? null,
            'last_reminded_at' => $autoSubmit ? now() : null,
        ]);

        $pdf = $this->exportPurchaseRequestPdf($request);
        $message = "{$request->request_code} created. Reply /pr add {$request->request_code} sku=... qty=... est=... or SUBMIT {$request->request_code}.";

        if ($autoSubmit) {
            $approvers = $this->buildPurchaseRequestApprovals($tenantId, $request);
            $first = $approvers->first();
            if ($first?->approver_id) {
                $this->notifyApprovalRequest($request, $first);
            }
            $this->notifyPrStatus($request, "PR {$request->request_code} submitted for approval.");
            $this->notifyEngineeringManager($tenantId, "PR {$request->request_code} submitted for approval.", $pdf);
            $message = "PR {$request->request_code} submitted for approval.";
        } else {
            $this->notifyPrStatus($request, "PR {$request->request_code} created (draft).");
            $this->notifyEngineeringManager($tenantId, "PR {$request->request_code} created (draft).", $pdf);
        }

        return array_filter([
            'message' => $message,
            'file_path' => $pdf['path'] ?? null,
            'filename' => $pdf['filename'] ?? null,
        ]);
    }

    private function purchaseRequestsAddLine(int $tenantId, array $args, User $user): array
    {
        $code = $args['pr'] ?? $args['request'] ?? null;
        if (!$code) {
            return ['message' => 'Provide PR code (pr=PR-00001).'];
        }

        $request = PurchaseRequest::where('tenant_id', $tenantId)
            ->where('request_code', $code)
            ->first();
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        $sku = $args['sku'] ?? null;
        $name = $args['description'] ?? $args['item'] ?? $args['name'] ?? null;
        if (!$sku && !$name) {
            return ['message' => 'Provide sku= or description=.'];
        }

        $part = $this->findPart($tenantId, $sku, $name);
        $quantity = $this->numericValue($args['qty'] ?? $args['quantity'] ?? 1) ?? 1;
        $estUnit = $this->numericValue($args['est'] ?? $args['est_unit_cost'] ?? null);
        $lineNumber = (int) PurchaseRequestItem::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->max('line_number') + 1;

        PurchaseRequestItem::create([
            'tenant_id' => $tenantId,
            'purchase_request_id' => $request->id,
            'line_number' => $lineNumber,
            'part_id' => $part?->id,
            'sku' => $part?->sku ?? $sku,
            'item_name' => $part?->name ?? $name,
            'description' => $name,
            'quantity' => $quantity,
            'unit' => $args['uom'] ?? $args['unit'] ?? null,
            'est_unit_cost' => $estUnit,
            'preferred_vendor' => $args['vendor'] ?? $args['preferred_vendor'] ?? null,
        ]);

        $request->update(['total_estimated_cost' => $this->sumPurchaseRequestCost($request)]);

        return ['message' => "Line added to {$request->request_code}. Reply SUBMIT {$request->request_code} when ready."];
    }

    private function purchaseRequestsSubmit(int $tenantId, array $args, User $user): array
    {
        $request = $this->findPurchaseRequest($tenantId, $args);
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        $request->update([
            'status' => 'submitted',
            'submitted_at' => now(),
            'total_estimated_cost' => $this->sumPurchaseRequestCost($request),
            'last_reminded_at' => now(),
        ]);

        $approvers = $this->buildPurchaseRequestApprovals($tenantId, $request);
        $first = $approvers->first();
        if ($first?->approver_id) {
            $this->notifyApprovalRequest($request, $first);
        }

        $this->notifyPrStatus($request, "PR {$request->request_code} submitted for approval.");

        return ['message' => "PR {$request->request_code} submitted for approval."];
    }

    private function purchaseRequestsApprove(int $tenantId, array $args, User $user): array
    {
        $request = $this->findPurchaseRequest($tenantId, $args);
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        $approval = PurchaseRequestApproval::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->where('status', 'pending')
            ->orderBy('step')
            ->first();

        if (!$approval) {
            return ['message' => 'No pending approval found.'];
        }

        if ($approval->approver_id && $approval->approver_id !== $user->id && !$this->hasAnyRole($user, ['admin'])) {
            return ['message' => 'You are not the assigned approver.'];
        }

        $approval->update([
            'status' => 'approved',
            'notes' => $args['notes'] ?? $args['comment'] ?? $args['reason'] ?? null,
            'decided_at' => now(),
        ]);
        $stepLabel = ((int) $approval->step === 1) ? 'Engineering Manager' : 'Finance Manager';

        $next = PurchaseRequestApproval::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->where('status', 'pending')
            ->orderBy('step')
            ->first();

        if ($next) {
            if ($next->approver_id) {
                $this->notifyApprovalRequest($request, $next);
            }
            $request->update(['last_reminded_at' => now()]);
            $this->notifyPrStatus($request, "PR {$request->request_code} approved by {$stepLabel}. Awaiting Finance approval.");
            return ['message' => "PR {$request->request_code} approved. Next approver notified."];
        }

        $request->update([
            'status' => 'approved',
            'approved_at' => now(),
            'last_reminded_at' => now(),
        ]);

        $this->notifyProcurement($request);
        $this->notifyPrStatus($request, "PR {$request->request_code} approved by {$stepLabel}. Final approval complete.");

        $pdf = $this->exportPurchaseRequestPdf($request);
        $this->notifyEngineeringManager($tenantId, "PR {$request->request_code} approved. PDF updated.", $pdf);
        if ($request->requested_by && $pdf && !empty($pdf['path'])) {
            $this->notifier->notifyUserDocument($request->requested_by, $pdf['path'], $pdf['filename'] ?? null, 'PR PDF');
        }

        return array_filter([
            'message' => "PR {$request->request_code} fully approved. Procurement notified.",
            'file_path' => $pdf['path'] ?? null,
            'filename' => $pdf['filename'] ?? null,
        ]);
    }

    private function purchaseRequestsReject(int $tenantId, array $args, User $user): array
    {
        $request = $this->findPurchaseRequest($tenantId, $args);
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        $approval = PurchaseRequestApproval::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->where('status', 'pending')
            ->orderBy('step')
            ->first();

        if ($approval && $approval->approver_id && $approval->approver_id !== $user->id && !$this->hasAnyRole($user, ['admin'])) {
            return ['message' => 'You are not the assigned approver.'];
        }

        if ($approval) {
            $approval->update([
                'status' => 'rejected',
                'notes' => $args['reason'] ?? $args['notes'] ?? null,
                'decided_at' => now(),
            ]);
        }

        $request->update(['status' => 'rejected', 'last_reminded_at' => now()]);
        $reason = $args['reason'] ?? 'Not provided.';
        $this->notifyPrStatus($request, "PR {$request->request_code} rejected. Reason: {$reason}");

        return ['message' => "PR {$request->request_code} rejected."];
    }

    private function purchaseRequestsEdit(int $tenantId, array $args, User $user): array
    {
        $request = $this->findPurchaseRequest($tenantId, $args);
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        if (!in_array($request->status, ['draft', 'rejected', 'submitted'], true) && !$this->hasAnyRole($user, ['engineering manager', 'finance manager', 'admin'])) {
            return ['message' => 'Only draft/rejected PRs can be edited unless you are an approver.'];
        }

        $line = $args['line'] ?? $args['line_number'] ?? null;
        if (!$line) {
            return ['message' => 'Provide line= number to edit.'];
        }

        $item = PurchaseRequestItem::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->where('line_number', (int) $line)
            ->first();

        if (!$item) {
            return ['message' => 'Line item not found.'];
        }

        $updates = array_intersect_key($args, array_flip([
            'quantity', 'qty', 'unit', 'uom', 'est_unit_cost', 'est', 'description', 'item_name', 'preferred_vendor',
        ]));

        $normalized = [];
        if (isset($updates['qty'])) {
            $normalized['quantity'] = $this->numericValue($updates['qty']);
        }
        if (isset($updates['quantity'])) {
            $normalized['quantity'] = $this->numericValue($updates['quantity']);
        }
        if (isset($updates['unit']) || isset($updates['uom'])) {
            $normalized['unit'] = $updates['unit'] ?? $updates['uom'];
        }
        if (isset($updates['est_unit_cost']) || isset($updates['est'])) {
            $normalized['est_unit_cost'] = $this->numericValue($updates['est_unit_cost'] ?? $updates['est']);
        }
        if (isset($updates['description'])) {
            $normalized['description'] = $updates['description'];
        }
        if (isset($updates['item_name'])) {
            $normalized['item_name'] = $updates['item_name'];
        }
        if (isset($updates['preferred_vendor'])) {
            $normalized['preferred_vendor'] = $updates['preferred_vendor'];
        }

        if (!$normalized) {
            return ['message' => 'No fields to update.'];
        }

        $item->update($normalized);
        $request->update(['total_estimated_cost' => $this->sumPurchaseRequestCost($request)]);

        return ['message' => "PR {$request->request_code} updated. Reply SUBMIT {$request->request_code} to resubmit."];
    }

    private function purchaseOrdersConvert(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['procurement', 'buyer', 'admin'])) {
            return ['message' => 'You do not have permission to convert PRs to POs.'];
        }

        $request = $this->findPurchaseRequest($tenantId, $args);
        if (!$request) {
            return ['message' => 'Purchase request not found.'];
        }

        if ($request->status !== 'approved') {
            return ['message' => 'PR must be approved before converting to PO.'];
        }

        $poNumber = $this->nextReference($tenantId, 'PO', 'purchase_orders', 'po_number');
        $vendorName = $args['vendor'] ?? $args['vendor_name'] ?? null;
        $vendorId = null;
        if ($vendorName) {
            $vendor = Vendor::where('tenant_id', $tenantId)->where('name', 'like', '%' . $vendorName . '%')->first();
            $vendorId = $vendor?->id;
            $vendorName = $vendor?->name ?? $vendorName;
        }

        $order = PurchaseOrder::create([
            'tenant_id' => $tenantId,
            'purchase_request_id' => $request->id,
            'vendor_id' => $vendorId,
            'po_number' => $poNumber,
            'vendor_name' => $vendorName,
            'status' => 'draft',
            'terms' => $args['terms'] ?? null,
            'delivery_location' => $args['delivery'] ?? $args['delivery_location'] ?? null,
            'total_estimated_cost' => $request->total_estimated_cost,
            'currency' => $request->currency,
            'created_by' => $user->id,
        ]);

        foreach ($request->items as $item) {
            PurchaseOrderItem::create([
                'tenant_id' => $tenantId,
                'purchase_order_id' => $order->id,
                'part_id' => $item->part_id,
                'sku' => $item->sku,
                'description' => $item->description ?? $item->item_name,
                'quantity' => $item->quantity,
                'unit' => $item->unit,
                'unit_cost' => $item->est_unit_cost,
                'total_cost' => $item->est_unit_cost !== null ? $item->est_unit_cost * (float) $item->quantity : null,
            ]);
        }

        return ['message' => "{$order->po_number} created for {$order->vendor_name}. Reply SEND {$order->po_number}."];
    }

    private function purchaseOrdersSend(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['procurement', 'buyer', 'admin'])) {
            return ['message' => 'You do not have permission to send POs.'];
        }

        $poNumber = $args['po'] ?? $args['po_number'] ?? null;
        if (!$poNumber) {
            return ['message' => 'Provide PO number.'];
        }

        $order = PurchaseOrder::where('tenant_id', $tenantId)
            ->where('po_number', $poNumber)
            ->with('items')
            ->first();

        if (!$order) {
            return ['message' => 'Purchase order not found.'];
        }

        $headers = ['SKU', 'Description', 'Qty', 'Unit', 'Unit Cost', 'Total'];
        $rows = $order->items->map(fn ($item) => [
            $item->sku ?? '-',
            $item->description ?? '-',
            $item->quantity ?? '-',
            $item->unit ?? '-',
            $item->unit_cost ?? '-',
            $item->total_cost ?? '-',
        ])->all();

        $title = "{$order->po_number} - Purchase Order";
        $tenant = \App\Models\Tenant::find($tenantId);
        $exporter = app(\App\Services\TelegramExportService::class);
        $file = $exporter->exportTable($title, $headers, $rows, $tenant, 'pdf');

        $order->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        return [
            'message' => "PO {$order->po_number} ready to send.",
            'file_path' => $file['path'] ?? null,
            'filename' => $file['filename'] ?? null,
        ];
    }

    private function goodsReceiptCreate(int $tenantId, array $args, User $user): array
    {
        if (!$this->hasAnyRole($user, ['stores', 'storekeeper', 'inventory', 'admin'])) {
            return ['message' => 'You do not have permission to receive goods.'];
        }

        $poNumber = $args['po'] ?? $args['po_number'] ?? null;
        $reference = $args['reference'] ?? $args['grn'] ?? null;
        if (!$reference) {
            return ['message' => 'Provide GRN reference.'];
        }

        $order = null;
        if ($poNumber) {
            $order = PurchaseOrder::where('tenant_id', $tenantId)
                ->where('po_number', $poNumber)
                ->first();
        }

        $receipt = GoodsReceipt::firstOrCreate([
            'tenant_id' => $tenantId,
            'reference' => $reference,
        ], [
            'purchase_order_id' => $order?->id,
            'received_at' => now(),
            'received_by' => $user->id,
            'notes' => $args['notes'] ?? null,
        ]);

        $items = $args['items'] ?? [];
        if (empty($items)) {
            return ['message' => "GRN {$reference} created. Upload delivery note to capture line items."];
        }
        foreach ($items as $item) {
            $part = $this->findPart($tenantId, $item['sku'] ?? null, $item['description'] ?? null);
            $qty = $this->numericValue($item['qty'] ?? $item['quantity'] ?? null);
            if ($qty === null) {
                continue;
            }

            GoodsReceiptItem::create([
                'tenant_id' => $tenantId,
                'goods_receipt_id' => $receipt->id,
                'part_id' => $part?->id,
                'sku' => $part?->sku ?? ($item['sku'] ?? null),
                'description' => $item['description'] ?? null,
                'quantity' => $qty,
                'unit' => $item['unit'] ?? null,
                'unit_cost' => $this->numericValue($item['unit_cost'] ?? null),
            ]);

            if ($part) {
                $inventory = InventoryItem::firstOrCreate(
                    [
                        'tenant_id' => $tenantId,
                        'part_id' => $part->id,
                    ],
                    [
                        'quantity' => 0,
                    ]
                );
                $inventory->update(['quantity' => (float) $inventory->quantity + (float) $qty]);
            }
        }

        return ['message' => "GRN {$reference} recorded. Stock updated."];
    }

    private function partsSearch(int $tenantId, array $args): array
    {
        $query = $args['q'] ?? null;
        if (!$query) {
            return ['message' => 'Provide q= to search parts.'];
        }

        $parts = Part::where('tenant_id', $tenantId)
            ->where(function ($builder) use ($query) {
                $builder->where('name', 'like', '%' . $query . '%')
                    ->orWhere('sku', 'like', '%' . $query . '%');
            })
            ->limit(5)
            ->get();

        if ($parts->isEmpty()) {
            return ['message' => 'No parts found.'];
        }

        $lines = $parts->map(fn ($part) => "{$part->sku} - {$part->name}");
        return ['message' => "Parts:\n" . $lines->implode("\n")];
    }

    private function stockBalance(int $tenantId, array $args): array
    {
        $sku = $args['sku'] ?? null;
        if (!$sku) {
            return ['message' => 'Provide sku= for stock balance.'];
        }

        $item = InventoryItem::with('part')
            ->where('tenant_id', $tenantId)
            ->whereHas('part', function ($builder) use ($sku) {
                $builder->where('sku', $sku);
            })
            ->first();

        if (!$item) {
            $suggestions = $this->suggestSkus($tenantId, $sku);
            if ($suggestions->isNotEmpty()) {
                $lines = $suggestions->map(fn ($part) => "{$part->sku} - {$part->name}");
                return ['message' => "Stock item not found. Try:\n" . $lines->implode("\n")];
            }
            return ['message' => 'Stock item not found. Try /inventory to list SKUs.'];
        }

        $partName = $item->part?->name ?? $sku;
        return ['message' => "{$sku} ({$partName}) balance: {$item->quantity}"];
    }

    private function meterCreate(int $tenantId, array $args): array
    {
        $assetTag = $args['asset'] ?? $args['asset_tag'] ?? null;
        $assetId = $args['asset_id'] ?? null;
        $reading = $args['reading'] ?? null;

        if ((!$assetTag && !$assetId) || $reading === null) {
            return ['message' => 'Provide asset= and reading=.'];
        }

        if ($assetId) {
            $asset = is_numeric($assetId)
                ? Asset::where('tenant_id', $tenantId)->where('id', $assetId)->first()
                : Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetId)->first();
        } else {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetTag)->first();
        }
        if (!$asset) {
            return $this->assetNotFoundResponse($tenantId, (string) ($assetTag ?? $assetId));
        }

        MeterReading::create([
            'tenant_id' => $tenantId,
            'asset_id' => $asset->id,
            'meter_type' => $args['meter_type'] ?? 'hours',
            'value' => (float) $reading,
            'source' => 'telegram',
            'recorded_at' => now(),
        ]);

        return ['message' => 'Meter reading logged for ' . $this->assetLabel($asset) . '.'];
    }

    private function moduleList(int $tenantId, array $args): array
    {
        $module = $args['module'] ?? null;
        if (!$module) {
            return ['message' => 'Provide module= for list.'];
        }

        return $this->moduleData->list($module, $args, $tenantId);
    }

    private function moduleGet(int $tenantId, array $args): array
    {
        $module = $args['module'] ?? null;
        if (!$module) {
            return ['message' => 'Provide module= for get.'];
        }

        return $this->moduleData->get($module, $args, $tenantId);
    }

    private function moduleExport(int $tenantId, array $args): array
    {
        $module = $args['module'] ?? null;
        if (!$module) {
            return ['message' => 'Provide module= for export.'];
        }

        return $this->moduleData->export($module, $args, $tenantId);
    }

    private function resolveWorkOrder(int $tenantId, array $args): ?WorkOrder
    {
        $reference = $args['wo'] ?? $args['ref'] ?? null;
        if ($reference) {
            return WorkOrder::where('tenant_id', $tenantId)
                ->where('reference_code', $reference)
                ->first();
        }

        if (!empty($args['id'])) {
            return WorkOrder::where('tenant_id', $tenantId)->where('id', $args['id'])->first();
        }

        return null;
    }

    private function resolveUserReference(int $tenantId, string $reference): ?User
    {
        $reference = trim($reference);
        $reference = ltrim($reference, '@');

        if (is_numeric($reference)) {
            return User::where('tenant_id', $tenantId)->where('id', (int) $reference)->first();
        }

        if (str_contains($reference, '@')) {
            return User::where('tenant_id', $tenantId)->where('email', $reference)->first();
        }

        return User::where('tenant_id', $tenantId)
            ->where(function ($builder) use ($reference) {
                $builder->where('name', 'like', '%' . $reference . '%')
                    ->orWhere('email', 'like', '%' . $reference . '%');
            })
            ->first();
    }

    private function evaluateApprovalRequirement(string $priority, ?float $estimatedCost): array
    {
        $threshold = (float) config('work_orders.approval_threshold', 2500);
        $requires = strtolower($priority) === 'high' || ($estimatedCost !== null && $estimatedCost > $threshold);

        return [
            'requires_approval' => $requires,
            'approval_status' => $requires ? 'pending' : null,
            'status' => $requires ? 'pending_approval' : 'open',
        ];
    }

    private function findPurchaseRequest(int $tenantId, array $args): ?PurchaseRequest
    {
        $code = $args['pr'] ?? $args['request'] ?? $args['request_code'] ?? null;
        if (!$code) {
            return null;
        }

        return PurchaseRequest::where('tenant_id', $tenantId)
            ->where('request_code', $code)
            ->first();
    }

    private function buildPurchaseRequestApprovals(int $tenantId, PurchaseRequest $request)
    {
        $existing = PurchaseRequestApproval::where('tenant_id', $tenantId)
            ->where('purchase_request_id', $request->id)
            ->orderBy('step')
            ->get();

        if ($existing->isNotEmpty()) {
            return $existing;
        }

        $steps = [
            1 => $this->firstUserWithRole($tenantId, ['engineering manager', 'engineering', 'eng manager']),
            2 => $this->firstUserWithRole($tenantId, ['finance manager', 'finance', 'accounts']),
        ];

        $created = collect();
        foreach ($steps as $step => $approver) {
            $created->push(PurchaseRequestApproval::create([
                'tenant_id' => $tenantId,
                'purchase_request_id' => $request->id,
                'step' => $step,
                'approver_id' => $approver?->id,
                'status' => 'pending',
            ]));
        }

        return $created;
    }

    private function notifyApprovalRequest(PurchaseRequest $request, PurchaseRequestApproval $approval): void
    {
        if (!$approval->approver_id) {
            return;
        }

        $lines = $request->items()->limit(2)->get()->map(function ($item) {
            return ($item->sku ?? $item->item_name ?? 'Item') . ' x' . ($item->quantity ?? 0);
        })->implode('; ');

        $message = "PR {$request->request_code} ({$request->department}) est " .
            '$' . number_format((float) $request->total_estimated_cost, 2) .
            ' | Needed by ' . ($request->needed_by?->toDateString() ?? '-') . "\n" .
            "Top lines: {$lines}\n" .
            "Use the buttons below to approve or reject.";

        $buttons = $this->purchaseRequestActionMarkup($request->request_code);
        $this->notifier->notifyUser($approval->approver_id, $message, $buttons);

        $pdf = $this->exportPurchaseRequestPdf($request);
        if ($pdf && !empty($pdf['path'])) {
            $this->notifier->notifyUserDocument(
                $approval->approver_id,
                $pdf['path'],
                $pdf['filename'] ?? null,
                'PR PDF',
                $buttons
            );
        }
    }

    private function notifyPrStatus(PurchaseRequest $request, string $message): void
    {
        if ($request->requested_by) {
            $this->notifier->notifyUser($request->requested_by, $message);
        }
        $engineering = $this->firstUserWithRole($request->tenant_id, ['engineering manager', 'engineering', 'eng manager']);
        if ($engineering) {
            $this->notifier->notifyUser($engineering->id, $message);
        }
    }

    private function notifyEngineeringManager(int $tenantId, string $message, ?array $pdf = null): void
    {
        $engineering = $this->firstUserWithRole($tenantId, ['engineering manager', 'engineering', 'eng manager']);
        if (!$engineering) {
            return;
        }

        $this->notifier->notifyUser($engineering->id, $message);
        if ($pdf && !empty($pdf['path'])) {
            $this->notifier->notifyUserDocument($engineering->id, $pdf['path'], $pdf['filename'] ?? null, 'PR PDF');
        }
    }

    private function purchaseRequestActionMarkup(string $code): array
    {
        return [
            'inline_keyboard' => [
                [
                    ['text' => 'Approve', 'callback_data' => 'approve ' . $code],
                    ['text' => 'Reject', 'callback_data' => 'reject ' . $code],
                ],
            ],
        ];
    }

    private function exportPurchaseRequestPdf(PurchaseRequest $request): ?array
    {
        $tenant = Tenant::find($request->tenant_id);
        if (!$tenant) {
            return null;
        }

        return $this->exportService->exportPurchaseRequest($request, $tenant);
    }

    private function exportWorkOrderPdf(WorkOrder $order): ?array
    {
        $tenant = Tenant::find($order->tenant_id);
        if (!$tenant) {
            return null;
        }

        return $this->exportService->exportWorkOrder($order, $tenant);
    }

    private function notifyWoStatus(WorkOrder $order, string $message): void
    {
        $reference = $order->reference_code ?? ('WO-' . $order->id);
        $buttons = $this->workOrderActionMarkup($reference);
        $pdf = $this->exportWorkOrderPdf($order);
        $userIds = array_filter([
            $order->reported_by,
            $order->assigned_to,
            $this->firstUserWithRole($order->tenant_id, ['engineering manager', 'engineering', 'eng manager'])?->id,
        ]);
        $this->notifyWoChats($userIds, $message, $buttons, $pdf, $reference);
    }

    private function notifyWoParticipants(WorkOrder $order, string $message, ?User $actor = null): void
    {
        $reference = $order->reference_code ?? ('WO-' . $order->id);
        $buttons = $this->workOrderActionMarkup($reference);
        $pdf = $this->exportWorkOrderPdf($order);
        $skipId = $actor?->id;
        $userIds = array_filter([$order->reported_by, $order->assigned_to]);
        if ($skipId) {
            $userIds = array_values(array_filter($userIds, fn ($id) => $id !== $skipId));
        }
        $this->notifyWoChats($userIds, $message, $buttons, $pdf, $reference);
    }

    private function notifyWoChats(array $userIds, string $message, array $buttons, ?array $pdf, string $reference): void
    {
        $userIds = array_values(array_unique(array_filter($userIds)));
        if (empty($userIds)) {
            return;
        }

        $links = TelegramLink::whereIn('user_id', $userIds)
            ->whereNotNull('telegram_chat_id')
            ->get()
            ->groupBy('telegram_chat_id');

        foreach ($links as $chatId => $group) {
            $this->notifier->notifyChat($chatId, $message, $buttons);
            if ($pdf && !empty($pdf['path'])) {
                $this->notifier->notifyChatDocument(
                    $chatId,
                    $pdf['path'],
                    $pdf['filename'] ?? null,
                    "WO {$reference} PDF (updated)",
                    $buttons
                );
            }
        }
    }

    private function workOrderActionMarkup(string $reference): array
    {
        return [
            'inline_keyboard' => [
                [
                    ['text' => 'Accept', 'callback_data' => 'accept ' . $reference],
                    ['text' => 'Schedule', 'callback_data' => 'schedule ' . $reference],
                ],
                [
                    ['text' => 'Create PR', 'callback_data' => 'pr ' . $reference],
                    ['text' => 'Close', 'callback_data' => 'close ' . $reference],
                ],
            ],
        ];
    }

    private function normalizeWorkOrderParts(array $args): array
    {
        $parts = $args['parts'] ?? [];
        if (is_string($parts)) {
            $parts = $this->parsePartsText($parts);
        }
        if (empty($parts) && isset($args['items']) && is_string($args['items'])) {
            $parts = $this->parsePartsText($args['items']);
        }
        if (isset($args['sku']) || isset($args['description'])) {
            $parts[] = [
                'sku' => $args['sku'] ?? null,
                'description' => $args['description'] ?? $args['item'] ?? null,
                'qty' => $args['qty'] ?? $args['quantity'] ?? 1,
                'unit' => $args['unit'] ?? $args['uom'] ?? null,
            ];
        }

        return array_values(array_filter($parts));
    }

    private function parsePartsText(string $text): array
    {
        $parts = [];
        $segments = preg_split('/[;,]/', $text);
        foreach ($segments as $segment) {
            $segment = trim($segment);
            if ($segment === '') {
                continue;
            }

            if (preg_match('/(\\d+(?:\\.\\d+)?)\\s+(.+)/', $segment, $matches)) {
                $parts[] = [
                    'qty' => (float) $matches[1],
                    'description' => trim($matches[2]),
                ];
                continue;
            }

            if (preg_match('/(.+)\\s+x\\s*(\\d+(?:\\.\\d+)?)/', $segment, $matches)) {
                $parts[] = [
                    'qty' => (float) $matches[2],
                    'description' => trim($matches[1]),
                ];
                continue;
            }

            $parts[] = [
                'qty' => 1,
                'description' => $segment,
            ];
        }

        return array_values(array_filter($parts, fn ($part) => !empty($part['description'])));
    }

    private function createPrForWorkOrder(int $tenantId, WorkOrder $order, User $user, array $parts): ?PurchaseRequest
    {
        if (!$parts) {
            return null;
        }

        $request = PurchaseRequest::create([
            'tenant_id' => $tenantId,
            'request_code' => $this->nextReference($tenantId, 'PR', 'purchase_requests', 'request_code'),
            'title' => 'WO ' . $order->reference_code . ' spares',
            'department' => 'Maintenance',
            'status' => 'submitted',
            'priority' => $order->priority ?? 'medium',
            'requested_by' => $user->id,
            'needed_by' => $order->due_at?->toDateString(),
            'currency' => 'USD',
            'notes' => 'Auto-generated from work order ' . $order->reference_code,
            'submitted_at' => now(),
            'last_reminded_at' => now(),
        ]);

        $lineNumber = 1;
        foreach ($parts as $partData) {
            $sku = $partData['sku'] ?? null;
            $desc = $partData['description'] ?? $partData['item'] ?? null;
            $part = $this->findPart($tenantId, $sku, $desc);
            $qty = $this->numericValue($partData['qty'] ?? $partData['quantity'] ?? 1) ?? 1;

            PurchaseRequestItem::create([
                'tenant_id' => $tenantId,
                'purchase_request_id' => $request->id,
                'line_number' => $lineNumber++,
                'part_id' => $part?->id,
                'sku' => $part?->sku ?? $sku,
                'item_name' => $part?->name ?? $desc,
                'description' => $desc,
                'quantity' => $qty,
                'unit' => $partData['unit'] ?? null,
                'est_unit_cost' => $this->numericValue($partData['est_unit_cost'] ?? null),
            ]);
        }

        $request->update(['total_estimated_cost' => $this->sumPurchaseRequestCost($request)]);

        $approvers = $this->buildPurchaseRequestApprovals($tenantId, $request);
        $first = $approvers->first();
        if ($first?->approver_id) {
            $this->notifyApprovalRequest($request, $first);
        }

        $this->notifyPrStatus($request, "PR {$request->request_code} created for WO {$order->reference_code}.");

        return $request;
    }

    private function createDraftPrForWorkOrder(int $tenantId, WorkOrder $order, User $user): ?PurchaseRequest
    {
        $request = PurchaseRequest::create([
            'tenant_id' => $tenantId,
            'request_code' => $this->nextReference($tenantId, 'PR', 'purchase_requests', 'request_code'),
            'title' => 'WO ' . $order->reference_code . ' spares',
            'department' => 'Maintenance',
            'status' => 'draft',
            'priority' => $order->priority ?? 'medium',
            'requested_by' => $user->id,
            'needed_by' => $order->due_at?->toDateString(),
            'currency' => 'USD',
            'notes' => 'Draft PR created from work order ' . $order->reference_code . '. Add line items then submit.',
            'last_reminded_at' => now(),
        ]);

        $this->notifyPrStatus($request, "Draft PR {$request->request_code} created for WO {$order->reference_code}.");

        return $request;
    }

    private function notifyProcurement(PurchaseRequest $request): void
    {
        $users = $this->usersByRole($request->tenant_id, ['procurement', 'buyer']);
        if ($users->isEmpty()) {
            return;
        }

        $message = "PR {$request->request_code} approved. Reply CONVERT {$request->request_code} TO PO vendor=\"...\"";
        $this->notifier->notifyUsers($users, $message);
    }

    private function sumPurchaseRequestCost(PurchaseRequest $request): float
    {
        return (float) $request->items()->get()->sum(function ($item) {
            $qty = (float) ($item->quantity ?? 0);
            $unit = (float) ($item->est_unit_cost ?? $item->quote_amount_usd ?? 0);
            return $qty * $unit;
        });
    }

    private function nextReference(int $tenantId, string $prefix, string $table, string $column): string
    {
        $codes = \DB::table($table)
            ->where('tenant_id', $tenantId)
            ->where($column, 'like', $prefix . '-%')
            ->pluck($column);

        $number = 0;
        $pattern = '/^' . preg_quote($prefix, '/') . '-0*(\\d+)$/i';
        foreach ($codes as $code) {
            if ($code && preg_match($pattern, (string) $code, $matches)) {
                $number = max($number, (int) $matches[1]);
            }
        }

        return sprintf('%s-%05d', $prefix, $number + 1);
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $clean = preg_replace('/[^0-9\\.\\-]/', '', (string) $value);
        return is_numeric($clean) ? (float) $clean : null;
    }

    private function boolValue(mixed $value): bool
    {
        if (is_bool($value)) {
            return $value;
        }

        if (is_numeric($value)) {
            return ((int) $value) === 1;
        }

        $value = strtolower(trim((string) $value));
        return in_array($value, ['1', 'true', 'yes', 'y', 'on'], true);
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function dateTimeValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        try {
            return Carbon::parse($value)->toDateTimeString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function hasAnyRole(User $user, array $roles): bool
    {
        $roleNames = $user->roles()->pluck('name')->map(fn ($name) => strtolower($name))->all();
        if (empty($roleNames)) {
            return true;
        }

        if (in_array('admin', $roleNames, true)) {
            return true;
        }

        foreach ($roles as $role) {
            if (in_array(strtolower($role), $roleNames, true)) {
                return true;
            }
        }

        return false;
    }

    private function firstUserWithRole(int $tenantId, array $roles): ?User
    {
        foreach ($roles as $role) {
            $user = User::where('tenant_id', $tenantId)
                ->whereHas('roles', fn ($q) => $q->where('name', 'like', '%' . $role . '%'))
                ->first();
            if ($user) {
                return $user;
            }
        }

        return null;
    }

    private function usersByRole(int $tenantId, array $roles)
    {
        return User::where('tenant_id', $tenantId)
            ->whereHas('roles', function ($q) use ($roles) {
                $first = true;
                foreach ($roles as $role) {
                    if ($first) {
                        $q->where('name', 'like', '%' . $role . '%');
                        $first = false;
                        continue;
                    }
                    $q->orWhere('name', 'like', '%' . $role . '%');
                }
            })
            ->get();
    }

    private function findPart(int $tenantId, ?string $sku, ?string $name): ?Part
    {
        $sku = $sku ? trim($sku) : null;
        if ($sku) {
            $part = Part::where('tenant_id', $tenantId)->where('sku', $sku)->first();
            if ($part) {
                return $part;
            }
        }

        $name = $name ? trim($name) : null;
        if ($name) {
            return Part::where('tenant_id', $tenantId)
                ->where('name', 'like', '%' . $name . '%')
                ->first();
        }

        return null;
    }

    private function parseSearchFilters(?string $query): array
    {
        if (!$query) {
            return [[], null];
        }

        $filters = [];
        $pattern = '/\b(status|category|location|site)\s*[:=]\s*("([^"]+)"|\'([^\']+)\'|([^\s]+))/i';
        if (preg_match_all($pattern, $query, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $value = $match[3] ?: $match[4] ?: $match[5];
                $key = strtolower($match[1]);
                if ($key === 'site') {
                    $key = 'location';
                }
                $filters[$key] = $value;
            }
            $query = trim(preg_replace($pattern, '', $query));
        }

        return [$filters, $query ?: null];
    }

    private function assetNotFoundResponse(int $tenantId, ?string $needle): array
    {
        $suggestions = $this->suggestAssets($tenantId, $needle);
        if ($suggestions->isNotEmpty()) {
            $lines = $suggestions->map(fn ($asset) => "{$asset->asset_tag} - {$asset->name}");
            return ['message' => "Asset not found. Try:\n" . $lines->implode("\n")];
        }

        $locations = $this->suggestLocations($tenantId, $needle);
        if (!empty($locations)) {
            return ['message' => "Asset not found. Nearby locations:\n" . implode("\n", $locations)];
        }

        $sites = $this->suggestSites($tenantId, $needle);
        if (!empty($sites)) {
            return ['message' => "Asset not found. Nearby sites:\n" . implode("\n", $sites)];
        }

        return ['message' => 'Asset not found. Try /list module=assets to see available tags.'];
    }

    private function workOrderNotFoundResponse(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle !== '') {
            $orders = WorkOrder::where('tenant_id', $tenantId)
                ->where(function ($builder) use ($needle) {
                    $builder->where('reference_code', 'like', '%' . $needle . '%')
                        ->orWhere('description', 'like', '%' . $needle . '%');
                })
                ->limit(5)
                ->get();

            if ($orders->isNotEmpty()) {
                $lines = $orders->map(function ($order) {
                    $ref = $order->reference_code ?? ('WO-' . $order->id);
                    $summary = Str::limit((string) ($order->description ?? ''), 40, '...');
                    return "{$ref} - {$summary}";
                });

                return ['message' => "Work order not found. Did you mean:\n" . $lines->implode("\n")];
            }
        }

        return ['message' => 'Work order not found. Try /list module=workorders.'];
    }

    private function suggestAssets(int $tenantId, ?string $needle)
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return collect();
        }

        return Asset::where('tenant_id', $tenantId)
            ->where(function ($builder) use ($needle) {
                $builder->where('asset_tag', 'like', '%' . $needle . '%')
                    ->orWhere('name', 'like', '%' . $needle . '%');
            })
            ->limit(5)
            ->get(['asset_tag', 'name']);
    }

    private function suggestLocations(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return Asset::where('tenant_id', $tenantId)
            ->whereNotNull('location')
            ->where('location', '!=', '')
            ->where('location', 'like', '%' . $needle . '%')
            ->distinct()
            ->limit(5)
            ->pluck('location')
            ->map(fn ($location) => (string) $location)
            ->all();
    }

    private function suggestSites(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return Site::where('tenant_id', $tenantId)
            ->where('name', 'like', '%' . $needle . '%')
            ->limit(5)
            ->pluck('name')
            ->map(fn ($name) => (string) $name)
            ->all();
    }

    private function suggestSkus(int $tenantId, string $sku)
    {
        $sku = trim($sku);
        if ($sku === '') {
            return collect();
        }

        return Part::where('tenant_id', $tenantId)
            ->where('sku', 'like', '%' . $sku . '%')
            ->limit(5)
            ->get(['sku', 'name']);
    }

    private function actionForTool(string $tool): string
    {
        if ($tool === 'workorders.pr') {
            return 'create';
        }
        if (str_contains($tool, 'create') || str_contains($tool, 'add')) {
            return 'create';
        }
        if (str_contains($tool, 'update') || str_contains($tool, 'assign') || str_contains($tool, 'accept') || str_contains($tool, 'approve') || str_contains($tool, 'submit') || str_contains($tool, 'reject') || str_contains($tool, 'convert')) {
            return 'update';
        }
        if (str_contains($tool, 'close')) {
            return 'close';
        }
        return 'read';
    }

    private function isDeniedTool(string $tool): bool
    {
        return (bool) preg_match('/(delete|destroy|remove|reverse)/i', $tool);
    }

    private function friendlyDatabaseError(QueryException $e): string
    {
        $message = strtolower($e->getMessage());

        if (str_contains($message, 'no such table') || str_contains($message, 'does not exist')) {
            return 'Database schema is missing. Please run migrations: php artisan migrate.';
        }

        if (str_contains($message, 'no column') || str_contains($message, 'unknown column')) {
            return 'Database schema is out of date. Please run migrations: php artisan migrate.';
        }

        return 'Database error occurred. Please try again or run php artisan migrate.';
    }

    private function logTool(User $user, string $tool, array $args, array $result, string $channel): void
    {
        AuditLog::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'channel' => $channel,
            'action' => $tool,
            'payload_json' => [
                'input' => $args,
                'output' => $result,
            ],
            'created_at' => now(),
        ]);
    }

    private function assetLabel(?Asset $asset, ?string $fallbackTag = null): string
    {
        $tag = $asset?->asset_tag ?? $fallbackTag ?? '-';
        $name = $asset?->name ?? null;
        if ($name) {
            return "{$tag} - {$name}";
        }

        return (string) $tag;
    }
}
