<?php

namespace App\Domain\Telegram;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use RuntimeException;

class TelegramApiClient
{
    private Client $client;
    private string $token;
    private string $baseUrl;
    private bool $verifySsl;

    public function __construct(?Client $client = null)
    {
        $token = config('services.telegram.bot_token');
        if (!$token) {
            throw new RuntimeException('Telegram bot token not configured.');
        }

        $this->token = $token;
        $this->baseUrl = 'https://api.telegram.org/bot' . $token . '/';
        $verify = config('services.telegram.verify_ssl', true);
        if (is_string($verify)) {
            $normalized = strtolower(trim($verify));
            if (in_array($normalized, ['false', '0', 'off', 'no'], true)) {
                $verify = false;
            } elseif (in_array($normalized, ['true', '1', 'on', 'yes'], true)) {
                $verify = true;
            } else {
                $verify = trim($verify);
            }
        }

        if (is_string($verify) && $verify !== '') {
            $path = $verify;
            if (!is_file($path)) {
                $candidate = base_path($verify);
                if (is_file($candidate)) {
                    $path = $candidate;
                }
            }
            $verify = $path;
        }

        $this->verifySsl = $verify;
        if (!$client || !$client->getConfig('base_uri')) {
            $client = new Client([
                'base_uri' => $this->baseUrl,
                'timeout' => 15,
                'verify' => $this->verifySsl,
            ]);
        }

        $this->client = $client;
    }

    public function sendMessage(string $chatId, string $text, ?array $replyMarkup = null): array
    {
        $payload = [
            'chat_id' => $chatId,
            'text' => $text,
            'disable_web_page_preview' => true,
        ];

        if ($replyMarkup) {
            $payload['reply_markup'] = $replyMarkup;
        }

        return $this->request('sendMessage', $payload);
    }

    public function sendDocument(string $chatId, string $filePath, ?string $filename = null, ?string $caption = null, ?array $replyMarkup = null): array
    {
        if (!is_file($filePath)) {
            throw new RuntimeException('Telegram document file not found.');
        }

        $multipart = [
            [
                'name' => 'chat_id',
                'contents' => $chatId,
            ],
            [
                'name' => 'document',
                'contents' => fopen($filePath, 'rb'),
                'filename' => $filename ?: basename($filePath),
            ],
        ];

        if ($caption) {
            $multipart[] = [
                'name' => 'caption',
                'contents' => $caption,
            ];
        }

        if ($replyMarkup) {
            $multipart[] = [
                'name' => 'reply_markup',
                'contents' => json_encode($replyMarkup, JSON_UNESCAPED_UNICODE),
            ];
        }

        return $this->requestMultipart('sendDocument', $multipart);
    }

    public function getFile(string $fileId): array
    {
        return $this->request('getFile', ['file_id' => $fileId]);
    }

    public function answerCallbackQuery(string $callbackQueryId, ?string $text = null, bool $showAlert = false): array
    {
        $payload = [
            'callback_query_id' => $callbackQueryId,
        ];

        if ($text !== null) {
            $payload['text'] = $text;
        }

        if ($showAlert) {
            $payload['show_alert'] = true;
        }

        return $this->request('answerCallbackQuery', $payload);
    }

    public function downloadFile(string $filePath): string
    {
        $fileUrl = 'https://api.telegram.org/file/bot' . $this->token . '/' . ltrim($filePath, '/');

        try {
            $response = $this->client->get($fileUrl, [
                'http_errors' => false,
                'verify' => $this->verifySsl,
            ]);
        } catch (GuzzleException $e) {
            throw new RuntimeException('Unable to download Telegram file.');
        }

        if ($response->getStatusCode() >= 400) {
            throw new RuntimeException('Telegram file download failed.');
        }

        return (string) $response->getBody();
    }

    private function requestMultipart(string $method, array $multipart): array
    {
        try {
            $response = $this->client->post($method, [
                'multipart' => $multipart,
                'http_errors' => false,
                'verify' => $this->verifySsl,
            ]);
        } catch (GuzzleException $e) {
            $message = str_replace($this->token, '[redacted]', $e->getMessage());
            throw new RuntimeException('Telegram API request failed: ' . $message);
        }

        if ($response->getStatusCode() >= 400) {
            throw new RuntimeException('Telegram API error.');
        }

        $body = json_decode((string) $response->getBody(), true);
        if (!$body || empty($body['ok'])) {
            throw new RuntimeException('Telegram API returned an error.');
        }

        return $body['result'] ?? [];
    }

    private function request(string $method, array $payload): array
    {
        try {
            $response = $this->client->post($method, [
                'json' => $payload,
                'http_errors' => false,
                'verify' => $this->verifySsl,
            ]);
        } catch (GuzzleException $e) {
            $message = str_replace($this->token, '[redacted]', $e->getMessage());
            throw new RuntimeException('Telegram API request failed: ' . $message);
        }

        if ($response->getStatusCode() >= 400) {
            throw new RuntimeException('Telegram API error.');
        }

        $body = json_decode((string) $response->getBody(), true);
        if (!$body || empty($body['ok'])) {
            throw new RuntimeException('Telegram API returned an error.');
        }

        return $body['result'] ?? [];
    }
}
