<?php

namespace App\Domain\Telegram;

use App\Models\ChatUpload;
use App\Models\ChatConversation;
use App\Models\AuditLog;
use App\Models\DocumentIntake;
use App\Models\DocumentIntakeBatch;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use App\Models\Tenant;
use App\Models\User;
use App\Domain\Telegram\TelegramNotifier;
use App\Services\DocumentIntakeApplyService;
use App\Services\DocumentIntakeService;
use App\Services\TelegramExportService;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class TelegramFileIngestService
{
    public function __construct(
        private TelegramApiClient $client,
        private DocumentIntakeService $intakeService,
        private DocumentIntakeApplyService $applyService,
        private TelegramExportService $exportService,
        private TelegramNotifier $notifier
    ) {
    }

    public function ingest(array $fileMeta, User $user, string $chatId): array
    {
        $fileId = $fileMeta['file_id'] ?? null;
        if (!$fileId) {
            return ['error' => 'Missing file_id.'];
        }

        $fileInfo = $this->client->getFile($fileId);
        $filePath = $fileInfo['file_path'] ?? null;
        if (!$filePath) {
            return ['error' => 'Unable to retrieve file path.'];
        }

        $contents = $this->client->downloadFile($filePath);
        $originalName = $fileMeta['file_name'] ?? basename($filePath);
        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        $safeName = now()->format('Ymd_His') . '_' . Str::slug(pathinfo($originalName, PATHINFO_FILENAME));
        $filename = $safeName . ($extension ? '.' . $extension : '');
        $storedPath = "tenants/{$user->tenant_id}/uploads/telegram/{$filename}";

        Storage::put($storedPath, $contents);

        $document = DocumentIntake::create([
            'tenant_id' => $user->tenant_id,
            'status' => 'review',
            'source_channel' => 'telegram',
            'uploaded_by' => $user->id,
            'original_filename' => $originalName,
            'stored_path' => $storedPath,
            'mime_type' => $fileMeta['mime_type'] ?? null,
            'meta_json' => [
                'telegram_file_id' => $fileId,
                'telegram_chat_id' => $chatId,
                'telegram_user_id' => $fileMeta['telegram_user_id'] ?? null,
            ],
        ]);

        $conversation = $this->resolveConversation($user);
        $upload = ChatUpload::create([
            'tenant_id' => $user->tenant_id,
            'conversation_id' => $conversation->id,
            'user_id' => $user->id,
            'original_name' => $originalName,
            'storage_path' => $storedPath,
            'mime_type' => $fileMeta['mime_type'] ?? null,
            'size_bytes' => strlen($contents),
            'checksum' => sha1($contents),
        ]);

        $batch = $this->intakeService->intakeUploads([$upload], $user->tenant_id, $conversation->id, $user->id);

        $document->update([
            'status' => 'review',
            'meta_json' => array_merge($document->meta_json ?? [], [
                'batch_id' => $batch->id,
                'doc_type' => $batch->doc_type,
                'summary' => $batch->summary,
            ]),
        ]);

        $previewRows = $batch->rows()->limit(5)->get();

        AuditLog::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'channel' => 'telegram',
            'action' => 'telegram.intake',
            'payload_json' => [
                'intake_id' => $document->id,
                'batch_id' => $batch->id,
                'file' => $originalName,
            ],
            'created_at' => now(),
        ]);

        return [
            'intake_id' => $document->id,
            'batch_id' => $batch->id,
            'doc_type' => $batch->doc_type,
            'summary' => $batch->summary,
            'warnings' => $batch->warnings,
            'preview' => $previewRows->map(fn ($row) => [
                'row_index' => $row->row_index,
                'data' => $row->data,
                'errors' => $row->errors,
                'warnings' => $row->warnings,
                'matches' => $row->matches,
            ])->all(),
        ];
    }

    public function apply(DocumentIntake $intake, ?int $userId): array
    {
        $batchId = data_get($intake->meta_json, 'batch_id');
        if (!$batchId) {
            return ['status' => 'error', 'message' => 'No intake batch found.'];
        }

        $batch = DocumentIntakeBatch::find($batchId);
        if (!$batch) {
            return ['status' => 'error', 'message' => 'Intake batch not found.'];
        }

        $summary = $this->applyService->applyBatch($batch, $userId);
        $intake->update([
            'status' => 'applied',
            'meta_json' => array_merge($intake->meta_json ?? [], ['apply_summary' => $summary]),
        ]);

        $pdf = null;
        if ($batch->doc_type === 'purchase_request') {
            $request = PurchaseRequest::where('tenant_id', $intake->tenant_id)
                ->where('request_code', 'PR-AI-' . $batch->id)
                ->first();
            if ($request) {
                $request->update([
                    'status' => 'submitted',
                    'submitted_at' => now(),
                    'total_estimated_cost' => $this->sumPurchaseRequestCost($request),
                    'last_reminded_at' => now(),
                ]);

                $approvals = $this->buildPurchaseRequestApprovals($request);
                $first = $approvals->first();
                if ($first?->approver_id) {
                    $this->notifyApprovalRequest($request, $first);
                }

                $tenant = Tenant::find($request->tenant_id);
                if ($tenant) {
                    $pdf = $this->exportService->exportPurchaseRequest($request, $tenant);
                }

                $message = "PR {$request->request_code} submitted for approval from upload.";
                $this->notifyPrStatus($request, $message);

                $engineering = $this->firstUserWithRole($request->tenant_id, ['engineering manager', 'engineering', 'eng manager']);
                if ($engineering && $pdf && !empty($pdf['path'])) {
                    $this->notifier->notifyUserDocument($engineering->id, $pdf['path'], $pdf['filename'] ?? null, 'PR PDF');
                }

                if ($request->requested_by && $pdf && !empty($pdf['path'])) {
                    $this->notifier->notifyUserDocument($request->requested_by, $pdf['path'], $pdf['filename'] ?? null, 'PR PDF');
                }
            }
        }

        AuditLog::create([
            'tenant_id' => $intake->tenant_id,
            'user_id' => $userId,
            'channel' => 'telegram',
            'action' => 'telegram.intake.apply',
            'payload_json' => [
                'intake_id' => $intake->id,
                'batch_id' => $batch->id,
                'summary' => $summary,
            ],
            'created_at' => now(),
        ]);

        return array_filter([
            'status' => 'applied',
            'summary' => $summary,
            'file_path' => $pdf['path'] ?? null,
            'filename' => $pdf['filename'] ?? null,
        ]);
    }

    private function firstUserWithRole(int $tenantId, array $roles): ?User
    {
        foreach ($roles as $role) {
            $user = User::where('tenant_id', $tenantId)
                ->whereHas('roles', fn ($q) => $q->where('name', 'like', '%' . $role . '%'))
                ->first();
            if ($user) {
                return $user;
            }
        }

        return null;
    }

    private function resolveConversation(User $user): ChatConversation
    {
        return ChatConversation::firstOrCreate(
            [
                'tenant_id' => $user->tenant_id,
                'user_id' => $user->id,
                'title' => 'Telegram Intake',
            ],
            [
                'status' => 'active',
                'last_message_at' => now(),
            ]
        );
    }

    private function buildPurchaseRequestApprovals(PurchaseRequest $request)
    {
        $existing = PurchaseRequestApproval::where('tenant_id', $request->tenant_id)
            ->where('purchase_request_id', $request->id)
            ->orderBy('step')
            ->get();

        if ($existing->isNotEmpty()) {
            return $existing;
        }

        $steps = [
            1 => $this->firstUserWithRole($request->tenant_id, ['engineering manager', 'engineering', 'eng manager']),
            2 => $this->firstUserWithRole($request->tenant_id, ['finance manager', 'finance', 'accounts']),
        ];

        $created = collect();
        foreach ($steps as $step => $approver) {
            $created->push(PurchaseRequestApproval::create([
                'tenant_id' => $request->tenant_id,
                'purchase_request_id' => $request->id,
                'step' => $step,
                'approver_id' => $approver?->id,
                'status' => 'pending',
            ]));
        }

        return $created;
    }

    private function notifyApprovalRequest(PurchaseRequest $request, PurchaseRequestApproval $approval): void
    {
        if (!$approval->approver_id) {
            return;
        }

        $lines = $request->items()->limit(2)->get()->map(function ($item) {
            return ($item->sku ?? $item->item_name ?? 'Item') . ' x' . ($item->quantity ?? 0);
        })->implode('; ');

        $message = "PR {$request->request_code} ({$request->department}) est " .
            '$' . number_format((float) $request->total_estimated_cost, 2) .
            ' | Needed by ' . ($request->needed_by?->toDateString() ?? '-') . "\n" .
            "Top lines: {$lines}\n" .
            "Reply: APPROVE {$request->request_code} notes=\"...\" or REJECT {$request->request_code} reason=\"...\"";

        $this->notifier->notifyUser($approval->approver_id, $message);
    }

    private function notifyPrStatus(PurchaseRequest $request, string $message): void
    {
        if ($request->requested_by) {
            $this->notifier->notifyUser($request->requested_by, $message);
        }

        $engineering = $this->firstUserWithRole($request->tenant_id, ['engineering manager', 'engineering', 'eng manager']);
        if ($engineering) {
            $this->notifier->notifyUser($engineering->id, $message);
        }
    }

    private function sumPurchaseRequestCost(PurchaseRequest $request): float
    {
        return (float) $request->items()->get()->sum(function ($item) {
            $qty = (float) ($item->quantity ?? 0);
            $unit = (float) ($item->est_unit_cost ?? $item->quote_amount_usd ?? 0);
            return $qty * $unit;
        });
    }
}
