<?php

namespace App\Domain\Telegram;

use App\Models\TelegramLink;
use App\Models\TelegramLinkCode;
use App\Models\User;
use App\Models\AuditLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class TelegramLinkService
{
    public function resolveUser(string $telegramUserId, ?string $chatId = null): ?User
    {
        $link = TelegramLink::with('user')
            ->where('telegram_user_id', $telegramUserId)
            ->first();

        if ($link?->user) {
            return $link->user;
        }

        if ($chatId) {
            $candidates = TelegramLink::with('user', 'user.roles')
                ->where('telegram_chat_id', $chatId)
                ->get()
                ->pluck('user')
                ->filter();

            if ($candidates->isEmpty()) {
                return null;
            }

            $preferred = $candidates->first(function ($user) {
                $roles = $user->roles->pluck('name')->map(fn ($name) => strtolower($name));
                return $roles->contains('admin') || $roles->contains('engineering manager');
            });

            return $preferred ?: $candidates->first();
        }

        return null;
    }

    public function linkByCode(string $code, string $telegramUserId, ?string $chatId = null): array
    {
        $linkCode = TelegramLinkCode::where('code', $code)
            ->whereNull('used_at')
            ->first();

        if (!$linkCode) {
            return ['linked' => false, 'message' => 'Invalid or expired link code.'];
        }

        if ($linkCode->expires_at && $linkCode->expires_at->isPast()) {
            return ['linked' => false, 'message' => 'Link code has expired.'];
        }

        return DB::transaction(function () use ($linkCode, $telegramUserId, $chatId) {
            $existing = TelegramLink::where('telegram_user_id', $telegramUserId)->first();
            if ($existing) {
                return ['linked' => true, 'message' => 'Telegram account is already linked.'];
            }

            $link = TelegramLink::create([
                'tenant_id' => $linkCode->tenant_id,
                'user_id' => $linkCode->user_id,
                'telegram_user_id' => $telegramUserId,
                'telegram_chat_id' => $chatId,
                'linked_at' => now(),
            ]);

            $linkCode->update(['used_at' => now()]);

            AuditLog::create([
                'tenant_id' => $link->tenant_id,
                'user_id' => $link->user_id,
                'channel' => 'telegram',
                'action' => 'telegram.link',
                'payload_json' => [
                    'telegram_user_id' => $telegramUserId,
                    'telegram_chat_id' => $chatId,
                ],
                'created_at' => now(),
            ]);

            return [
                'linked' => true,
                'message' => 'Linked successfully. You can now use Telegram commands.',
                'link_id' => $link->id,
            ];
        });
    }

    public function createLinkCode(User $user, int $minutes = 30): TelegramLinkCode
    {
        return TelegramLinkCode::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'code' => strtoupper(Str::random(8)),
            'expires_at' => now()->addMinutes($minutes),
        ]);
    }
}
