<?php

namespace App\Domain\Telegram;

use App\Domain\AI\AiOrchestrator;
use App\Domain\AI\ToolExecutor;
use App\Models\AuditLog;
use App\Models\DocumentIntake;
use App\Models\InventoryItem;
use App\Models\TelegramEvent;
use App\Models\ReconciliationLink;
use App\Models\User;
use App\Models\WorkOrderPart;
use App\Services\TelegramEventService;
use App\Services\TelegramGroupService;
use App\Services\TelegramReportService;
use App\Services\LinkingReconciliationService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;

class TelegramRouterService
{
    public function __construct(
        private TelegramLinkService $linkService,
        private TelegramFileIngestService $fileIngestService,
        private ToolExecutor $toolExecutor,
        private AiOrchestrator $orchestrator,
        private TelegramEventService $eventService,
        private TelegramReportService $reportService,
        private TelegramGroupService $groupService,
        private LinkingReconciliationService $reconciliationService
    ) {
    }

    public function handle(array $context): array
    {
        $text = $context['text'] ?? '';
        $telegramUserId = (string) ($context['telegram_user_id'] ?? '');
        $chatId = (string) ($context['chat_id'] ?? '');
        $chatType = $context['chat_type'] ?? null;
        $attachments = $context['attachments'] ?? [];

        if (!$telegramUserId || !$chatId) {
            return ['reply' => null];
        }

        if (Str::startsWith(Str::lower($text), '/link')) {
            return $this->toResponse($this->handleLink($text, $telegramUserId, $chatId));
        }

        $user = $this->linkService->resolveUser($telegramUserId, $chatId);
        if (!$user) {
            return $this->toResponse('Not linked. Go to the web app -> Integrations -> Telegram to get a link code, then send /link CODE.');
        }

        $text = $this->applyAssetContext($text, $user);
        $text = $this->applyEntityContext($text, $user);
        $this->rememberAssetFromText($text, $user);

        if ($pending = $this->getConversationContext($user)['pending_action'] ?? null) {
            $pendingResponse = $this->handlePendingAction($pending, $text, $user);
            if ($pendingResponse) {
                return $this->toResponse($pendingResponse);
            }
        }

        if ($this->isDeleteIntent($text)) {
            return $this->toResponse('Deletion is not allowed on Telegram; use the web app.');
        }

        if ($this->isGroupCommand($text)) {
            return $this->toResponse($this->handleGroupCommand($text, $user, $chatId, $chatType));
        }

        if ($this->isEventAction($text)) {
            return $this->toResponse($this->handleEventAction($text, $user));
        }

        if ($this->isDetailsCommand($text)) {
            return $this->toResponse($this->handleDetails($text, $user));
        }

        if ($this->isReconciliationCommand($text)) {
            return $this->toResponse($this->handleReconciliationCommand($text, $user));
        }

        if ($attachments) {
            return $this->toResponse($this->handleAttachments($attachments, $user, $chatId, $telegramUserId, $text));
        }

        if ($this->isApplyCommand($text)) {
            return $this->toResponse($this->handleApply($text, $user));
        }

        if (Str::startsWith($text, '/report')) {
            return $this->toResponse($this->handleReport($text, $user));
        }

        if (Str::startsWith($text, '/help')) {
            return $this->toResponse($this->helpText());
        }

        if (Str::startsWith($text, '/list')) {
            return $this->toResponse($this->handleList($text, $user));
        }

        if (Str::startsWith($text, '/get')) {
            return $this->toResponse($this->handleGet($text, $user));
        }

        if (Str::startsWith($text, '/export')) {
            return $this->toResponse($this->handleExport($text, $user));
        }

        if (Str::startsWith($text, '/assets')) {
            return $this->toResponse($this->handleAssets($text, $user));
        }

        if (Str::startsWith($text, '/wo')) {
            return $this->toResponse($this->handleWorkOrders($text, $user));
        }

        if (Str::startsWith($text, '/pr')) {
            return $this->toResponse($this->handlePurchaseRequests($text, $user));
        }

        if (Str::startsWith($text, '/po')) {
            return $this->toResponse($this->handlePurchaseOrders($text, $user));
        }

        if (Str::startsWith($text, '/parts')) {
            return $this->toResponse($this->handleParts($text, $user));
        }

        if (Str::startsWith($text, '/stock')) {
            return $this->toResponse($this->handleStock($text, $user));
        }

        if (Str::startsWith($text, '/meter')) {
            return $this->toResponse($this->handleMeter($text, $user));
        }

        if (Str::startsWith($text, '/intake')) {
            return $this->toResponse($this->handleIntake($user));
        }

        $moduleResponse = $this->handleModuleList($text, $user);
        if ($moduleResponse) {
            return $this->toResponse($moduleResponse);
        }

        return $this->toResponse($this->handleNaturalLanguage($text, $user));
    }

    private function handleLink(string $text, string $telegramUserId, string $chatId): string
    {
        $parts = preg_split('/\s+/', trim($text));
        $code = $parts[1] ?? null;
        if (!$code) {
            return 'Usage: /link CODE';
        }

        $result = $this->linkService->linkByCode($code, $telegramUserId, $chatId);
        return $result['message'] ?? 'Linking failed.';
    }

    private function handleAttachments(array $attachments, User $user, string $chatId, string $telegramUserId, string $text): array
    {
        $messages = [];
        $documents = [];
        foreach ($attachments as $attachment) {
            $attachment['telegram_user_id'] = $telegramUserId;
            $result = $this->fileIngestService->ingest($attachment, $user, $chatId);
            if (!empty($result['error'])) {
                $messages[] = 'Upload failed: ' . $result['error'];
                continue;
            }

            $workOrderRef = $this->extractWorkOrderRefFromText($text);
            if ($workOrderRef) {
                $linked = $this->attachIntakeToWorkOrder($workOrderRef, $result['intake_id'] ?? null, $user);
                if ($linked) {
                    $messages[] = "Attached evidence to {$workOrderRef}.";
                }
            }

            if ($this->isWorkOrderUploadIntent($text)) {
                $workOrderReply = $this->createWorkOrderFromIntake($result, $user, $text);
                if (is_array($workOrderReply)) {
                    if (!empty($workOrderReply['reply'])) {
                        $messages[] = $workOrderReply['reply'];
                    }
                    if (!empty($workOrderReply['documents'])) {
                        $documents = array_merge($documents, $workOrderReply['documents']);
                    }
                } elseif ($workOrderReply) {
                    $messages[] = $workOrderReply;
                }
            }

            $preview = $this->formatIntakePreview($result);
            $messages[] = $preview;
        }

        return ['reply' => implode("\n\n", array_filter($messages)), 'documents' => $documents];
    }

    private function handleApply(string $text, User $user): array
    {
        preg_match('/apply\s+(\d+)/i', $text, $matches);
        $intakeId = $matches[1] ?? null;
        if (!$intakeId) {
            return ['reply' => 'Usage: APPLY <intake_id>'];
        }

        $intake = DocumentIntake::where('tenant_id', $user->tenant_id)
            ->where('id', $intakeId)
            ->first();

        if (!$intake) {
            return ['reply' => 'Intake not found.'];
        }

        $result = $this->fileIngestService->apply($intake, $user->id);
        if (($result['status'] ?? '') !== 'applied') {
            return ['reply' => $result['message'] ?? 'Unable to apply intake.'];
        }

        $reply = 'Applied intake ' . $intakeId . '. Rows applied: ' . ($result['summary']['applied'] ?? 0) . '.';
        $documents = [];
        if (!empty($result['file_path'])) {
            $documents[] = [
                'path' => $result['file_path'],
                'filename' => $result['filename'] ?? null,
            ];
        }

        return ['reply' => $reply, 'documents' => $documents];
    }

    private function handleAssets(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/assets');
        $parts = preg_split('/\s+/', trim($command));
        $action = $parts[0] ?? 'search';
        $args = $this->parseArgs($command);

        $tool = match ($action) {
            'create' => 'assets.create',
            'update' => 'assets.update',
            default => 'assets.search',
        };

        return $this->executeTool($tool, $args, $user);
    }

    private function handleWorkOrders(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/wo');
        $parts = preg_split('/\s+/', trim($command));
        $action = $parts[0] ?? 'list';
        $args = $this->parseArgs($command);

        $tool = match ($action) {
            'create' => 'workorders.create',
            'assign' => 'workorders.assign',
            'accept' => 'workorders.accept',
            'update' => 'workorders.update',
            'schedule' => 'workorders.update',
            'pr' => 'workorders.pr',
            'approve' => 'workorders.approve',
            'reject' => 'workorders.reject',
            'comment' => 'workorders.update',
            'parts' => 'workorders.add_parts',
            'rcfa' => 'workorders.rcfa',
            'close' => 'workorders.close',
            default => 'workorders.list',
        };

        if (empty($args['wo']) && preg_match('/\b(WO-[A-Z0-9-]+)\b/i', $command, $matches)) {
            $args['wo'] = strtoupper($matches[1]);
        }

        if ($action === 'update' || $action === 'comment') {
            if (empty($args['progress'])) {
                $args['progress'] = $this->extractProgressPercent($command);
            }
            if (empty($args['notes'])) {
                $args['notes'] = $this->extractNoteFromText($command);
            }
        }

        if ($action === 'assign' && empty($args['user'])) {
            $args['user'] = $this->extractUserMention($command);
        }

        if ($action === 'schedule') {
            if (empty($args['planned_start'])) {
                $args['planned_start'] = $this->extractDateAfterKeyword($command, 'start') ?? $this->extractDateAfterKeyword($command, 'planned start');
            }
            if (empty($args['due_at'])) {
                $args['due_at'] = $this->extractDateAfterKeyword($command, 'due');
            }
        }

        if ($action === 'pr' && empty($args['parts'])) {
            $parts = $this->extractPartsFromText($command);
            if (!$parts) {
                $parts = $this->extractPrLineItems($command);
            }
            if (!$parts) {
                $parts = $this->extractLoosePartsList($command);
            }
            if ($parts) {
                $args['parts'] = $parts;
            }
        }

        if ($action === 'close') {
            if (empty($args['notes'])) {
                $args['notes'] = $this->extractNoteFromText($command);
            }
            if (empty($args['cause'])) {
                $args['cause'] = $this->extractCauseFromText($command);
            }
            if (empty($args['downtime'])) {
                $args['downtime'] = $this->extractDowntimeHours($command);
            }
        }

        if ($action === 'reject' && empty($args['reason'])) {
            $args['reason'] = $this->extractNoteFromText($command);
        }

        if ($action === 'schedule' && empty($args['planned_start']) && empty($args['due_at'])) {
            $ref = $args['wo'] ?? $this->extractWorkOrderRefFromText($command);
            if ($ref) {
                $this->storeConversationContext($user, ['pending_action' => 'schedule', 'pending_wo' => $ref]);
                return [
                    'reply' => 'Please provide start and/or due date/time. Example: start=2026-01-28 08:00 due=2026-01-28 17:00',
                    'documents' => [],
                ];
            }
        }

        if ($action === 'pr' && empty($args['parts'])) {
            $ref = $args['wo'] ?? $this->extractWorkOrderRefFromText($command);
            if ($ref) {
                $this->storeConversationContext($user, ['pending_action' => 'pr', 'pending_wo' => $ref]);
                return [
                    'reply' => 'Reply with parts list (parts=2x belts, 1x bearing) or type "draft" to start an empty PR.',
                    'documents' => [],
                ];
            }
        }

        return $this->executeTool($tool, $args, $user);
    }

    private function handlePurchaseRequests(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/pr');
        $parts = preg_split('/\s+/', trim($command));
        $action = $parts[0] ?? 'list';
        $args = $this->parseArgs($command);

        if (empty($args['pr']) && preg_match('/\b(PR-[A-Z0-9-]+)\b/i', $command, $matches)) {
            $args['pr'] = strtoupper($matches[1]);
        }

        $tool = match ($action) {
            'create' => 'purchase_requests.create',
            'add' => 'purchase_requests.add_line',
            'submit' => 'purchase_requests.submit',
            'approve' => 'purchase_requests.approve',
            'reject' => 'purchase_requests.reject',
            'edit' => 'purchase_requests.edit',
            default => 'data.list',
        };

        if ($tool === 'data.list') {
            $args['module'] = 'purchase_requests';
        }

        return $this->executeTool($tool, $args, $user);
    }

    private function handlePurchaseOrders(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/po');
        $parts = preg_split('/\s+/', trim($command));
        $action = $parts[0] ?? 'list';
        $args = $this->parseArgs($command);

        if (empty($args['po']) && preg_match('/\b(PO-[A-Z0-9-]+)\b/i', $command, $matches)) {
            $args['po'] = strtoupper($matches[1]);
        }

        $tool = match ($action) {
            'send' => 'purchase_orders.send',
            default => 'data.list',
        };

        if ($tool === 'data.list') {
            $args['module'] = 'purchase_orders';
        }

        return $this->executeTool($tool, $args, $user);
    }

    private function handleParts(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/parts');
        $args = $this->parseArgs($command);

        return $this->executeTool('parts.search', $args, $user);
    }

    private function handleStock(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/stock');
        $args = $this->parseArgs($command);

        return $this->executeTool('stock.balance', $args, $user);
    }

    private function handleMeter(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/meter');
        $args = $this->parseArgs($command);

        return $this->executeTool('meter.create', $args, $user);
    }

    private function handleIntake(User $user): string
    {
        $recent = DocumentIntake::where('tenant_id', $user->tenant_id)
            ->orderByDesc('id')
            ->limit(3)
            ->get();

        if ($recent->isEmpty()) {
            return 'No intake batches yet. Upload a file and I will prepare a preview.';
        }

        $lines = $recent->map(function ($intake) {
            $docType = data_get($intake->meta_json, 'doc_type', 'document');
            return "Intake {$intake->id}: {$docType} ({$intake->status}).";
        })->all();

        return implode("\n", $lines) . "\nUse APPLY <intake_id> to apply.";
    }

    private function handleList(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/list');
        $args = $this->parseArgs($command);
        return $this->executeTool('data.list', $args, $user);
    }

    private function handleGet(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/get');
        $args = $this->parseArgs($command);
        return $this->executeTool('data.get', $args, $user);
    }

    private function handleExport(string $text, User $user): array
    {
        $command = $this->commandSuffix($text, '/export');
        $args = $this->parseArgs($command);
        return $this->executeTool('data.export', $args, $user);
    }

    private function handleReport(string $text, User $user): string
    {
        $command = trim($this->commandSuffix($text, '/report'));
        $query = $command !== '' ? $command : 'summary';
        return $this->reportService->report($user->tenant_id, $query);
    }

    private function handleDetails(string $text, User $user): string
    {
        $lower = Str::lower($text);
        $topic = Str::startsWith($lower, '/details')
            ? trim($this->commandSuffix($lower, '/details'))
            : trim(Str::after($lower, 'details'));
        return $this->reportService->details($user->tenant_id, trim($topic));
    }

    private function handleReconciliationCommand(string $text, User $user): string
    {
        $lower = Str::lower(trim($text));
        $action = Str::startsWith($lower, 'confirm')
            ? 'confirm'
            : (Str::startsWith($lower, 'reject') ? 'reject' : null);

        if (!$action) {
            return 'Usage: CONFIRM LINK <id> or REJECT LINK <id>';
        }

        if (!preg_match('/link\s+(\d+)/i', $text, $matches)) {
            return 'Usage: CONFIRM LINK <id> or REJECT LINK <id>';
        }

        $linkId = (int) $matches[1];
        $link = ReconciliationLink::where('tenant_id', $user->tenant_id)->find($linkId);
        if (!$link) {
            return 'Link suggestion not found.';
        }

        if ($action === 'reject') {
            $link->status = 'ignored';
            $link->applied_by = $user->id;
            $link->applied_at = now();
            $link->save();
            return "Link {$linkId} rejected.";
        }

        $this->reconciliationService->applyLink($link, $user->id);
        return "Link {$linkId} confirmed.";
    }

    private function handleGroupCommand(string $text, User $user, string $chatId, ?string $chatType): string
    {
        $command = $this->commandSuffix($text, '/group');
        $parts = preg_split('/\s+/', trim($command));
        $action = $parts[0] ?? 'list';
        $args = $this->parseArgs($command);

        if ($action === 'set' || $action === 'link') {
            if ($chatType !== 'group' && $chatType !== 'supergroup') {
                return 'Group linking must be run inside a Telegram group chat.';
            }
            return $this->groupService->linkGroup($user->tenant_id, $chatId, $args);
        }

        return $this->groupService->listGroups($user->tenant_id);
    }

    private function handleEventAction(string $text, User $user): string
    {
        $lower = Str::lower(trim($text));
        $eventCode = $this->extractEventCodeFromText($text);

        if (Str::startsWith($lower, 'ack')) {
            if (!$eventCode) {
                return 'Usage: ACK EVT-00001';
            }
            $event = $this->eventService->ackEvent($user->tenant_id, $eventCode, $user->id);
            if (!$event) {
                return 'Event not found.';
            }
            $this->logEventAction($user, 'event.ack', $eventCode);
            return "Event {$eventCode} acknowledged.";
        }

        if (Str::startsWith($lower, 'snooze')) {
            if (!$eventCode) {
                return 'Usage: SNOOZE EVT-00001 24H reason="..."';
            }
            $minutes = $this->extractSnoozeMinutes($text) ?? 60;
            $reason = $this->extractAfterKeyword($text, 'reason');
            $event = $this->eventService->snoozeEvent($user->tenant_id, $eventCode, $user->id, $minutes, $reason);
            if (!$event) {
                return 'Event not found.';
            }
            $this->logEventAction($user, 'event.snooze', $eventCode, ['minutes' => $minutes, 'reason' => $reason]);
            return "Event {$eventCode} snoozed for {$minutes} minutes.";
        }

        if (Str::startsWith($lower, 'create wo')) {
            if (!$eventCode) {
                return 'Usage: CREATE WO EVT-00001';
            }
            $order = $this->eventService->createWorkOrderFromEvent($user->tenant_id, $eventCode, $user->id);
            if (!$order) {
                return 'Unable to create work order. Check the event or asset.';
            }
            $this->logEventAction($user, 'event.create_wo', $eventCode, ['wo' => $order->reference_code]);
            return "WO created: {$order->reference_code} (pending Engineering approval). Reply APPROVE {$order->reference_code}.";
        }

        if (Str::startsWith($lower, 'dispatch')) {
            if (!$eventCode) {
                return 'Usage: DISPATCH EVT-00001 @Tech';
            }
            $assignee = $this->extractUserMention($text);
            if (!$assignee) {
                return 'Provide @Tech for dispatch.';
            }

            $event = $this->resolveEvent($user->tenant_id, $eventCode);
            if (!$event) {
                return 'Event not found.';
            }

            $order = $event->workOrder;
            if (!$order) {
                $order = $this->eventService->createWorkOrderFromEvent($user->tenant_id, $eventCode, $user->id);
            }
            if (!$order) {
                return 'Unable to create work order for dispatch.';
            }

            $result = $this->executeTool('workorders.assign', [
                'wo' => $order->reference_code,
                'user' => $assignee,
            ], $user);

            return $result['reply'] ?? 'Dispatch completed.';
        }

        if (Str::startsWith($lower, 'create pr')) {
            if (!$eventCode) {
                return 'Usage: CREATE PR EVT-00001';
            }
            $event = $this->resolveEvent($user->tenant_id, $eventCode);
            if (!$event) {
                return 'Event not found.';
            }
            return $this->createPrFromEvent($event, $user);
        }

        if (Str::startsWith($lower, 'view usage')) {
            if (!$eventCode) {
                return 'Usage: VIEW USAGE EVT-00001';
            }
            $event = $this->resolveEvent($user->tenant_id, $eventCode);
            if (!$event) {
                return 'Event not found.';
            }

            $payload = $event->payload ?? [];
            $inventoryId = $payload['inventory_item_id'] ?? null;
            if (!$inventoryId) {
                return 'No inventory item linked to this event.';
            }

            $item = InventoryItem::where('tenant_id', $user->tenant_id)
                ->with('part')
                ->where('id', $inventoryId)
                ->first();
            if (!$item || !$item->part_id) {
                return 'Inventory item not found.';
            }

            $since = now()->subDays(30);
            $qty = WorkOrderPart::where('tenant_id', $user->tenant_id)
                ->where('part_id', $item->part_id)
                ->where('created_at', '>=', $since)
                ->sum('quantity');

            $sku = $item->part?->sku ?? 'SKU';
            $name = $item->part?->name ?? '';
            return "Usage last 30d for {$sku} {$name}: " . (float) $qty;
        }

        if (Str::startsWith($lower, 'request fuel')) {
            return $this->createFuelRequest($user, $text);
        }

        if (Str::startsWith($lower, 'start rca') || Str::startsWith($lower, 'start rcfa')) {
            return 'Reply RCFA <WO> notes="..." to capture root cause.';
        }

        return 'Unsupported event action.';
    }

    private function handleModuleList(string $text, User $user): ?array
    {
        $map = [
            '/maintenance' => 'maintenance',
            '/inventory' => 'inventory',
            '/fuel' => 'fuel',
            '/tyres' => 'tyres',
            '/telemetry' => 'telemetry',
            '/alerts' => 'alerts',
            '/kpi' => 'kpi',
            '/reports' => 'reports',
            '/tenants' => 'tenants',
            '/users' => 'users',
            '/supply' => 'supply',
            '/vendors' => 'vendors',
            '/purchase_requests' => 'purchase_requests',
            '/purchase_orders' => 'purchase_orders',
        ];

        foreach ($map as $command => $module) {
            if (Str::startsWith($text, $command)) {
                $commandText = $this->commandSuffix($text, $command);
                $args = $this->parseArgs($commandText);
                $args['module'] = $module;
                return $this->executeTool('data.list', $args, $user);
            }
        }

        return null;
    }

    private function handleNaturalLanguage(string $text, User $user): array
    {
        $intent = $this->detectIntent($text, $user);
        if ($intent) {
            return $intent;
        }

        $result = $this->orchestrator->route($text, $this->userContext($user));

        $reply = $result['reply'] ?? '';
        $toolCalls = $result['tool_calls'] ?? [];
        $documents = [];

        if ($toolCalls) {
            $responses = [];
            foreach ($toolCalls as $toolCall) {
                $toolName = $toolCall['tool'] ?? '';
                $args = $toolCall['args'] ?? [];
                if (!$toolName) {
                    continue;
                }
                $toolResult = $this->executeTool($toolName, $args, $user);
                $responses[] = $toolResult['reply'] ?? null;
                if (!empty($toolResult['documents'])) {
                    $documents = array_merge($documents, $toolResult['documents']);
                }
            }

            $combined = implode("\n", array_filter($responses));
            $finalReply = $reply ? $reply . "\n" . $combined : $combined;
            return ['reply' => $finalReply, 'documents' => $documents];
        }

        return [
            'reply' => $reply ?: 'Try a slash command like /list module=assets or /wo list.',
            'documents' => [],
        ];
    }

    private function toResponse($response): array
    {
        if (is_array($response)) {
            return array_merge(['reply' => null, 'documents' => []], $response);
        }

        return ['reply' => $response, 'documents' => []];
    }

    private function handlePendingAction(string $action, string $text, User $user): ?array
    {
        $lower = Str::lower($text);
        if (Str::contains($lower, ['cancel', 'stop'])) {
            $this->storeConversationContext($user, ['pending_action' => null, 'pending_wo' => null]);
            return ['reply' => 'Cancelled.'];
        }

        $context = $this->getConversationContext($user);
        $ref = $context['pending_wo'] ?? null;
        if (!$ref) {
            $this->storeConversationContext($user, ['pending_action' => null, 'pending_wo' => null]);
            return null;
        }

        if ($action === 'schedule') {
            $start = $this->extractDateAfterKeyword($text, 'start') ?? $this->extractDateAfterKeyword($text, 'planned start');
            $due = $this->extractDateAfterKeyword($text, 'due');
            if (!$start && !$due) {
                return ['reply' => 'Send start and/or due date/time in this format: start=YYYY-MM-DD HH:MM due=YYYY-MM-DD HH:MM'];
            }

            $this->storeConversationContext($user, ['pending_action' => null, 'pending_wo' => null]);
            return $this->executeTool('workorders.update', array_filter([
                'wo' => $ref,
                'planned_start' => $start,
                'due_at' => $due,
                'status' => 'scheduled',
            ]), $user);
        }

        if ($action === 'pr') {
            if (Str::contains($lower, ['draft', 'start'])) {
                $this->storeConversationContext($user, ['pending_action' => null, 'pending_wo' => null]);
                return $this->executeTool('workorders.pr', ['wo' => $ref], $user);
            }

            $parts = $this->extractPartsFromText($text);
            if (!$parts) {
                $parts = $this->extractPrLineItems($text);
            }
            if (!$parts) {
                $parts = $this->extractLoosePartsList($text);
            }
            if (!$parts) {
                return ['reply' => 'Reply with parts list like: parts=2x belts, 1x bearing or type "draft" to create an empty PR.'];
            }

            $this->storeConversationContext($user, ['pending_action' => null, 'pending_wo' => null]);
            return $this->executeTool('workorders.pr', array_filter([
                'wo' => $ref,
                'parts' => $parts,
            ]), $user);
        }

        return null;
    }

    private function executeTool(string $tool, array $args, User $user): array
    {
        try {
            $result = $this->toolExecutor->execute($tool, $args, $this->userContext($user), 'telegram');
            $this->logTool($user, $tool, $args, $result);
            $this->rememberLastAsset($user, $tool, $args, $result);
            $this->rememberConversationContext($user, $tool, $args, $result);

            $documents = [];
            if (!empty($result['file_path'])) {
                $documents[] = [
                    'path' => $result['file_path'],
                    'filename' => $result['filename'] ?? null,
                ];
            }

            return [
                'reply' => $result['message'] ?? 'Done.',
                'documents' => $documents,
            ];
        } catch (\Throwable $e) {
            $this->logTool($user, $tool, $args, ['error' => $e->getMessage()], false);
            return ['reply' => $e->getMessage(), 'documents' => []];
        }
    }

    private function detectIntent(string $text, User $user): ?array
    {
        $lower = Str::lower($text);

        if ($this->isDeleteIntent($lower)) {
            return ['reply' => 'Deletion is not allowed on Telegram; use the web app.', 'documents' => []];
        }

        $limit = $this->parseLimitFromText($lower);
        $assetTag = $this->extractAssetTagFromText($text) ?? $this->getLastAssetTag($user);

        if ($this->isMoreIntent($lower)) {
            $context = $this->getConversationContext($user);
            $module = $context['last_module'] ?? null;
            if ($module) {
                $args = $context['last_args'] ?? ['module' => $module];
                $args['module'] = $module;
                $args['limit'] = $this->bumpLimit($args['limit'] ?? null, $limit);
                return $this->executeTool('data.list', $args, $user);
            }
        }

        $format = $this->extractExportFormat($lower);
        if (Str::contains($lower, ['export', 'download']) && $this->mentionsModuleKeywords($lower)) {
            $module = $this->extractModuleFromText($lower);
            if ($module) {
                $args = $this->exportArgsFromText($text, $module, $limit);
                $args['format'] = $format ?: 'pdf';
                return $this->executeTool('data.export', $args, $user);
            }
        }

        if (Str::contains($lower, ['export', 'download']) && !$this->mentionsModuleKeywords($lower)) {
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                $args = $this->exportArgsFromText($text, 'purchase_requests', $limit);
                $args['code'] = $pr;
                $args['format'] = $format ?: 'pdf';
                return $this->executeTool('data.export', $args, $user);
            }
            $wo = $this->extractWorkOrderRefFromText($text);
            if ($wo) {
                $args = $this->exportArgsFromText($text, 'workorders', $limit);
                $args['wo'] = $wo;
                $args['format'] = $format ?: 'pdf';
                return $this->executeTool('data.export', $args, $user);
            }
            $context = $this->getConversationContext($user);
            $module = $context['last_module'] ?? null;
            if ($module) {
                $args = $this->mergeExportContext($module, $context['last_args'] ?? []);
                $args['format'] = $format ?: 'pdf';
                if ($limit) {
                    $args['limit'] = $limit;
                }
                return $this->executeTool('data.export', $args, $user);
            }
        }

        if (Str::contains($lower, ['export', 'download'])) {
            return [
                'reply' => 'Tell me what to export (assets, work orders, inventory, purchase requests). Example: "Export open work orders as Excel."',
                'documents' => [],
            ];
        }

        if ($this->startsWithVerb($lower, 'pr')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                $parts = $this->extractPartsFromText($text);
                if (!$parts) {
                    $parts = $this->extractPrLineItems($text);
                }
                if (!$parts) {
                    $parts = $this->extractLoosePartsList($text);
                }
                if (!$parts) {
                    $this->storeConversationContext($user, ['pending_action' => 'pr', 'pending_wo' => $ref]);
                    return [
                        'reply' => 'Reply with parts list (parts=2x belts, 1x bearing) or type "draft" to start an empty PR.',
                        'documents' => [],
                    ];
                }
                return $this->executeTool('workorders.pr', array_filter([
                    'wo' => $ref,
                    'parts' => $parts,
                ]), $user);
            }
        }

        if ($this->startsWithVerb($lower, 'assign')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            $assignee = $this->extractUserMention($text);
            $args = ['wo' => $ref, 'user' => $assignee];
            if ($priority = $this->extractPriority($lower)) {
                $args['priority'] = $priority;
            }
            if ($due = $this->extractDateAfterKeyword($text, 'due')) {
                $args['due_at'] = $due;
            }
            return $this->executeTool('workorders.assign', array_filter($args), $user);
        }

        if ($this->startsWithVerb($lower, 'schedule')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                $start = $this->extractDateAfterKeyword($text, 'start') ?? $this->extractDateAfterKeyword($text, 'planned start');
                $due = $this->extractDateAfterKeyword($text, 'due');
                if (!$start && !$due) {
                    $this->storeConversationContext($user, ['pending_action' => 'schedule', 'pending_wo' => $ref]);
                    return [
                        'reply' => 'Please provide start and/or due date/time. Example: start=2026-01-28 08:00 due=2026-01-28 17:00',
                        'documents' => [],
                    ];
                }
                return $this->executeTool('workorders.update', array_filter([
                    'wo' => $ref,
                    'planned_start' => $start,
                    'due_at' => $due,
                    'status' => 'scheduled',
                ]), $user);
            }
        }

        if ($this->startsWithVerb($lower, 'accept')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.accept', ['wo' => $ref], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'update')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.update', [
                    'wo' => $ref,
                    'progress' => $this->extractProgressPercent($text),
                    'notes' => $this->extractNoteFromText($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'comment') || $this->startsWithVerb($lower, 'note')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.update', [
                    'wo' => $ref,
                    'notes' => $this->extractNoteFromText($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'close')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.close', [
                    'wo' => $ref,
                    'notes' => $this->extractNoteFromText($text),
                    'cause' => $this->extractCauseFromText($text),
                    'downtime' => $this->extractDowntimeHours($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'rcfa')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.rcfa', [
                    'wo' => $ref,
                    'notes' => $this->extractNoteFromText($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'approve')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.approve', ['wo' => $ref], $user);
            }
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                $notes = $this->extractAfterKeyword($text, 'notes') ?? $this->extractAfterKeyword($text, 'reason') ?? $this->extractQuoted($text);
                return $this->executeTool('purchase_requests.approve', array_filter([
                    'pr' => $pr,
                    'notes' => $notes,
                ]), $user);
            }
        }

        if ($this->startsWithVerb($lower, 'reject') || $this->startsWithVerb($lower, 'decline')) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.reject', [
                    'wo' => $ref,
                    'reason' => $this->extractNoteFromText($text),
                ], $user);
            }
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                return $this->executeTool('purchase_requests.reject', [
                    'pr' => $pr,
                    'reason' => $this->extractNoteFromText($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'submit')) {
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                return $this->executeTool('purchase_requests.submit', ['pr' => $pr], $user);
            }
        }

        if (Str::contains($lower, ['convert']) && Str::contains($lower, ['to po', 'to purchase order'])) {
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                return $this->executeTool('purchase_orders.convert', [
                    'pr' => $pr,
                    'vendor' => $this->extractQuoted($text) ?: $this->extractVendorFromText($text),
                    'delivery' => $this->extractDeliveryFromText($text),
                    'terms' => $this->extractTermsFromText($text),
                ], $user);
            }
        }

        if ($this->startsWithVerb($lower, 'send')) {
            $po = $this->extractPurchaseOrderRef($text);
            if ($po) {
                return $this->executeTool('purchase_orders.send', ['po' => $po], $user);
            }
        }

        if (Str::contains($lower, ['grn'])) {
            $po = $this->extractPurchaseOrderRef($text);
            if ($po) {
                $reference = $this->extractReferenceFromText($text) ?: ('GRN-' . $po);
                return $this->executeTool('grn.create', [
                    'po' => $po,
                    'reference' => $reference,
                ], $user);
            }
        }

        if (Str::contains($lower, ['add parts', 'add part'])) {
            $ref = $this->extractWorkOrderRefFromText($text);
            if ($ref) {
                return $this->executeTool('workorders.add_parts', [
                    'wo' => $ref,
                    'sku' => $this->extractSkuFromText($text),
                    'qty' => $this->extractQuantityFromText($text),
                ], $user);
            }
        }

        if (Str::contains($lower, ['work order', 'workorder', 'wo'])) {
            if (Str::contains($lower, ['create', 'log', 'new'])) {
                if ($assetTag && $this->extractTitleFromText($text)) {
                    $args = [
                        'asset' => $assetTag,
                        'title' => $this->extractTitleFromText($text),
                        'description' => $this->extractNoteFromText($text),
                        'priority' => $this->extractPriority($lower),
                    ];
                    $parts = $this->extractPartsFromText($text);
                    if ($parts) {
                        $args['parts'] = $parts;
                    }
                    return $this->executeTool('workorders.create', array_filter($args), $user);
                }

                return ['reply' => 'To create a work order, provide asset tag and title. Example: /wo create asset=BP-01 title="Hydraulic leak"', 'documents' => []];
            }

            $args = ['module' => 'workorders'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($status = $this->extractStatus($lower)) {
                $args['status'] = $status;
            }
            if ($priority = $this->extractPriority($lower)) {
                $args['priority'] = $priority;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['maintenance', 'pm', 'service'])) {
            $args = ['module' => 'maintenance'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($status = $this->extractStatus($lower)) {
                $args['status'] = $status;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['fuel', 'diesel'])) {
            $args = ['module' => 'fuel'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($site = $this->extractSiteFromText($text)) {
                $args['site'] = $site;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['tyre', 'tire'])) {
            $args = ['module' => 'tyres'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($status = $this->extractStatus($lower)) {
                $args['status'] = $status;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['telemetry', 'meter', 'hours'])) {
            $args = ['module' => 'telemetry'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['purchase request', 'pr'])) {
            if (Str::contains($lower, ['create', 'raise', 'new'])) {
                $createArgs = [
                    'department' => $this->extractAfterKeyword($text, 'dept') ?? $this->extractAfterKeyword($text, 'department'),
                    'cost_center' => $this->extractAfterKeyword($text, 'cost center'),
                    'priority' => $this->extractPriority($lower),
                    'needed_by' => $this->extractAfterKeyword($text, 'needed by'),
                    'reason' => $this->extractAfterKeyword($text, 'for') ?? $this->extractQuoted($text),
                ];

                $created = $this->executeTool('purchase_requests.create', array_filter($createArgs), $user);
                $reply = $created['reply'] ?? '';
                $prCode = $this->extractPurchaseRequestRef($reply);

                $lineItems = $this->extractPrLineItems($text);
                if ($prCode && $lineItems) {
                    foreach ($lineItems as $item) {
                        $this->executeTool('purchase_requests.add_line', [
                            'pr' => $prCode,
                            'description' => $item['description'],
                            'qty' => $item['quantity'],
                        ], $user);
                    }
                    $this->executeTool('purchase_requests.submit', ['pr' => $prCode], $user);
                    $reply .= "\nLines added. PR submitted for approval.";
                }

                return ['reply' => $reply, 'documents' => []];
            }

            if (Str::contains($lower, ['submit'])) {
                $prCode = $this->extractPurchaseRequestRef($text);
                if ($prCode) {
                    return $this->executeTool('purchase_requests.submit', ['pr' => $prCode], $user);
                }
            }

            if (Str::contains($lower, ['approve'])) {
                $prCode = $this->extractPurchaseRequestRef($text);
                if ($prCode) {
                    return $this->executeTool('purchase_requests.approve', ['pr' => $prCode], $user);
                }
            }

            if (Str::contains($lower, ['reject'])) {
                $prCode = $this->extractPurchaseRequestRef($text);
                if ($prCode) {
                    return $this->executeTool('purchase_requests.reject', [
                        'pr' => $prCode,
                        'reason' => $this->extractNoteFromText($text),
                    ], $user);
                }
            }
        }

        if (Str::contains($lower, ['maintenance', 'service', 'pm', 'inspection'])) {
            $args = ['module' => 'maintenance'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($status = $this->extractStatus($lower)) {
                $args['status'] = $status;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['fuel', 'diesel', 'petrol', 'refuel'])) {
            $format = $this->extractExportFormat($lower);
            if (Str::contains($lower, ['export', 'download']) || $format) {
                $args = [
                    'module' => 'fuel',
                    'format' => $format ?: 'pdf',
                ];
                if ($assetTag) {
                    $args['asset'] = $assetTag;
                }
                if ($site = $this->extractSiteFromText($text)) {
                    $args['site'] = $site;
                }
                if ($limit) {
                    $args['limit'] = $limit;
                }
                return $this->executeTool('data.export', $args, $user);
            }

            $args = ['module' => 'fuel'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($site = $this->extractSiteFromText($text)) {
                $args['site'] = $site;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['tyre', 'tire'])) {
            $args = ['module' => 'tyres'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($status = $this->extractStatus($lower)) {
                $args['status'] = $status;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['telemetry', 'runtime', 'usage hours', 'usage'])) {
            $args = ['module' => 'telemetry'];
            if ($assetTag) {
                $args['asset'] = $assetTag;
            }
            if ($limit) {
                $args['limit'] = $limit;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['inventory', 'stock'])) {
            if (Str::contains($lower, ['balance', 'qty', 'quantity'])) {
                $sku = $this->extractSkuFromText($text);
                if ($sku) {
                    return $this->executeTool('stock.balance', ['sku' => $sku], $user);
                }
            }

            $args = ['module' => 'inventory'];
            if ($limit) {
                $args['limit'] = $limit;
            }
            if ($query = $this->extractQuoted($text)) {
                $args['q'] = $query;
            }

            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['parts', 'part'])) {
            $query = $this->extractQuoted($text) ?? $this->extractSearchTerm($lower, ['parts', 'part']);
            if ($query) {
                return $this->executeTool('parts.search', ['q' => $query], $user);
            }

            $args = ['module' => 'parts'];
            if ($limit) {
                $args['limit'] = $limit;
            }
            return $this->executeTool('data.list', $args, $user);
        }

        if (Str::contains($lower, ['report', 'reports'])) {
            if (Str::contains($lower, ['export', 'download']) || $format) {
                $args = [
                    'module' => 'reports',
                    'format' => $format ?: 'pdf',
                ];
                if ($limit) {
                    $args['limit'] = $limit;
                }
                return $this->executeTool('data.export', $args, $user);
            }

            $args = ['module' => 'reports'];
            if ($limit) {
                $args['limit'] = $limit;
            }
            return $this->executeTool('data.list', $args, $user);
        }

        return null;
    }

    private function applyAssetContext(string $text, User $user): string
    {
        $tag = $this->getLastAssetTag($user);
        if (!$tag) {
            return $text;
        }

        $replacements = [
            'last asset',
            'previous asset',
            'that asset',
            'this asset',
        ];

        foreach ($replacements as $phrase) {
            if (stripos($text, $phrase) !== false) {
                return str_ireplace($phrase, $tag, $text);
            }
        }

        if (Str::startsWith($text, '/wo') && stripos($text, 'asset=') === false) {
            return $text . " asset={$tag}";
        }

        if (Str::startsWith($text, '/meter') && stripos($text, 'asset=') === false) {
            return $text . " asset={$tag}";
        }

        return $text;
    }

    private function applyEntityContext(string $text, User $user): string
    {
        $context = $this->getConversationContext($user);

        $replacements = [
            'work order' => $context['last_workorder_ref'] ?? null,
            'workorder' => $context['last_workorder_ref'] ?? null,
            'purchase request' => $context['last_pr_code'] ?? null,
            'purchase order' => $context['last_po_number'] ?? null,
            'po' => $context['last_po_number'] ?? null,
            'vendor' => $context['last_vendor_name'] ?? null,
            'part' => $context['last_part_sku'] ?? null,
            'sku' => $context['last_part_sku'] ?? null,
        ];

        foreach ($replacements as $label => $value) {
            if (!$value) {
                continue;
            }
            foreach (['last', 'previous', 'that', 'this'] as $prefix) {
                $phrase = $prefix . ' ' . $label;
                if (stripos($text, $phrase) !== false) {
                    return str_ireplace($phrase, (string) $value, $text);
                }
            }
        }

        return $text;
    }

    private function rememberAssetFromText(string $text, User $user): void
    {
        $tag = $this->extractAssetTagFromText($text);
        if ($tag) {
            $this->storeLastAsset($user, $tag);
        }
    }

    private function rememberLastAsset(User $user, string $tool, array $args, array $result): void
    {
        $candidate = $args['asset'] ?? $args['asset_tag'] ?? $args['tag'] ?? null;
        if ($candidate) {
            $this->storeLastAsset($user, (string) $candidate);
            return;
        }

        if ($tool === 'data.get' && ($args['module'] ?? '') === 'assets') {
            $tag = $this->extractTagFromRows($result['rows'] ?? []);
            if ($tag) {
                $this->storeLastAsset($user, $tag);
            }
            return;
        }

        if ($tool === 'data.list' && ($args['module'] ?? '') === 'assets') {
            $tag = $this->extractTagFromRows($result['rows'] ?? []);
            if ($tag) {
                $this->storeLastAsset($user, $tag);
            }
        }
    }

    private function rememberConversationContext(User $user, string $tool, array $args, array $result): void
    {
        $updates = [];

        if (in_array($tool, ['data.list', 'data.get', 'data.export'], true)) {
            $module = $args['module'] ?? null;
            if ($module) {
                $updates['last_module'] = $module;
            }
            $updates['last_args'] = $args;

            if (isset($args['site'])) {
                $updates['last_site'] = $args['site'];
            }
            if (isset($args['location'])) {
                $updates['last_location'] = $args['location'];
            }
            if (isset($args['asset'])) {
                $updates['last_asset_tag'] = $args['asset'];
            }
        }

        if ($tool === 'workorders.list') {
            $updates['last_module'] = 'workorders';
            $updates['last_args'] = array_merge(['module' => 'workorders'], $args);
        }

        if ($tool === 'workorders.create' && !empty($result['message'])) {
            if (preg_match('/work order\\s+([A-Z0-9-]+)/i', $result['message'], $matches)) {
                $updates['last_workorder_ref'] = $matches[1];
            }
        }

        if (Str::startsWith($tool, 'purchase_requests.')) {
            $updates['last_module'] = 'purchase_requests';
            $updates['last_args'] = array_merge(['module' => 'purchase_requests'], $args);
        }

        if (!empty($result['message'])) {
            if (preg_match('/\b(PR-[A-Z0-9-]+)\b/i', $result['message'], $matches)) {
                $updates['last_pr_code'] = strtoupper($matches[1]);
            }
            if (preg_match('/\b(PO-[A-Z0-9-]+)\b/i', $result['message'], $matches)) {
                $updates['last_po_number'] = strtoupper($matches[1]);
            }
        }

        $rows = $result['rows'] ?? null;
        if (is_array($rows) && !empty($rows)) {
            $module = $updates['last_module'] ?? ($args['module'] ?? null);
            $first = $rows[0] ?? null;
            if ($module === 'workorders' && is_array($first)) {
                $updates['last_workorder_ref'] = $first[0] ?? ($updates['last_workorder_ref'] ?? null);
                $updates['last_asset_tag'] = $first[1] ?? ($updates['last_asset_tag'] ?? null);
            } elseif (in_array($module, ['inventory', 'parts'], true) && is_array($first)) {
                $updates['last_part_sku'] = $first[0] ?? ($updates['last_part_sku'] ?? null);
            } elseif ($module === 'purchase_requests' && is_array($first)) {
                $updates['last_pr_code'] = $first[0] ?? ($updates['last_pr_code'] ?? null);
            } elseif ($module === 'vendors' && is_array($first)) {
                $updates['last_vendor_name'] = $first[0] ?? ($updates['last_vendor_name'] ?? null);
            } elseif ($module === 'purchase_orders' && is_array($first)) {
                $updates['last_po_number'] = $first[0] ?? ($updates['last_po_number'] ?? null);
            } elseif ($module === 'assets' && is_array($first)) {
                $updates['last_asset_tag'] = $first[0] ?? ($updates['last_asset_tag'] ?? null);
            }
        }

        if ($updates) {
            $this->storeConversationContext($user, $updates);
        }
    }

    private function storeLastAsset(User $user, string $tag): void
    {
        $tag = trim($tag);
        if ($tag === '') {
            return;
        }

        Cache::put($this->lastAssetCacheKey($user), $tag, now()->addHours(12));
        $this->storeConversationContext($user, ['last_asset_tag' => $tag]);
    }

    private function getLastAssetTag(User $user): ?string
    {
        return Cache::get($this->lastAssetCacheKey($user));
    }

    private function lastAssetCacheKey(User $user): string
    {
        return 'telegram:last_asset:' . $user->id;
    }

    private function conversationContextKey(User $user): string
    {
        return 'telegram:context:' . $user->id;
    }

    private function getConversationContext(User $user): array
    {
        return Cache::get($this->conversationContextKey($user), []);
    }

    private function storeConversationContext(User $user, array $updates): void
    {
        $context = $this->getConversationContext($user);
        foreach ($updates as $key => $value) {
            if ($value === null) {
                unset($context[$key]);
                continue;
            }
            $context[$key] = $value;
        }
        Cache::put($this->conversationContextKey($user), $context, now()->addHours(12));
    }

    private function isMoreIntent(string $text): bool
    {
        return Str::contains($text, ['more', 'next', 'another', 'continue']);
    }

    private function bumpLimit($current, ?int $requested): int
    {
        $current = is_numeric($current) ? (int) $current : 5;
        $bump = $requested ?: 5;
        return min($current + $bump, 50);
    }

    private function mentionsModuleKeywords(string $text): bool
    {
        return Str::contains($text, [
            'asset', 'work order', 'workorder', 'maintenance', 'inventory', 'stock',
            'fuel', 'tyre', 'tire', 'telemetry', 'alert', 'report', 'vendor', 'part',
            'purchase request', 'purchase order', 'po', 'pr', 'user', 'tenant',
        ]);
    }

    private function mergeExportContext(string $module, array $lastArgs): array
    {
        $allowed = ['asset', 'status', 'priority', 'site', 'location', 'type', 'q', 'from', 'to'];
        $args = ['module' => $module];
        foreach ($allowed as $key) {
            if (isset($lastArgs[$key])) {
                $args[$key] = $lastArgs[$key];
            }
        }

        return $args;
    }

    private function extractTagFromRows(array $rows): ?string
    {
        if (!$rows) {
            return null;
        }

        $first = $rows[0] ?? null;
        if (is_array($first)) {
            $tag = $first[0] ?? null;
            if (is_string($tag) && $tag !== '-') {
                return $tag;
            }
        }

        foreach ($rows as $row) {
            if (isset($row[0], $row[1]) && $row[0] === 'Tag' && is_string($row[1])) {
                return $row[1];
            }
        }

        return null;
    }

    private function extractAssetTagFromText(string $text): ?string
    {
        if (!preg_match('/\\b[A-Z0-9]{1,6}(?:-[A-Z0-9]{1,6}){1,2}\\b/', strtoupper($text), $matches)) {
            return null;
        }

        $tag = strtoupper($matches[0]);
        if (Str::startsWith($tag, ['WO-', 'PR-'])) {
            return null;
        }

        return $tag;
    }

    private function parseLimitFromText(string $text): ?int
    {
        if (preg_match('/\\b(top|first|latest)\\s+(\\d{1,3})\\b/i', $text, $matches)) {
            return (int) $matches[2];
        }
        if (preg_match('/\\b(\\d{1,3})\\s+(items|records|assets|work orders|workorders|parts|reports)\\b/i', $text, $matches)) {
            return (int) $matches[1];
        }

        return null;
    }

    private function extractExportFormat(string $text): ?string
    {
        if (Str::contains($text, ['xlsx', 'excel'])) {
            return 'xlsx';
        }
        if (Str::contains($text, ['pdf'])) {
            return 'pdf';
        }

        return null;
    }

    private function exportArgsFromText(string $text, string $module, ?int $limit): array
    {
        $args = $this->parseArgs($text);
        $normalized = ['module' => $module];

        foreach (['status', 'priority', 'site', 'location', 'type', 'q', 'from', 'to', 'asset', 'asset_tag', 'tag', 'department', 'cost_center', 'needed_by', 'code', 'request_code'] as $key) {
            if (isset($args[$key])) {
                $normalized[$key] = $args[$key];
            }
        }

        if (empty($normalized['asset'])) {
            $assetTag = $this->extractAssetTagFromText($text);
            if ($assetTag) {
                $normalized['asset'] = $assetTag;
            }
        }

        if (empty($normalized['status'])) {
            $status = $this->extractStatus(Str::lower($text));
            if ($status) {
                $normalized['status'] = $status;
            }
        }

        if (empty($normalized['priority'])) {
            $priority = $this->extractPriority(Str::lower($text));
            if ($priority) {
                $normalized['priority'] = $priority;
            }
        }

        if (empty($normalized['site'])) {
            $site = $this->extractSiteFromText($text);
            if ($site) {
                $normalized['site'] = $site;
            }
        }

        if (empty($normalized['location'])) {
            $location = $this->extractAfterKeyword($text, 'location');
            if ($location) {
                $normalized['location'] = $location;
            }
        }

        if (empty($normalized['from'])) {
            $from = $this->extractDateAfterKeyword($text, 'from');
            if ($from) {
                $normalized['from'] = $from;
            }
        }

        if (empty($normalized['to'])) {
            $to = $this->extractDateAfterKeyword($text, 'to');
            if ($to) {
                $normalized['to'] = $to;
            }
        }

        if ($limit) {
            $normalized['limit'] = $limit;
        }

        if ($module === 'purchase_requests' && empty($normalized['code'])) {
            $pr = $this->extractPurchaseRequestRef($text);
            if ($pr) {
                $normalized['code'] = $pr;
            }
        }

        if ($module === 'workorders' && empty($normalized['wo'])) {
            $wo = $this->extractWorkOrderRefFromText($text);
            if ($wo) {
                $normalized['wo'] = $wo;
            }
        }

        return $normalized;
    }

    private function extractModuleFromText(string $text): ?string
    {
        $map = [
            'work order' => 'workorders',
            'work orders' => 'workorders',
            'workorder' => 'workorders',
            'wo' => 'workorders',
            'maintenance' => 'maintenance',
            'pm' => 'maintenance',
            'inventory' => 'inventory',
            'stock' => 'inventory',
            'fuel' => 'fuel',
            'tyre' => 'tyres',
            'tire' => 'tyres',
            'telemetry' => 'telemetry',
            'alert' => 'alerts',
            'alarm' => 'alerts',
            'kpi' => 'kpi',
            'report' => 'reports',
            'tenant' => 'tenants',
            'user' => 'users',
            'purchase request' => 'purchase_requests',
            'pr' => 'purchase_requests',
            'purchase order' => 'purchase_orders',
            'po' => 'purchase_orders',
            'vendor' => 'vendors',
            'part' => 'parts',
            'supply' => 'supply',
            'asset' => 'assets',
        ];

        foreach ($map as $keyword => $module) {
            if (Str::contains($text, $keyword)) {
                return $module;
            }
        }

        return null;
    }

    private function extractPriority(string $text): ?string
    {
        if (Str::contains($text, 'high')) {
            return 'high';
        }
        if (Str::contains($text, 'low')) {
            return 'low';
        }
        if (Str::contains($text, 'medium')) {
            return 'medium';
        }

        return null;
    }

    private function extractStatus(string $text): ?string
    {
        foreach (['open', 'closed', 'completed', 'pending', 'overdue'] as $status) {
            if (Str::contains($text, $status)) {
                return $status;
            }
        }

        return null;
    }

    private function extractTitleFromText(string $text): ?string
    {
        $quoted = $this->extractQuoted($text);
        if ($quoted) {
            return $quoted;
        }

        if (preg_match('/for\\s+([\\w\\s-]{4,})$/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractSiteFromText(string $text): ?string
    {
        if (preg_match('/site\\s*[:=]\\s*([A-Za-z0-9\\s\\-]+)/i', $text, $matches)) {
            $value = trim($matches[1]);
            return $this->looksLikeAssetTag($value) ? null : $value;
        }

        $quoted = $this->extractQuoted($text);
        if ($quoted && !$this->looksLikeAssetTag($quoted)) {
            return $quoted;
        }

        if (preg_match('/\\bat\\s+([A-Za-z0-9\\s\\-]{2,})$/i', $text, $matches)) {
            $value = trim($matches[1]);
            return $this->looksLikeAssetTag($value) ? null : $value;
        }

        return null;
    }

    private function looksLikeAssetTag(string $value): bool
    {
        return (bool) preg_match('/^[A-Z0-9]{1,6}(?:-[A-Z0-9]{1,6}){1,2}$/', strtoupper($value));
    }

    private function extractSkuFromText(string $text): ?string
    {
        if (preg_match('/\\b[A-Z0-9]{2,}-[A-Z0-9-]{2,}\\b/', strtoupper($text), $matches)) {
            return $matches[0];
        }

        return null;
    }

    private function extractQuoted(string $text): ?string
    {
        if (preg_match('/\"([^\"]+)\"/', $text, $matches)) {
            return trim($matches[1]);
        }
        if (preg_match("/'([^']+)'/", $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractSearchTerm(string $text, array $keywords): ?string
    {
        foreach ($keywords as $keyword) {
            if (strpos($text, $keyword) !== false) {
                $parts = explode($keyword, $text);
                $tail = trim($parts[1] ?? '');
                if ($tail) {
                    return trim($tail);
                }
            }
        }

        return null;
    }

    private function startsWithVerb(string $text, string $verb): bool
    {
        return Str::startsWith(trim($text), $verb . ' ');
    }

    private function extractWorkOrderRefFromText(string $text): ?string
    {
        if (preg_match('/\b(WO-[A-Z0-9-]+)\b/i', $text, $matches)) {
            return strtoupper($matches[1]);
        }

        return null;
    }

    private function extractPurchaseRequestRef(string $text): ?string
    {
        if (preg_match('/\b(PR-[A-Z0-9-]+)\b/i', $text, $matches)) {
            return strtoupper($matches[1]);
        }

        return null;
    }

    private function extractPurchaseOrderRef(string $text): ?string
    {
        if (preg_match('/\b(PO-[A-Z0-9-]+)\b/i', $text, $matches)) {
            return strtoupper($matches[1]);
        }

        return null;
    }

    private function extractUserMention(string $text): ?string
    {
        if (preg_match('/@([A-Za-z0-9._-]+)/', $text, $matches)) {
            return $matches[1];
        }

        return null;
    }

    private function extractProgressPercent(string $text): ?float
    {
        if (preg_match('/(\d{1,3})\s*%/', $text, $matches)) {
            return (float) $matches[1];
        }

        return null;
    }

    private function extractNoteFromText(string $text): ?string
    {
        $quoted = $this->extractQuoted($text);
        if ($quoted) {
            return $quoted;
        }

        $parts = preg_split('/\b(wo-[a-z0-9-]+)\b/i', $text);
        if (!empty($parts[1])) {
            return trim($parts[1]);
        }

        return null;
    }

    private function extractCauseFromText(string $text): ?string
    {
        if (preg_match('/cause\s*[:=]\s*([A-Za-z0-9\s-]+)/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractDowntimeHours(string $text): ?float
    {
        if (preg_match('/downtime\s*[:=]?\s*([0-9.]+)\s*h/i', $text, $matches)) {
            return (float) $matches[1];
        }

        return null;
    }

    private function extractDateAfterKeyword(string $text, string $keyword): ?string
    {
        if (preg_match('/' . preg_quote($keyword, '/') . '\s*[:=]?\s*([A-Za-z0-9\-\s]+)/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractVendorFromText(string $text): ?string
    {
        if (preg_match('/vendor\s*[:=]\s*([A-Za-z0-9\s-]+)/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractDeliveryFromText(string $text): ?string
    {
        if (preg_match('/delivery\s*[:=]\s*([A-Za-z0-9\s-]+)/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractTermsFromText(string $text): ?string
    {
        if (preg_match('/terms\s*[:=]\s*([A-Za-z0-9\s-]+)/i', $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractReferenceFromText(string $text): ?string
    {
        if (preg_match('/\b(GRN-[A-Z0-9-]+)\b/i', $text, $matches)) {
            return strtoupper($matches[1]);
        }

        return null;
    }

    private function extractQuantityFromText(string $text): ?float
    {
        if (preg_match('/\b(\d+(\.\d+)?)\b/', $text, $matches)) {
            return (float) $matches[1];
        }

        return null;
    }

    private function extractAfterKeyword(string $text, string $keyword): ?string
    {
        $pattern = '/' . preg_quote($keyword, '/') . '\s*[:=]?\s*([A-Za-z0-9\\s\\-]+)/i';
        if (preg_match($pattern, $text, $matches)) {
            return trim($matches[1]);
        }

        return null;
    }

    private function extractPrLineItems(string $text): array
    {
        $items = [];
        $segments = preg_split('/[;,]/', $text);
        foreach ($segments as $segment) {
            if (preg_match('/\b(\d+(?:\.\d+)?)\s+([A-Za-z0-9\\s\\-]+)/', trim($segment), $matches)) {
                $qty = (float) $matches[1];
                $desc = trim($matches[2]);
                if ($desc !== '') {
                    $items[] = ['quantity' => $qty, 'description' => $desc];
                }
            }
            if (preg_match('/\b(\d+(?:\.\d+)?)\s*x\s*(.+)/i', trim($segment), $matches)) {
                $qty = (float) $matches[1];
                $desc = trim($matches[2]);
                if ($desc !== '') {
                    $items[] = ['quantity' => $qty, 'description' => $desc];
                }
            }
            if (preg_match('/(.+?)\s*x\s*(\d+(?:\.\d+)?)/i', trim($segment), $matches)) {
                $desc = trim($matches[1]);
                $qty = (float) $matches[2];
                if ($desc !== '') {
                    $items[] = ['quantity' => $qty, 'description' => $desc];
                }
            }
        }

        return $items;
    }

    private function extractLoosePartsList(string $text): array
    {
        $segments = preg_split('/[;,]|\\band\\b/i', $text);
        $items = [];
        foreach ($segments as $segment) {
            $desc = trim($segment);
            if ($desc === '') {
                continue;
            }
            if (preg_match('/\\bWO-[A-Z0-9-]+\\b/i', $desc)) {
                continue;
            }
            if (preg_match('/^\\s*pr\\s*$/i', $desc)) {
                continue;
            }
            if (preg_match('/^\\s*pr\\s+wo-[A-Z0-9-]+/i', $desc)) {
                continue;
            }
            $items[] = ['quantity' => 1, 'description' => $desc];
        }

        return $items;
    }

    private function extractPartsFromText(string $text): array
    {
        $lower = Str::lower($text);
        $keyword = null;
        foreach (['parts', 'spares', 'resources'] as $candidate) {
            if (Str::contains($lower, $candidate)) {
                $keyword = $candidate;
                break;
            }
        }

        if (!$keyword) {
            return [];
        }

        $pos = stripos($text, $keyword);
        $tail = $pos !== false ? substr($text, $pos + strlen($keyword)) : $text;
        $tail = trim(ltrim($tail, ':'));

        $parts = [];
        $segments = preg_split('/[;,]/', $tail);
        foreach ($segments as $segment) {
            $pieces = preg_split('/\\band\\b/i', $segment);
            foreach ($pieces as $piece) {
                $piece = trim($piece);
                if ($piece === '') {
                    continue;
                }

                if (preg_match('/(\\d+(?:\\.\\d+)?)\\s+(.+)/', $piece, $matches)) {
                    $parts[] = [
                        'qty' => (float) $matches[1],
                        'description' => trim($matches[2]),
                    ];
                    continue;
                }

                if (preg_match('/(.+)\\s+x\\s*(\\d+(?:\\.\\d+)?)/', $piece, $matches)) {
                    $parts[] = [
                        'qty' => (float) $matches[2],
                        'description' => trim($matches[1]),
                    ];
                    continue;
                }
            }
        }

        return array_values(array_filter($parts, fn ($part) => !empty($part['description'])));
    }

    private function extractPartsFromRow(array $data): array
    {
        $items = $data['parts'] ?? $data['items'] ?? $data['spares'] ?? null;
        if (is_string($items)) {
            $decoded = json_decode($items, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $items = $decoded;
            }
        }

        if (!is_array($items)) {
            return [];
        }

        $parts = [];
        foreach ($items as $item) {
            if (is_string($item)) {
                $desc = trim($item);
                if ($desc !== '') {
                    $parts[] = ['description' => $desc, 'qty' => 1];
                }
                continue;
            }

            if (!is_array($item)) {
                continue;
            }

            $parts[] = [
                'sku' => $item['sku'] ?? null,
                'description' => $item['description'] ?? $item['item'] ?? $item['name'] ?? null,
                'qty' => $item['qty'] ?? $item['quantity'] ?? 1,
                'unit' => $item['unit'] ?? $item['uom'] ?? null,
            ];
        }

        return array_values(array_filter($parts, fn ($part) => !empty($part['description']) || !empty($part['sku'])));
    }

    private function isWorkOrderUploadIntent(string $text): bool
    {
        $lower = Str::lower($text);
        return Str::contains($lower, ['work order', 'wo', 'defect', 'job card', 'leak', 'breakdown']);
    }

    private function createWorkOrderFromIntake(array $result, User $user, string $text): ?array
    {
        $batchId = $result['batch_id'] ?? null;
        if (!$batchId) {
            return null;
        }

        $batch = \App\Models\DocumentIntakeBatch::find($batchId);
        if (!$batch) {
            return null;
        }

        $row = $batch->rows()->first();
        if (!$row) {
            return null;
        }

        $data = $row->data ?? [];
        $intakeId = $result['intake_id'] ?? null;
        $assetTag = $data['asset_tag'] ?? $this->extractAssetTagFromText($text);
        $description = $data['description'] ?? $data['issue'] ?? $text;

        if (!$assetTag) {
            return ['reply' => 'Upload received. I could not detect an asset tag - please reply with asset tag to create the work order.', 'documents' => []];
        }

        $asset = \App\Models\Asset::where('tenant_id', $user->tenant_id)
            ->where('asset_tag', $assetTag)
            ->first();
        if (!$asset) {
            return ['reply' => 'Asset not found for tag ' . $assetTag . '. Reply with a valid asset tag.', 'documents' => []];
        }

        $parts = array_merge(
            $this->extractPartsFromText($text),
            $this->extractPartsFromRow($data)
        );

        $args = [
            'asset' => $assetTag,
            'title' => Str::limit($description, 80, ''),
            'description' => $description,
            'priority' => $this->extractPriority(Str::lower($text)) ?? 'medium',
        ];

        if (!empty($parts)) {
            $args['parts'] = $parts;
        }

        $toolResult = $this->executeTool('workorders.create', array_filter($args), $user);
        $reply = $toolResult['reply'] ?? '';
        $documents = $toolResult['documents'] ?? [];

        $ref = $this->extractWorkOrderRefFromText($reply);
        if ($ref) {
            $attached = $this->attachIntakeToWorkOrder($ref, $intakeId, $user);
            if ($attached) {
                $reply = trim($reply . "\nEvidence attached.");
            }
        }

        return ['reply' => $reply, 'documents' => $documents];
    }

    private function attachIntakeToWorkOrder(string $reference, ?int $intakeId, User $user): bool
    {
        if (!$intakeId) {
            return false;
        }

        $order = \App\Models\WorkOrder::where('tenant_id', $user->tenant_id)
            ->where('reference_code', $reference)
            ->first();
        if (!$order) {
            return false;
        }

        $intake = \App\Models\DocumentIntake::find($intakeId);
        if (!$intake) {
            return false;
        }

        \App\Models\WorkOrderDocument::firstOrCreate([
            'tenant_id' => $user->tenant_id,
            'work_order_id' => $order->id,
            'storage_path' => $intake->stored_path,
        ], [
            'original_name' => $intake->original_filename,
            'uploaded_by' => $user->id,
            'source' => 'telegram',
        ]);

        return true;
    }

    private function formatIntakePreview(array $result): string
    {
        $summary = $result['summary'] ?? [];
        $rows = $summary['rows'] ?? 0;
        $errors = $summary['errors'] ?? 0;
        $warnings = $summary['warnings'] ?? 0;
        $docType = $result['doc_type'] ?? 'document';
        $intakeId = $result['intake_id'] ?? '-';

        return "Intake {$intakeId} ({$docType}) - rows: {$rows}, errors: {$errors}, warnings: {$warnings}.\nReply APPLY {$intakeId} to apply.";
    }

    private function logTool(User $user, string $tool, array $args, array $output, bool $success = true): void
    {
        AuditLog::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'channel' => 'telegram',
            'action' => $tool,
            'payload_json' => [
                'input' => $args,
                'output' => $output,
                'success' => $success,
            ],
            'created_at' => now(),
        ]);
    }

    private function parseArgs(string $text): array
    {
        $args = [];
        preg_match_all('/(\\w+)=((\"[^\"]+\")|(\\\'[^\\\']+\\\')|([^\\s]+))/', $text, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $key = $match[1];
            $value = trim($match[2], "\"'");
            $args[$key] = $value;
        }

        return $args;
    }

    private function commandSuffix(string $text, string $command): string
    {
        $suffix = trim(Str::after($text, $command));
        return ltrim($suffix);
    }

    private function isDeleteIntent(string $text): bool
    {
        return (bool) preg_match('/\\b(delete|remove|destroy|reverse)\\b/i', $text);
    }

    private function isApplyCommand(string $text): bool
    {
        return (bool) preg_match('/^apply\\s+\\d+/i', trim($text));
    }

    private function isGroupCommand(string $text): bool
    {
        return Str::startsWith(Str::lower(trim($text)), '/group');
    }

    private function isDetailsCommand(string $text): bool
    {
        $lower = Str::lower(trim($text));
        return Str::startsWith($lower, 'details') || Str::startsWith($lower, '/details');
    }

    private function isReconciliationCommand(string $text): bool
    {
        return (bool) preg_match('/^(confirm|reject)\\s+link\\s+\\d+/i', trim($text));
    }

    private function isEventAction(string $text): bool
    {
        $lower = Str::lower(trim($text));
        if (Str::startsWith($lower, ['ack', 'snooze', 'request fuel', 'start rca', 'start rcfa'])) {
            return true;
        }

        if (!Str::contains($lower, 'evt-')) {
            return false;
        }

        return Str::startsWith($lower, [
            'create wo',
            'dispatch',
            'create pr',
            'view usage',
        ]);
    }

    private function extractEventCodeFromText(string $text): ?string
    {
        if (preg_match('/\\b(EVT-\\d{4,})\\b/i', $text, $matches)) {
            return strtoupper($matches[1]);
        }

        return null;
    }

    private function extractSnoozeMinutes(string $text): ?int
    {
        if (preg_match('/\\b(\\d{1,3})\\s*(m|min|mins|minute|minutes)\\b/i', $text, $matches)) {
            return (int) $matches[1];
        }
        if (preg_match('/\\b(\\d{1,3})\\s*(h|hr|hrs|hour|hours)\\b/i', $text, $matches)) {
            return (int) $matches[1] * 60;
        }
        if (preg_match('/\\b(\\d{1,3})\\s*(d|day|days)\\b/i', $text, $matches)) {
            return (int) $matches[1] * 60 * 24;
        }

        return null;
    }

    private function resolveEvent(int $tenantId, string $eventCode): ?TelegramEvent
    {
        return TelegramEvent::where('tenant_id', $tenantId)
            ->where('event_code', $eventCode)
            ->first();
    }

    private function createPrFromEvent(TelegramEvent $event, User $user): string
    {
        $payload = $event->payload ?? [];
        $inventoryId = $payload['inventory_item_id'] ?? null;
        if (!$inventoryId) {
            return 'Event does not contain inventory details.';
        }

        $item = InventoryItem::where('tenant_id', $user->tenant_id)
            ->with('part')
            ->where('id', $inventoryId)
            ->first();
        if (!$item) {
            return 'Inventory item not found.';
        }

        $sku = $item->part?->sku ?? ($payload['sku'] ?? null);
        $description = $item->part?->name ?? 'Replenishment';
        if (!$sku) {
            return 'Missing SKU for PR line.';
        }

        $reorder = $item->reorder_point ?? $item->min_quantity ?? 1;
        $qty = max(1, (int) ceil((float) $reorder - (float) $item->quantity));

        $create = $this->executeTool('purchase_requests.create', [
            'title' => 'Stock replenishment: ' . $sku,
            'department' => 'Stores',
            'priority' => 'high',
            'notes' => 'Auto-generated from stock event ' . $event->event_code,
        ], $user);

        $message = $create['reply'] ?? 'PR created.';
        $prCode = $this->extractPurchaseRequestRef($message);
        if (!$prCode) {
            return $message;
        }

        $add = $this->executeTool('purchase_requests.add_line', [
            'pr' => $prCode,
            'sku' => $sku,
            'qty' => $qty,
            'description' => $description,
        ], $user);

        return $message . "\n" . ($add['reply'] ?? 'Line added.');
    }

    private function createFuelRequest(User $user, ?string $text = null): string
    {
        $qty = $text ? $this->extractQuantityFromText($text) : null;
        $qty = $qty ?: 1;
        $unit = 'L';

        $create = $this->executeTool('purchase_requests.create', [
            'title' => 'Fuel request',
            'department' => 'Operations',
            'priority' => 'high',
            'notes' => 'Telegram fuel request',
        ], $user);

        $message = $create['reply'] ?? 'PR created.';
        $prCode = $this->extractPurchaseRequestRef($message);
        if (!$prCode) {
            return $message;
        }

        $add = $this->executeTool('purchase_requests.add_line', [
            'pr' => $prCode,
            'description' => 'Diesel fuel',
            'qty' => $qty,
            'unit' => $unit,
        ], $user);

        return $message . "\n" . ($add['reply'] ?? 'Fuel line added.');
    }

    private function logEventAction(User $user, string $action, string $eventCode, array $extra = []): void
    {
        AuditLog::create([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'channel' => 'telegram',
            'action' => $action,
            'payload_json' => array_merge([
                'event_code' => $eventCode,
            ], $extra),
            'created_at' => now(),
        ]);
    }

    private function helpText(): string
    {
        return implode("\n", [
            'Commands:',
            '/link CODE',
            '/report pm due 7d',
            '/report compliance due 30d',
            '/report compliance overdue',
            '/group set name=FW-Quarry-Maint type=maintenance',
            '/group list',
            '/list module=assets limit=5',
            '/get module=workorders wo=WO-0210',
            '/export module=assets format=pdf',
            'Export work orders status=open as excel',
            '/maintenance status=pending',
            '/inventory q=filter',
            '/fuel asset=<asset_tag>',
            '/alerts status=open',
            '/assets search q=<name|tag>',
            '/assets create tag=<tag> name=<name> category=<category> meter_type=hours|km',
            '/assets update tag=<tag> location=\"Quarry - Pit\"',
            '/wo list asset=<asset_tag> status=open',
            '/wo create asset=<asset_tag> title=\"Hydraulic leak\" priority=high',
            '/wo assign WO-01234 user=denford due=2026-01-22',
            'ASSIGN WO-01234 @Denford',
            'ACCEPT WO-01234',
            'UPDATE WO-01234 60% done. Waiting for hose.',
            'COMMENT WO-01234 "Waiting on parts"',
            '/wo reject WO-01234 reason="Duplicate request"',
            '/wo close wo=WO-0210 notes=\"Completed repair\"',
            'CONFIRM LINK <id>',
            'REJECT LINK <id>',
            '/parts search q=\"oil filter\"',
            '/stock balance sku=FLT-100',
            '/meter create asset=<asset_tag> reading=12450 meter_type=hours',
            '/pr create dept=Crushing needed_by=2026-01-30 priority=high reason=\"Conveyor\"',
            '/pr add PR-00451 sku=IDLER-SET-650 qty=10 est=22.50',
            '/pr submit PR-00451',
            'APPROVE PR-00451 notes=\"...\" or REJECT PR-00451 reason=\"...\"',
            'CONVERT PR-00451 TO PO vendor=\"Conveyor World\"',
            'SEND PO-00218',
            'APPLY <intake_id>',
            'ACK EVT-00001',
            'SNOOZE EVT-00001 24H reason="..."',
            'CREATE WO EVT-00001',
            'CREATE PR EVT-00001',
            'VIEW USAGE EVT-00001',
            'REQUEST FUEL 500',
            'DETAILS PM|STOCK|DOWNTIME',
            'Tip: use /list module=assets to find asset tags.',
        ]);
    }

    private function userContext(User $user): array
    {
        return [
            'user' => $user,
            'tenant_id' => $user->tenant_id,
            'role_names' => $user->roles()->pluck('name')->all(),
        ];
    }
}
