<?php

namespace App\Http\Controllers;

use App\Models\ChatConversation;
use App\Models\ChatMessage;
use App\Models\ChatUpload;
use App\Models\DocumentIntakeBatch;
use App\Models\Tenant;
use App\Services\DocumentIntakeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class AiChatController extends Controller
{
    public function chat(Request $request)
    {
        $data = $request->validate([
            'message' => 'nullable|string|max:2000',
            'conversation_id' => 'nullable|integer',
            'history' => 'nullable',
            'context' => 'nullable',
            'files' => 'nullable|array',
            'files.*' => 'file|max:51200',
        ]);

        $history = $this->decodeJsonInput($request->input('history'));
        $context = $this->decodeJsonInput($request->input('context'));

        $tenant = $this->resolveTenant($request);
        if (!$tenant) {
            return response()->json([
                'reply' => 'No tenant context available.',
                'intent' => 'error',
            ], 422);
        }

        app()->instance('tenant', $tenant);
        $user = $request->user();
        $userId = $user?->id;

        $conversation = $this->resolveConversation($tenant->id, $userId, $data['conversation_id'] ?? null, $data['message'] ?? null);
        $uploads = $this->storeUploads($request, $tenant->id, $conversation->id, $userId);

        if (empty($data['message']) && !$uploads) {
            return response()->json([
                'reply' => 'Send a message or upload a file to continue.',
                'intent' => 'error',
                'conversation_id' => $conversation->id,
            ], 422);
        }

        if (!empty($data['message'])) {
            ChatMessage::create([
                'tenant_id' => $tenant->id,
                'conversation_id' => $conversation->id,
                'user_id' => $userId,
                'role' => 'user',
                'content' => $data['message'],
                'payload' => ['uploads' => collect($uploads)->pluck('id')->all()],
            ]);
            $conversation->update(['last_message_at' => now()]);
        }

        if ($uploads) {
            $intake = app(DocumentIntakeService::class)->intakeUploads($uploads, $tenant->id, $conversation->id, $userId);
            $previewRows = $intake->rows()->limit(5)->get();

            return response()->json([
                'reply' => $this->buildIntakeReply($intake),
                'intent' => 'review',
                'conversation_id' => $conversation->id,
                'intake' => [
                    'batch_id' => $intake->id,
                    'doc_type' => $intake->doc_type,
                    'summary' => $intake->summary,
                    'warnings' => $intake->warnings,
                    'preview' => $previewRows->map(fn ($row) => [
                        'row_index' => $row->row_index,
                        'data' => $row->data,
                        'errors' => $row->errors,
                        'warnings' => $row->warnings,
                        'matches' => $row->matches,
                    ]),
                ],
            ]);
        }

        $key = config('services.openai.key');
        if (!$key) {
            return response()->json([
                'reply' => 'OpenAI API key is missing. Set OPENAI_API_KEY in your .env file.',
                'intent' => 'error',
                'conversation_id' => $conversation->id,
            ], 422);
        }

        $routes = collect($context['routes'] ?? [])
            ->map(function ($route) {
                $label = $route['label'] ?? '';
                $path = $route['path'] ?? '';
                if (!$label || !$path) {
                    return null;
                }
                return "{$label}: {$path}";
            })
            ->filter()
            ->values()
            ->all();

        $routeList = $routes ? "\nAllowed routes:\n- " . implode("\n- ", $routes) : '';
        $currentPath = $context['current_path'] ?? '/ui';

        $system = "You are Fleet Copilot for a fleet management web app. " .
            "Help the user navigate, start workflows, or answer questions. " .
            "Respond ONLY in JSON with keys: reply, intent, route, action_label. " .
            "intent must be one of: navigate, action, upload, answer. " .
            "If navigation is needed, set route to an allowed path. " .
            "If a workflow is needed, set action_label to a short title like \"Create Work Order\". " .
            "Current path: {$currentPath}." . $routeList;

        $messages = [
            ['role' => 'system', 'content' => $system],
        ];

        $history = array_slice($history ?: [], -8);
        foreach ($history as $item) {
            if (!empty($item['role']) && !empty($item['content'])) {
                $messages[] = [
                    'role' => $item['role'],
                    'content' => $item['content'],
                ];
            }
        }

        $messages[] = ['role' => 'user', 'content' => $data['message']];

        $payload = [
            'model' => config('services.openai.model'),
            'messages' => $messages,
            'temperature' => 0.2,
            'max_tokens' => 500,
        ];

        $requestBuilder = Http::withToken($key)->timeout((int) config('services.openai.timeout', 20));
        $verify = config('services.openai.verify_ssl', true);
        if (is_string($verify)) {
            $normalized = strtolower(trim($verify));
            if (in_array($normalized, ['false', '0', 'off', 'no'], true)) {
                $verify = false;
            } elseif (in_array($normalized, ['true', '1', 'on', 'yes'], true)) {
                $verify = true;
            } else {
                $verify = trim($verify);
            }
        }

        if (is_string($verify) && $verify !== '') {
            $path = $verify;
            if (!is_file($path)) {
                $candidate = base_path($verify);
                if (is_file($candidate)) {
                    $path = $candidate;
                }
            }
            $verify = $path;
        }

        if ($verify !== '' && $verify !== null) {
            $requestBuilder = $requestBuilder->withOptions([
                'verify' => $verify,
            ]);
        }

        if (config('services.openai.organization')) {
            $requestBuilder = $requestBuilder->withHeaders([
                'OpenAI-Organization' => config('services.openai.organization'),
            ]);
        }
        if (config('services.openai.project')) {
            $requestBuilder = $requestBuilder->withHeaders([
                'OpenAI-Project' => config('services.openai.project'),
            ]);
        }

        $response = $requestBuilder->post(config('services.openai.endpoint'), $payload);

        if ($response->failed()) {
            return response()->json([
                'reply' => 'AI service is unavailable right now. Please try again.',
                'intent' => 'error',
                'conversation_id' => $conversation->id,
            ], 502);
        }

        $content = data_get($response->json(), 'choices.0.message.content', '');
        $decoded = json_decode($content, true);

        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            $reply = [
                'reply' => (string) ($decoded['reply'] ?? ''),
                'intent' => (string) ($decoded['intent'] ?? 'answer'),
                'route' => $decoded['route'] ?? null,
                'action_label' => $decoded['action_label'] ?? null,
                'conversation_id' => $conversation->id,
            ];

            ChatMessage::create([
                'tenant_id' => $tenant->id,
                'conversation_id' => $conversation->id,
                'user_id' => $userId,
                'role' => 'assistant',
                'content' => $reply['reply'],
                'payload' => $decoded,
            ]);
            $conversation->update(['last_message_at' => now()]);

            return response()->json($reply);
        }

        ChatMessage::create([
            'tenant_id' => $tenant->id,
            'conversation_id' => $conversation->id,
            'user_id' => $userId,
            'role' => 'assistant',
            'content' => trim((string) $content),
            'payload' => null,
        ]);
        $conversation->update(['last_message_at' => now()]);

        return response()->json([
            'reply' => trim((string) $content),
            'intent' => 'answer',
            'conversation_id' => $conversation->id,
        ]);
    }

    private function resolveTenant(Request $request): ?Tenant
    {
        if ($request->user()?->tenant_id) {
            return Tenant::find($request->user()->tenant_id);
        }

        $headerTenant = $request->header('X-Tenant-ID') ?: $request->input('tenant_id');
        if ($headerTenant) {
            return Tenant::find($headerTenant);
        }

        return Tenant::where('name', 'Fourways Group')->first() ?? Tenant::first();
    }

    private function resolveConversation(int $tenantId, ?int $userId, ?int $conversationId, ?string $message): ChatConversation
    {
        if ($conversationId) {
            $existing = ChatConversation::where('tenant_id', $tenantId)->find($conversationId);
            if ($existing) {
                return $existing;
            }
        }

        return ChatConversation::create([
            'tenant_id' => $tenantId,
            'user_id' => $userId,
            'title' => $message ? Str::limit($message, 60) : 'Chat Session',
            'status' => 'active',
            'last_message_at' => now(),
        ]);
    }

    private function storeUploads(Request $request, int $tenantId, int $conversationId, ?int $userId): array
    {
        $files = $request->file('files', []);
        if (!$files) {
            return [];
        }

        $uploads = [];
        foreach ($files as $file) {
            $original = $file->getClientOriginalName();
            $safeName = time() . '_' . Str::slug(pathinfo($original, PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs("chat_uploads/{$tenantId}/{$conversationId}", $safeName);
            $uploads[] = ChatUpload::create([
                'tenant_id' => $tenantId,
                'conversation_id' => $conversationId,
                'user_id' => $userId,
                'original_name' => $original,
                'storage_path' => $path,
                'mime_type' => $file->getClientMimeType(),
                'size_bytes' => $file->getSize(),
                'checksum' => hash_file('sha1', $file->getRealPath()),
            ]);
        }

        return $uploads;
    }

    private function decodeJsonInput($value): array
    {
        if (is_array($value)) {
            return $value;
        }

        if (is_string($value)) {
            $decoded = json_decode($value, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                return $decoded;
            }
        }

        return [];
    }

    private function buildIntakeReply(DocumentIntakeBatch $batch): string
    {
        $summary = $batch->summary ?? [];
        $rows = $summary['rows'] ?? 0;
        $errors = $summary['errors'] ?? 0;
        $warnings = $summary['warnings'] ?? 0;
        $docType = $batch->doc_type ?: 'document';

        return "I read the uploads as {$docType}. Found {$rows} rows, {$errors} errors, {$warnings} warnings. Review and apply when ready.";
    }
}
