<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\BehaviorMetric;
use App\Models\CamFrame;
use App\Models\House;
use App\Services\Ai\PoultryAiService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class CamIngestController extends Controller
{
    public function __construct(private PoultryAiService $aiService)
    {
    }

    public function storeFrame(Request $request)
    {
        $device = $request->attributes->get('device');
        $house = $this->resolveHouse($request);
        if (!$device || !$house) {
            return response()->json(['message' => 'Device or house not resolved.'], 422);
        }

        $ts = $this->parseTimestamp($request->input('ts'));
        $meta = $request->input('meta', []);
        $filePath = null;
        $fileSize = null;

        if ($request->hasFile('frame')) {
            $file = $request->file('frame');
            $ext = $file->getClientOriginalExtension() ?: 'jpg';
            $filename = Str::uuid()->toString() . '.' . $ext;
            $dir = 'cam/' . $house->id . '/' . $ts->toDateString();
            $filePath = $dir . '/' . $filename;
            Storage::disk('public')->put($filePath, file_get_contents($file->getRealPath()));
            $fileSize = $file->getSize();
        } elseif ($request->filled('image_b64')) {
            $raw = $request->input('image_b64');
            $raw = preg_replace('/^data:image\/(jpeg|jpg|png);base64,/', '', $raw);
            $contents = base64_decode($raw);
            if ($contents === false) {
                return response()->json(['message' => 'Invalid base64 image.'], 422);
            }

            $filename = Str::uuid()->toString() . '.jpg';
            $dir = 'cam/' . $house->id . '/' . $ts->toDateString();
            $filePath = $dir . '/' . $filename;
            Storage::disk('public')->put($filePath, $contents);
            $fileSize = strlen($contents);
        } else {
            return response()->json(['message' => 'No frame provided.'], 422);
        }

        $meta = array_merge(is_array($meta) ? $meta : [], [
            'size_bytes' => $fileSize,
        ]);

        $frame = CamFrame::create([
            'house_id' => $house->id,
            'device_id' => $device->id,
            'ts' => $ts->toDateTimeString(),
            'path' => $filePath,
            'meta_json' => $meta,
        ]);

        $mode = $request->input('mode');
        $skip = $request->boolean('skip_inference');
        $behavior = null;
        if (!$skip && $mode !== 'edge') {
            $behavior = $this->aiService->inferBehaviorFromFrame($house, $device->id, $ts, $meta, $filePath);
        }

        return response()->json([
            'status' => 'ok',
            'frame_id' => $frame->id,
            'path' => $filePath,
            'behavior_metric_id' => $behavior?->id,
        ]);
    }

    public function storeEvent(Request $request)
    {
        $device = $request->attributes->get('device');
        $house = $this->resolveHouse($request);
        if (!$device || !$house) {
            return response()->json(['message' => 'Device or house not resolved.'], 422);
        }

        $data = $request->validate([
            'ts' => ['nullable', 'date'],
            'activity_score' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'clustering_score' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'abnormal_score' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'note' => ['nullable', 'string'],
        ]);

        $ts = $this->parseTimestamp($data['ts'] ?? null);
        $metric = BehaviorMetric::updateOrCreate([
            'house_id' => $house->id,
            'device_id' => $device->id,
            'ts' => $ts->toDateTimeString(),
        ], [
            'activity_score' => $data['activity_score'] ?? null,
            'clustering_score' => $data['clustering_score'] ?? null,
            'abnormal_score' => $data['abnormal_score'] ?? null,
            'meta_json' => array_filter([
                'note' => $data['note'] ?? null,
                'mode' => 'edge',
            ], fn ($value) => $value !== null && $value !== ''),
        ]);

        return response()->json(['status' => 'ok', 'id' => $metric->id]);
    }

    private function resolveHouse(Request $request): ?House
    {
        $device = $request->attributes->get('device');
        $house = $request->attributes->get('house');
        if ($house) {
            return $house;
        }

        $houseId = $request->input('house_id');
        if ($houseId) {
            $house = House::find($houseId);
            if ($house && $device && !$device->house_id) {
                $device->house_id = $house->id;
                $device->save();
            }
            return $house;
        }

        return null;
    }

    private function parseTimestamp(?string $value): Carbon
    {
        return $value ? Carbon::parse($value, 'UTC')->setTimezone('UTC') : Carbon::now('UTC');
    }
}
