<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ExpectedGrowthCurve;
use App\Models\House;
use App\Models\Sensor;
use App\Models\SensorReading;
use App\Models\WeightSample;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;

class IotIngestController extends Controller
{
    public function ingestSensor(Request $request)
    {
        $device = $request->attributes->get('device');
        $house = $this->resolveHouse($request);
        if (!$device || !$house) {
            return response()->json(['message' => 'Device or house not resolved.'], 422);
        }

        $payload = $request->all();
        $readings = $payload['readings'] ?? null;
        if (!$readings && isset($payload['type'], $payload['value'])) {
            $readings = [[
                'type' => $payload['type'],
                'value' => $payload['value'],
                'unit' => $payload['unit'] ?? null,
                'name' => $payload['name'] ?? null,
                'ts' => $payload['ts'] ?? null,
                'meta' => $payload['meta'] ?? null,
            ]];
        }

        if (!is_array($readings) || empty($readings)) {
            return response()->json(['message' => 'No sensor readings provided.'], 422);
        }

        $created = 0;
        foreach ($readings as $reading) {
            $type = Arr::get($reading, 'type');
            $value = Arr::get($reading, 'value');
            if (!is_string($type) || $type === '' || !is_numeric($value)) {
                continue;
            }

            $value = (float) $value;
            if (!$this->valueInRange($type, $value)) {
                return response()->json(['message' => "Value out of range for {$type}."], 422);
            }

            $ts = $this->parseTimestamp(Arr::get($reading, 'ts') ?? $payload['ts'] ?? null);
            $sensor = Sensor::firstOrCreate([
                'device_id' => $device->id,
                'type' => $type,
            ], [
                'unit' => Arr::get($reading, 'unit'),
                'name' => Arr::get($reading, 'name'),
            ]);

            if (!$sensor->unit && Arr::get($reading, 'unit')) {
                $sensor->unit = Arr::get($reading, 'unit');
                $sensor->save();
            }

            SensorReading::updateOrCreate([
                'sensor_id' => $sensor->id,
                'ts' => $ts->toDateTimeString(),
            ], [
                'value' => $value,
                'meta_json' => Arr::get($reading, 'meta'),
            ]);

            $created++;
        }

        return response()->json(['status' => 'ok', 'created' => $created]);
    }

    public function ingestWeight(Request $request)
    {
        $device = $request->attributes->get('device');
        $house = $this->resolveHouse($request);
        if (!$device || !$house) {
            return response()->json(['message' => 'Device or house not resolved.'], 422);
        }

        $data = $request->validate([
            'ts' => ['nullable', 'date'],
            'avg_weight_g' => ['required', 'numeric', 'min:10', 'max:6000'],
            'sample_count' => ['nullable', 'integer', 'min:1', 'max:2000'],
            'raw' => ['nullable'],
            'note' => ['nullable', 'string'],
        ]);

        $avgWeight = (float) $data['avg_weight_g'];
        if ($this->isWeightOutlier($house, $avgWeight)) {
            return response()->json(['message' => 'Weight sample outside expected range.'], 422);
        }

        $ts = $this->parseTimestamp($data['ts'] ?? null);
        $sample = WeightSample::updateOrCreate([
            'house_id' => $house->id,
            'source_device_id' => $device->id,
            'ts' => $ts->toDateTimeString(),
        ], [
            'avg_weight_g' => $avgWeight,
            'sample_count' => $data['sample_count'] ?? 0,
            'meta_json' => array_filter([
                'raw' => $data['raw'] ?? null,
                'note' => $data['note'] ?? null,
            ], fn ($value) => $value !== null && $value !== ''),
        ]);

        return response()->json(['status' => 'ok', 'id' => $sample->id]);
    }

    private function resolveHouse(Request $request): ?House
    {
        $device = $request->attributes->get('device');
        $house = $request->attributes->get('house');
        if ($house) {
            return $house;
        }

        $houseId = $request->input('house_id');
        if ($houseId) {
            $house = House::find($houseId);
            if ($house && $device && !$device->house_id) {
                $device->house_id = $house->id;
                $device->save();
            }
            return $house;
        }

        return null;
    }

    private function parseTimestamp(?string $value): Carbon
    {
        return $value ? Carbon::parse($value, 'UTC')->setTimezone('UTC') : Carbon::now('UTC');
    }

    private function valueInRange(string $type, float $value): bool
    {
        $ranges = config('poultry.sensor_ranges', []);
        if (!isset($ranges[$type])) {
            return true;
        }

        $min = $ranges[$type]['min'];
        $max = $ranges[$type]['max'];

        return $value >= $min && $value <= $max;
    }

    private function isWeightOutlier(House $house, float $avgWeight): bool
    {
        $expected = ExpectedGrowthCurve::where('age_day', $house->age_days)->value('expected_weight_g');
        if (!$expected) {
            return false;
        }

        $min = $expected * 0.5;
        $max = $expected * 1.6;
        return $avgWeight < $min || $avgWeight > $max;
    }
}
