<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Services\AuditService;
use Illuminate\Http\Request;

class AssetController extends Controller
{
    public function index()
    {
        $tenantId = auth()->user()->tenant_id;
        $assets = Asset::where('tenant_id', $tenantId)->get();
        return response()->json($assets);
    }

    public function show($id)
    {
        $tenantId = auth()->user()->tenant_id;
        $asset = Asset::where('tenant_id', $tenantId)->findOrFail($id);
        return response()->json($asset);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'asset_tag' => 'nullable|string|max:100',
            'make' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'serial_number' => 'nullable|string|max:255',
            'year' => 'nullable|string|max:10',
            'category' => 'nullable|string|max:100',
            'asset_type' => 'nullable|string|max:100',
            'lifecycle_status' => 'nullable|string|max:50',
            'commissioned_at' => 'nullable|date',
            'description' => 'nullable|string',
            'status' => 'required|string|in:active,inactive',
            'site_id' => 'nullable|exists:sites,id',
            'maintenance_interval' => 'nullable|integer|min:1',
            'meter_type' => 'nullable|string|max:50',
            'meter_reading' => 'nullable|numeric',
        ]);

        $data = $request->all();
        $data['tenant_id'] = $request->user()->tenant_id;

        $asset = Asset::create($data);
        app(AuditService::class)->log(
            'asset.created',
            Asset::class,
            $asset->id,
            $asset->toArray(),
            $request->user()->tenant_id,
            $request->user()->id,
            $request
        );
        return response()->json($asset, 201);
    }

    public function update(Request $request, $id)
    {
        $tenantId = $request->user()->tenant_id;
        $asset = Asset::where('tenant_id', $tenantId)->findOrFail($id);

        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'asset_tag' => 'nullable|string|max:100',
            'make' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'serial_number' => 'nullable|string|max:255',
            'year' => 'nullable|string|max:10',
            'category' => 'nullable|string|max:100',
            'asset_type' => 'nullable|string|max:100',
            'lifecycle_status' => 'nullable|string|max:50',
            'commissioned_at' => 'nullable|date',
            'description' => 'nullable|string',
            'status' => 'sometimes|required|string|in:active,inactive',
            'site_id' => 'nullable|exists:sites,id',
            'maintenance_interval' => 'nullable|integer|min:1',
            'meter_type' => 'nullable|string|max:50',
            'meter_reading' => 'nullable|numeric',
        ]);

        $asset->update($request->all());
        app(AuditService::class)->log(
            'asset.updated',
            Asset::class,
            $asset->id,
            $asset->getChanges(),
            $request->user()->tenant_id,
            $request->user()->id,
            $request
        );
        return response()->json($asset);
    }

    public function destroy(Request $request, $id)
    {
        $tenantId = $request->user()->tenant_id;
        $asset = Asset::where('tenant_id', $tenantId)->findOrFail($id);
        $asset->delete();
        app(AuditService::class)->log(
            'asset.deleted',
            Asset::class,
            $asset->id,
            [],
            $tenantId,
            $request->user()->id,
            $request
        );
        return response()->json(null, 204);
    }
}
