<?php

namespace App\Http\Controllers;

use App\Models\Device;
use App\Services\SensorIngestionService;
use App\Services\AuditService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class DeviceController extends Controller
{
    public function index()
    {
        return Device::with('asset')->latest()->get();
    }

    public function store(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $data = $request->validate([
            'asset_id' => [
                'nullable',
                Rule::exists('assets', 'id')->where('tenant_id', $tenantId),
            ],
            'name' => 'nullable|string|max:255',
            'type' => 'nullable|string|max:100',
            'identifier' => 'nullable|string|max:255',
            'meta' => 'nullable|array',
        ]);

        $device = Device::create($data);

        app(AuditService::class)->log(
            'device.created',
            Device::class,
            $device->id,
            $device->toArray(),
            $request->user()->tenant_id,
            $request->user()->id,
            $request
        );

        return response()->json($device, 201);
    }

    public function ingest(Request $request, Device $device, SensorIngestionService $service)
    {
        $data = $request->validate([
            'payload' => 'required|array',
            'timestamp' => 'nullable|date',
        ]);

        $payload = $data['payload'];
        if (!isset($payload['timestamp']) && isset($data['timestamp'])) {
            $payload['timestamp'] = $data['timestamp'];
        }

        $event = $service->ingest($payload, $device->tenant_id, $device->id);

        return response()->json($event, 201);
    }

    public function ingestByIdentifier(Request $request, SensorIngestionService $service)
    {
        $data = $request->validate([
            'identifier' => 'required|string',
            'payload' => 'required|array',
            'timestamp' => 'nullable|date',
        ]);

        $payload = $data['payload'];
        if (!isset($payload['timestamp']) && isset($data['timestamp'])) {
            $payload['timestamp'] = $data['timestamp'];
        }

        $event = $service->ingest($payload, auth()->user()->tenant_id, null, $data['identifier']);

        return response()->json($event, 201);
    }
}
