<?php

namespace App\Http\Controllers;

use App\Models\Tenant;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class DownloadTemplateController extends Controller
{
    public function dashboardSnapshot(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'Dashboard Snapshot',
            'Fleet Command overview export',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Scope' => 'Fleet Command Center',
            ],
            [
                ['Metric', 'Value', 'Notes'],
            ],
            [
                ['Availability', '94.8%', 'Rolling 7-day average'],
                ['Utilization', '86.2%', 'Quarry + concrete fleet'],
                ['Alerts', '6 open', 'Critical: 2'],
                ['Work Orders', '12 open', 'Due this week: 4'],
            ],
            Str::slug('dashboard-snapshot') . '.pdf'
        );
    }

    public function auditLog(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'Audit Log Export',
            'Tenant-wide change history',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Window' => 'Last 24 hours',
            ],
            [
                ['Actor', 'Action', 'Entity', 'Changes', 'IP', 'Time'],
            ],
            [
                ['Maria Lee', 'Updated', 'Work Order', 'Status: Open -> Closed', '10.4.2.18', '09:12'],
                ['Ops Bot', 'Created', 'Alarm', 'Coolant temp > 95', '10.4.2.12', '08:54'],
                ['Grace K.', 'Uploaded', 'Import Batch', 'fleet_register.csv', '10.4.2.20', '08:30'],
            ],
            Str::slug('audit-log') . '.pdf'
        );
    }

    public function auditEntry(Request $request, string $audit)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            "Audit Entry {$audit}",
            'Change detail export',
            $tenant,
            [
                'Actor' => 'Maria Lee',
                'Action' => 'work_order.updated',
                'Entity' => 'WO-221',
            ],
            [
                ['Field', 'Before', 'After'],
            ],
            [
                ['Status', 'Open', 'Closed'],
                ['Completed At', '-', '09:12'],
                ['Notes', '-', 'Resolved leak'],
            ],
            Str::slug("audit-entry-{$audit}") . '.pdf'
        );
    }

    public function kpis(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'KPI Studio Export',
            'Performance snapshot',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Period' => 'Last 30 days',
            ],
            [
                ['KPI', 'Value', 'Target', 'Trend'],
            ],
            [
                ['Utilization Rate', '86%', '85%', 'Up'],
                ['Cost per Ton', '$14.10', '$14.00', 'Flat'],
                ['Safety Incidents', '0', '0', 'Stable'],
                ['MTBF', '142 hrs', '140 hrs', 'Up'],
            ],
            Str::slug('kpi-studio-export') . '.pdf'
        );
    }

    public function maintenance(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'Maintenance Export',
            'Work orders and PM summary',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Open WOs' => '12',
            ],
            [
                ['Asset', 'Work Order', 'Type', 'Status', 'Due'],
            ],
            [
                ['Excavator EX-12', 'WO-221', 'Corrective', 'Open', 'Aug 23'],
                ['Hauler H-44', 'WO-228', 'Preventive', 'Scheduled', 'Aug 24'],
                ['Loader L-7', 'WO-230', 'Inspection', 'In progress', 'Aug 25'],
            ],
            Str::slug('maintenance-export') . '.pdf'
        );
    }

    public function sites(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'Sites Export',
            'Location register',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Locations' => '4 active sites',
            ],
            [
                ['Site', 'Type', 'Assets', 'Status'],
            ],
            [
                ['Fourways Quarry', 'Quarry', '146', 'Active'],
                ['Fourways Concrete', 'Plant', '68', 'Active'],
                ['Workshop', 'Workshop', '23', 'Active'],
                ['Yard', 'Yard', '14', 'Standby'],
            ],
            Str::slug('sites-export') . '.pdf'
        );
    }

    public function telemetry(Request $request, string $telemetry)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            "Telemetry Export {$telemetry}",
            'Sensor trend snapshot',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Metric' => 'Engine Temp',
            ],
            [
                ['Timestamp', 'Value', 'Status'],
            ],
            [
                ['08:00', '84 C', 'Normal'],
                ['10:00', '88 C', 'Watch'],
                ['12:00', '92 C', 'Alert'],
            ],
            Str::slug("telemetry-{$telemetry}") . '.pdf'
        );
    }

    public function payment(Request $request, string $payment)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            "Payment {$payment}",
            'Invoice template',
            $tenant,
            [
                'Vendor' => 'Summit Industrial',
                'Status' => 'Scheduled',
                'Due' => 'Aug 30',
            ],
            [
                ['Item', 'Qty', 'Unit', 'Price', 'Total', 'Status'],
            ],
            [
                ['Hydraulic Pump Kit', '2', 'pcs', '$14,200', '$28,400', 'Received'],
                ['Seal Set', '4', 'pcs', '$1,200', '$4,800', 'Pending'],
                ['Hydraulic Oil', '12', 'L', '$95', '$1,140', 'Pending'],
            ],
            Str::slug("payment-{$payment}") . '.pdf'
        );
    }

    public function purchaseOrder(Request $request, string $order)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            "Purchase Order {$order}",
            'Purchase order template',
            $tenant,
            [
                'Vendor' => 'Summit Industrial',
                'Site' => 'Quarry',
                'Status' => 'In transit',
            ],
            [
                ['Item', 'Qty', 'Unit', 'Price', 'Total', 'Status'],
            ],
            [
                ['Hydraulic Pump Kit', '2', 'pcs', '$14,200', '$28,400', 'On route'],
                ['Seal Set', '4', 'pcs', '$1,200', '$4,800', 'Pending'],
                ['Hydraulic Oil', '12', 'L', '$95', '$1,140', 'Pending'],
            ],
            Str::slug("purchase-order-{$order}") . '.pdf'
        );
    }

    public function report(Request $request, string $report)
    {
        $tenant = $this->resolveTenant($request);
        $reportName = urldecode($report);

        return $this->renderPdf(
            "Report {$reportName}",
            'Auto-generated operations report',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Period' => 'Last 7 days',
            ],
            [
                ['Highlight', 'Detail'],
            ],
            [
                ['Availability', '94.8% average'],
                ['Downtime', '6.2h total, waiting parts'],
                ['Cost driver', 'Hydraulic components'],
                ['Next actions', 'Reorder filters, close WO-221'],
            ],
            Str::slug("report-{$reportName}") . '.pdf'
        );
    }

    public function reportPack(Request $request)
    {
        $tenant = $this->resolveTenant($request);

        return $this->renderPdf(
            'Operational Report Pack',
            'Weekly summary bundle',
            $tenant,
            [
                'Generated' => now()->format('M d, Y H:i'),
                'Contents' => 'Weekly Operations, Maintenance Digest, Compliance',
            ],
            [
                ['Section', 'Summary'],
            ],
            [
                ['Weekly Operations', 'Availability 94.8%, utilization 86.2%'],
                ['Maintenance Digest', '12 open work orders, 4 due'],
                ['Compliance Checklist', '98% audit readiness'],
            ],
            Str::slug('report-pack') . '.pdf'
        );
    }

    public function reportPpt(Request $request, string $report)
    {
        $tenant = $this->resolveTenant($request);
        $reportName = urldecode($report);
        $logo = $this->logoData();

        $html = view('exports.report-presentation', [
            'title' => "Report {$reportName}",
            'subtitle' => 'Presentation template',
            'tenant' => $tenant?->name ?? 'Tenant',
            'generated' => now()->format('M d, Y H:i'),
            'logo' => $logo,
        ])->render();

        $filename = Str::slug("report-{$reportName}") . '.ppt';

        return response($html, 200, [
            'Content-Type' => 'application/vnd.ms-powerpoint',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ]);
    }

    public function importTemplate()
    {
        $headers = [
            'asset_tag',
            'name',
            'make',
            'model',
            'category',
            'location',
            'serial_number',
            'purchase_date',
            'current_value',
        ];

        $rows = [
            ['A-100', 'Hauler 10', 'Caterpillar', '797F', 'Vehicle', 'Fourways Quarry', 'CAT797F-0001', '2022-04-10', '420000'],
            ['A-101', 'Crusher 6', 'Metso', 'HP400', 'Fixed Plant', 'Fourways Quarry', 'MET-HP400-22', '2020-11-02', '210000'],
        ];

        $filename = 'asset-import-template.csv';

        return response()->streamDownload(function () use ($headers, $rows) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, $headers);
            foreach ($rows as $row) {
                fputcsv($handle, $row);
            }
            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
        ]);
    }

    private function renderPdf(
        string $title,
        string $subtitle,
        ?Tenant $tenant,
        array $meta,
        array $headers,
        array $rows,
        string $filename
    ) {
        $logo = $this->logoData();

        $pdf = Pdf::loadView('exports.template', [
            'title' => $title,
            'subtitle' => $subtitle,
            'tenant' => $tenant?->name ?? 'Tenant',
            'meta' => $meta,
            'headers' => $headers,
            'rows' => $rows,
            'logo' => $logo,
        ])->setPaper('a4', 'portrait');

        return $pdf->download($filename);
    }

    private function logoData(): ?string
    {
        $logoPath = public_path('ui-assets/images/fourways-logo.png');
        if (!is_file($logoPath)) {
            return null;
        }

        return 'data:image/png;base64,' . base64_encode(file_get_contents($logoPath));
    }

    private function resolveTenant(Request $request): ?Tenant
    {
        if ($request->user()?->tenant_id) {
            return Tenant::find($request->user()->tenant_id);
        }

        $headerTenant = $request->header('X-Tenant-ID') ?: $request->input('tenant_id');
        if ($headerTenant) {
            return Tenant::find($headerTenant);
        }

        return Tenant::where('name', 'Fourways Group')->first() ?? Tenant::first();
    }
}
