<?php

namespace App\Http\Controllers;

use App\Models\PurchaseRequestItem;
use App\Models\PurchaseRequestItemDocument;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PurchaseRequestItemController extends Controller
{
    public function update(Request $request, PurchaseRequestItem $item)
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $item->tenant_id !== $tenant->id) {
            return response()->json(['message' => 'Not found.'], 404);
        }

        $data = $request->validate([
            'item_name' => 'nullable|string|max:255',
            'quantity' => 'nullable|numeric',
            'unit' => 'nullable|string|max:50',
            'purpose_cost_center' => 'nullable|string|max:255',
            'supplier_name' => 'nullable|string|max:255',
            'selected_supplier' => 'nullable|string|max:255',
            'quote_amount_usd' => 'nullable|numeric',
            'quote_amount_zwl' => 'nullable|numeric',
            'comments' => 'nullable|string|max:1000',
        ]);

        $item->fill($data);
        $item->save();

        return response()->json([
            'id' => $item->id,
            'item_name' => $item->item_name,
            'quantity' => $item->quantity,
            'unit' => $item->unit,
            'purpose_cost_center' => $item->purpose_cost_center,
            'supplier_name' => $item->supplier_name,
            'selected_supplier' => $item->selected_supplier,
            'quote_amount_usd' => $item->quote_amount_usd,
            'quote_amount_zwl' => $item->quote_amount_zwl,
            'comments' => $item->comments,
        ]);
    }

    public function uploadDocument(Request $request, PurchaseRequestItem $item)
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $item->tenant_id !== $tenant->id) {
            return response()->json(['message' => 'Not found.'], 404);
        }

        $data = $request->validate([
            'file' => 'required|file|max:51200',
        ]);

        $file = $data['file'];
        $original = $file->getClientOriginalName();
        $safeName = time() . '_' . Str::slug(pathinfo($original, PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
        $path = $file->storeAs("purchase-request-items/{$item->id}", $safeName);

        $document = PurchaseRequestItemDocument::create([
            'tenant_id' => $tenant->id,
            'purchase_request_item_id' => $item->id,
            'uploaded_by' => $request->user()?->id,
            'original_name' => $original,
            'storage_path' => $path,
            'mime_type' => $file->getClientMimeType(),
            'size_bytes' => $file->getSize(),
        ]);

        return response()->json([
            'id' => $document->id,
            'name' => $document->original_name,
            'download_url' => route('purchase-request-item-documents.download', [$item->id, $document->id]),
        ]);
    }

    public function downloadDocument(Request $request, PurchaseRequestItem $item, PurchaseRequestItemDocument $document)
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $item->tenant_id !== $tenant->id || $document->tenant_id !== $tenant->id) {
            abort(404);
        }

        if ($document->purchase_request_item_id !== $item->id) {
            abort(404);
        }

        return Storage::download($document->storage_path, $document->original_name);
    }

    private function resolveTenant(Request $request): ?Tenant
    {
        if ($request->user()?->tenant_id) {
            return Tenant::find($request->user()->tenant_id);
        }

        $headerTenant = $request->header('X-Tenant-ID') ?: $request->input('tenant_id');
        if ($headerTenant) {
            return Tenant::find($headerTenant);
        }

        return Tenant::where('name', 'Fourways Group')->first() ?? Tenant::first();
    }
}
