<?php

namespace App\Http\Controllers;

use App\Models\DocumentIntakeRow;
use App\Models\GoodsReceipt;
use App\Models\PmSchedule;
use App\Models\PurchaseOrder;
use App\Models\PurchaseRequest;
use App\Models\ReconciliationLink;
use App\Models\Tenant;
use App\Models\WorkOrder;
use App\Services\LinkingReconciliationService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ReconciliationController extends Controller
{
    public function index(Request $request): View
    {
        $tenant = $this->resolveTenant($request);
        $links = $tenant
            ? ReconciliationLink::where('tenant_id', $tenant->id)->orderByDesc('created_at')->limit(200)->get()
            : collect();

        $idsByType = $this->collectIdsByType($links);
        $lookups = $this->buildLookups($tenant?->id, $idsByType);

        $rows = $links->map(function (ReconciliationLink $link) use ($lookups) {
            $sourceLabel = $this->labelFor($link->source_type, $link->source_id, $lookups, $link);
            $targetLabel = $this->labelFor($link->target_type, $link->target_id, $lookups, $link);
            $sourcePath = $this->pathFor($link->source_type, $link->source_id, $lookups);
            $targetPath = $this->pathFor($link->target_type, $link->target_id, $lookups);
            $typeLabel = $this->typeLabel($link->source_type, $link->target_type);
            $typeValue = strtolower(($link->source_type ?? 'unknown') . '->' . ($link->target_type ?? 'unknown'));

            return [
                'link' => $link,
                'source_label' => $sourceLabel,
                'target_label' => $targetLabel,
                'source_path' => $sourcePath,
                'target_path' => $targetPath,
                'type_label' => $typeLabel,
                'type_value' => $typeValue,
            ];
        });

        $statusOptions = $links->pluck('status')->filter()->unique()->values();
        $typeOptions = $rows->map(fn ($row) => ['value' => $row['type_value'], 'label' => $row['type_label']])
            ->unique('value')
            ->sortBy('label')
            ->values();

        return view('ui.reconciliation', [
            'tenant' => $tenant,
            'rows' => $rows,
            'statusOptions' => $statusOptions,
            'typeOptions' => $typeOptions,
        ]);
    }

    public function run(Request $request, LinkingReconciliationService $service): RedirectResponse
    {
        $tenant = $this->resolveTenant($request);
        if ($tenant) {
            $service->runForTenant($tenant->id);
        }

        return redirect()->back();
    }

    public function apply(Request $request, ReconciliationLink $link, LinkingReconciliationService $service): RedirectResponse
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $link->tenant_id !== $tenant->id) {
            abort(404);
        }

        $service->applyLink($link, $request->user()?->id);

        return redirect()->back();
    }

    public function ignore(Request $request, ReconciliationLink $link): RedirectResponse
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $link->tenant_id !== $tenant->id) {
            abort(404);
        }

        if ($link->status !== 'applied') {
            $link->status = 'ignored';
            $link->save();
        }

        return redirect()->back();
    }

    private function resolveTenant(Request $request): ?Tenant
    {
        if ($request->user()?->tenant_id) {
            return Tenant::find($request->user()->tenant_id);
        }

        $headerTenant = $request->header('X-Tenant-ID') ?: $request->input('tenant_id');
        if ($headerTenant) {
            return Tenant::find($headerTenant);
        }

        return Tenant::where('name', 'Fourways Group')->first() ?? Tenant::first();
    }

    private function collectIdsByType($links): array
    {
        $ids = [];
        foreach ($links as $link) {
            if ($link->source_type && $link->source_id) {
                $ids[$link->source_type][] = $link->source_id;
            }
            if ($link->target_type && $link->target_id) {
                $ids[$link->target_type][] = $link->target_id;
            }
        }

        foreach ($ids as $type => $values) {
            $ids[$type] = array_values(array_unique($values));
        }

        return $ids;
    }

    private function buildLookups(?int $tenantId, array $idsByType): array
    {
        $empty = [
            'purchase_order' => collect(),
            'purchase_request' => collect(),
            'goods_receipt' => collect(),
            'work_order' => collect(),
            'pm_schedule' => collect(),
            'invoice_row' => collect(),
        ];

        if (!$tenantId) {
            return $empty;
        }

        return [
            'purchase_order' => empty($idsByType['purchase_order']) ? $empty['purchase_order']
                : PurchaseOrder::where('tenant_id', $tenantId)->whereIn('id', $idsByType['purchase_order'])->get()->keyBy('id'),
            'purchase_request' => empty($idsByType['purchase_request']) ? $empty['purchase_request']
                : PurchaseRequest::where('tenant_id', $tenantId)->whereIn('id', $idsByType['purchase_request'])->get()->keyBy('id'),
            'goods_receipt' => empty($idsByType['goods_receipt']) ? $empty['goods_receipt']
                : GoodsReceipt::where('tenant_id', $tenantId)->whereIn('id', $idsByType['goods_receipt'])->get()->keyBy('id'),
            'work_order' => empty($idsByType['work_order']) ? $empty['work_order']
                : WorkOrder::where('tenant_id', $tenantId)->whereIn('id', $idsByType['work_order'])->get()->keyBy('id'),
            'pm_schedule' => empty($idsByType['pm_schedule']) ? $empty['pm_schedule']
                : PmSchedule::where('tenant_id', $tenantId)->whereIn('id', $idsByType['pm_schedule'])->get()->keyBy('id'),
            'invoice_row' => empty($idsByType['invoice_row']) ? $empty['invoice_row']
                : DocumentIntakeRow::where('tenant_id', $tenantId)->whereIn('id', $idsByType['invoice_row'])->get()->keyBy('id'),
        ];
    }

    private function labelFor(?string $type, ?int $id, array $lookups, ?ReconciliationLink $link = null): string
    {
        if (!$type || !$id) {
            return '-';
        }

        switch ($type) {
            case 'purchase_order':
                $order = $lookups['purchase_order']->get($id);
                return $order?->po_number ?: "PO-{$id}";
            case 'purchase_request':
                $request = $lookups['purchase_request']->get($id);
                return $request?->request_code ?: "PR-{$id}";
            case 'goods_receipt':
                $receipt = $lookups['goods_receipt']->get($id);
                return $receipt?->reference ?: "GRN-{$id}";
            case 'work_order':
                $workOrder = $lookups['work_order']->get($id);
                return $workOrder?->reference_code ?: "WO-{$id}";
            case 'pm_schedule':
                $schedule = $lookups['pm_schedule']->get($id);
                return $schedule?->name ? $schedule->name : "PM Schedule #{$id}";
            case 'invoice_row':
                $row = $lookups['invoice_row']->get($id);
                $invoiceNumber = $row?->data['invoice_number'] ?? ($link?->meta_json['invoice_number'] ?? null);
                $vendor = $row?->data['vendor'] ?? ($link?->meta_json['vendor'] ?? null);
                $label = $invoiceNumber ? "Invoice {$invoiceNumber}" : "Invoice Row #{$id}";
                return $vendor ? "{$label} - {$vendor}" : $label;
            default:
                return ucwords(str_replace('_', ' ', $type)) . " #{$id}";
        }
    }

    private function pathFor(?string $type, ?int $id, array $lookups): ?string
    {
        if (!$type || !$id) {
            return null;
        }

        switch ($type) {
            case 'purchase_order':
                return "/ui/purchase-orders/{$id}";
            case 'purchase_request':
                return "/ui/purchase-requests/{$id}";
            case 'work_order':
                return "/ui/work-orders/{$id}";
            case 'pm_schedule':
                return "/ui/pm-schedules/{$id}";
            case 'invoice_row':
                $row = $lookups['invoice_row']->get($id);
                return $row?->batch_id ? "/ui/imports/{$row->batch_id}" : null;
            default:
                return null;
        }
    }

    private function typeLabel(?string $sourceType, ?string $targetType): string
    {
        $source = $sourceType ? ucwords(str_replace('_', ' ', $sourceType)) : 'Unknown';
        $target = $targetType ? ucwords(str_replace('_', ' ', $targetType)) : 'Unlinked';
        return "{$source} -> {$target}";
    }
}
