<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem;
use App\Models\Tenant;
use App\Models\Vendor;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\PurchaseRequest;

class SupplyChainExportController extends Controller
{
    public function export(Request $request)
    {
        $data = $request->validate([
            'type' => 'required|string',
            'ids' => 'nullable|string',
        ]);

        $tenant = $this->resolveTenant($request);
        if (!$tenant) {
            return response()->json(['message' => 'Tenant not found.'], 404);
        }

        $ids = collect(explode(',', (string) ($data['ids'] ?? '')))
            ->map(fn ($value) => trim($value))
            ->filter()
            ->values();

        if ($ids->isEmpty()) {
            return response()->json(['message' => 'Select rows to export.'], 422);
        }

        $type = strtolower($data['type']);
        $export = $this->buildExportData($tenant->id, $type, $ids);

        if (!$export) {
            return response()->json(['message' => 'Unsupported export type.'], 422);
        }

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle($export['title']);

        $this->addBranding($sheet);

        $row = 4;
        $sheet->setCellValue("A{$row}", $export['title']);
        $sheet->mergeCells("A{$row}:{$export['end_col']}{$row}");
        $sheet->getStyle("A{$row}")->getFont()->setBold(true)->setSize(14);
        $row += 2;

        $sheet->fromArray($export['headers'], null, "A{$row}");
        $sheet->getStyle("A{$row}:{$export['end_col']}{$row}")->getFont()->setBold(true);
        $sheet->getStyle("A{$row}:{$export['end_col']}{$row}")->getFill()->setFillType('solid')->getStartColor()->setARGB('FF0B1220');
        $sheet->getStyle("A{$row}:{$export['end_col']}{$row}")->getFont()->getColor()->setARGB('FFE9EEF6');
        $row++;

        foreach ($export['rows'] as $dataRow) {
            $sheet->fromArray($dataRow, null, "A{$row}");
            $row++;
        }

        foreach (range('A', $export['end_col']) as $column) {
            $sheet->getColumnDimension($column)->setAutoSize(true);
        }

        $filename = Str::slug($export['title']) . '_' . now()->format('Ymd_His') . '.xlsx';
        $writer = new Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    public function exportRequest(Request $request, PurchaseRequest $purchaseRequest, string $format)
    {
        $tenant = $this->resolveTenant($request);
        if (!$tenant || $purchaseRequest->tenant_id !== $tenant->id) {
            abort(404);
        }

        $format = strtolower($format);
        if (!in_array($format, ['pdf', 'excel'], true)) {
            abort(404);
        }

        $purchaseRequest->load(['site', 'items']);

        if ($format === 'excel') {
            return $this->exportPurchaseRequestExcel($purchaseRequest);
        }

        return $this->exportPurchaseRequestPdf($purchaseRequest);
    }

    private function buildExportData(int $tenantId, string $type, $ids): ?array
    {
        if ($type === 'purchase-requests') {
            $requests = PurchaseRequest::with(['site', 'items'])
                ->where('tenant_id', $tenantId)
                ->whereIn('id', $ids)
                ->get();

            $headers = ['Request', 'Site', 'Category', 'Total USD', 'Total ZWL', 'Priority', 'Status'];
            $rows = $requests->map(function ($request) {
                $usd = $request->items->sum('quote_amount_usd');
                $zwl = $request->items->sum('quote_amount_zwl');
                return [
                    $request->request_code,
                    $request->site?->name ?? 'Unassigned',
                    $request->title ?? 'Procurement',
                    $usd ? number_format($usd, 2) : '-',
                    $zwl ? number_format($zwl, 2) : '-',
                    ucfirst($request->priority ?? 'medium'),
                    ucfirst($request->status ?? 'pending'),
                ];
            })->all();

            return ['title' => 'Purchase Requests', 'headers' => $headers, 'rows' => $rows, 'end_col' => 'G'];
        }

        if ($type === 'inventory') {
            $items = InventoryItem::with(['part', 'location'])
                ->where('tenant_id', $tenantId)
                ->whereIn('id', $ids)
                ->get();

            $headers = ['SKU', 'Part', 'Location', 'On Hand', 'Min', 'Max', 'Status'];
            $rows = $items->map(function ($item) {
                $needsReorder = $item->min_quantity !== null && $item->quantity <= $item->min_quantity;
                return [
                    $item->part?->sku ?? '-',
                    $item->part?->name ?? 'Part',
                    $item->location?->name ?? 'Main Store',
                    $item->quantity ?? 0,
                    $item->min_quantity ?? '-',
                    $item->max_quantity ?? '-',
                    $needsReorder ? 'Reorder' : 'Healthy',
                ];
            })->all();

            return ['title' => 'Inventory Export', 'headers' => $headers, 'rows' => $rows, 'end_col' => 'G'];
        }

        if ($type === 'vendors') {
            $vendors = Vendor::where('tenant_id', $tenantId)
                ->whereIn('id', $ids)
                ->get();

            if ($vendors->isEmpty()) {
                $vendors = Vendor::where('tenant_id', $tenantId)
                    ->whereIn('name', $ids)
                    ->get();
            }

            $headers = ['Vendor', 'Contact', 'Phone', 'Email', 'Lead Time', 'Notes'];
            $rows = $vendors->map(function ($vendor) {
                return [
                    $vendor->name,
                    $vendor->contact_name ?? '-',
                    $vendor->phone ?? '-',
                    $vendor->email ?? '-',
                    $vendor->lead_time_days ? $vendor->lead_time_days . ' days' : '-',
                    $vendor->notes ?? '-',
                ];
            })->all();

            if (!$rows) {
                $rows = $ids->map(fn ($name) => [$name, '-', '-', '-', '-', '-'])->all();
            }

            return ['title' => 'Vendors', 'headers' => $headers, 'rows' => $rows, 'end_col' => 'F'];
        }

        if ($type === 'purchase-orders') {
            $headers = ['PO', 'Vendor', 'Site', 'Value', 'ETA', 'Status'];
            $rows = $ids->map(function ($id) {
                return [$id, '-', '-', '-', '-', '-'];
            })->all();

            return ['title' => 'Purchase Orders', 'headers' => $headers, 'rows' => $rows, 'end_col' => 'F'];
        }

        if ($type === 'payments') {
            $headers = ['Invoice', 'Vendor', 'Site', 'Amount', 'Due', 'Status'];
            $rows = $ids->map(function ($id) {
                return [$id, '-', '-', '-', '-', '-'];
            })->all();

            return ['title' => 'Payments', 'headers' => $headers, 'rows' => $rows, 'end_col' => 'F'];
        }

        return null;
    }

    private function addBranding($sheet): void
    {
        $logoPath = public_path('ui-assets/images/fourways-logo.png');
        if (!is_file($logoPath)) {
            return;
        }

        $logo = new Drawing();
        $logo->setName('Logo');
        $logo->setPath($logoPath);
        $logo->setHeight(48);
        $logo->setCoordinates('A1');
        $logo->setWorksheet($sheet);

        $watermark = new Drawing();
        $watermark->setName('Watermark');
        $watermark->setPath($logoPath);
        $watermark->setHeight(260);
        $watermark->setCoordinates('C6');
        if (method_exists($watermark, 'setOpacity')) {
            $watermark->setOpacity(0.08);
        }
        $watermark->setWorksheet($sheet);
    }

    private function exportPurchaseRequestExcel(PurchaseRequest $requestModel)
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Purchase Request');

        $this->addBranding($sheet);

        $sheet->setCellValue('A4', 'Purchase Request ' . $requestModel->request_code);
        $sheet->mergeCells('A4:H4');
        $sheet->getStyle('A4')->getFont()->setBold(true)->setSize(14);

        $sheet->setCellValue('A6', 'Site');
        $sheet->setCellValue('B6', $requestModel->site?->name ?? 'Unassigned');
        $sheet->setCellValue('D6', 'Status');
        $sheet->setCellValue('E6', ucfirst($requestModel->status ?? 'pending'));

        $headers = ['Item', 'Qty', 'Unit', 'USD', 'ZWL', 'Supplier', 'Comments'];
        $sheet->fromArray($headers, null, 'A8');
        $sheet->getStyle('A8:G8')->getFont()->setBold(true);

        $row = 9;
        foreach ($requestModel->items as $item) {
            $sheet->fromArray([
                $item->item_name ?? 'Item',
                $item->quantity ?? '-',
                $item->unit ?? '-',
                $item->quote_amount_usd ?? '-',
                $item->quote_amount_zwl ?? '-',
                $item->selected_supplier ?? $item->supplier_name ?? '-',
                $item->comments ?? '-',
            ], null, "A{$row}");
            $row++;
        }

        foreach (range('A', 'G') as $column) {
            $sheet->getColumnDimension($column)->setAutoSize(true);
        }

        $filename = Str::slug('purchase-request-' . $requestModel->request_code) . '_' . now()->format('Ymd_His') . '.xlsx';
        $writer = new Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    private function exportPurchaseRequestPdf(PurchaseRequest $requestModel)
    {
        $logoPath = public_path('ui-assets/images/fourways-logo.png');
        $logoData = is_file($logoPath) ? 'data:image/png;base64,' . base64_encode(file_get_contents($logoPath)) : null;

        $pdf = Pdf::loadView('exports.purchase-request', [
            'request' => $requestModel,
            'logo' => $logoData,
        ])->setPaper('a4', 'portrait');

        $filename = Str::slug('purchase-request-' . $requestModel->request_code) . '_' . now()->format('Ymd_His') . '.pdf';
        return $pdf->download($filename);
    }

    private function resolveTenant(Request $request): ?Tenant
    {
        if ($request->user()?->tenant_id) {
            return Tenant::find($request->user()->tenant_id);
        }

        $headerTenant = $request->header('X-Tenant-ID') ?: $request->input('tenant_id');
        if ($headerTenant) {
            return Tenant::find($headerTenant);
        }

        return Tenant::where('name', 'Fourways Group')->first() ?? Tenant::first();
    }
}
