<?php

namespace App\Http\Controllers\Telegram;

use App\Domain\Telegram\TelegramApiClient;
use App\Domain\Telegram\TelegramRouterService;
use App\Models\AuditLog;
use App\Services\Notify\PoultryTelegramCommandService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class TelegramWebhookController
{
    public function handle(Request $request, TelegramRouterService $router, TelegramApiClient $client)
    {
        $secret = config('services.telegram.webhook_secret');
        if ($secret) {
            $header = (string) $request->header('X-Telegram-Bot-Api-Secret-Token');
            if (!hash_equals($secret, $header)) {
                return response()->json(['ok' => false], Response::HTTP_FORBIDDEN);
            }
        }

        $payload = $request->all();
        $context = $this->extractContext($payload);

        if (!$context['chat_id']) {
            return response()->json(['ok' => true]);
        }

        try {
            $this->logInbound($context, $payload);
        } catch (\Throwable $e) {
            // Audit failures should not break Telegram replies.
            report($e);
        }

        try {
            $poultryReply = app(PoultryTelegramCommandService::class)
                ->handle($context['text'] ?? null, (string) $context['chat_id']);
            if ($poultryReply) {
                $client->sendMessage((string) $context['chat_id'], $poultryReply);
                return response()->json(['ok' => true]);
            }
        } catch (\Throwable $e) {
            report($e);
        }

        try {
            $result = $router->handle($context);
        } catch (\Throwable $e) {
            report($e);
            $fallback = 'Sorry, I hit an error while processing that. Try again or use /help. If you uploaded a document, send /intake to check the status.';
            try {
                $client->sendMessage($context['chat_id'], $fallback);
            } catch (\Throwable $sendError) {
                report($sendError);
            }

            return response()->json(['ok' => true]);
        }
        if ($result['reply']) {
            try {
                $client->sendMessage($context['chat_id'], $result['reply']);
            } catch (\Throwable $e) {
                // Never fail the webhook on outbound send errors.
                report($e);
            }
        }

        $documents = $result['documents'] ?? [];
        foreach ($documents as $document) {
            $path = $document['path'] ?? null;
            if (!$path) {
                continue;
            }

            try {
                $client->sendDocument(
                    $context['chat_id'],
                    $path,
                    $document['filename'] ?? null,
                    $document['caption'] ?? null
                );

                if (str_starts_with($path, storage_path('app' . DIRECTORY_SEPARATOR . 'telegram-exports'))) {
                    @unlink($path);
                }
            } catch (\Throwable $e) {
                report($e);
            }
        }

        if (($context['update_type'] ?? null) === 'callback_query' && !empty($context['callback_query_id'])) {
            try {
                $client->answerCallbackQuery((string) $context['callback_query_id'], 'Done');
            } catch (\Throwable $e) {
                report($e);
            }
        }

        return response()->json(['ok' => true]);
    }

    private function extractContext(array $payload): array
    {
        $updateType = 'message';
        $message = $payload['message'] ?? null;
        $from = $message['from'] ?? null;
        $chat = $message['chat'] ?? null;
        $text = $message['text'] ?? null;

        if (isset($payload['edited_message'])) {
            $updateType = 'edited_message';
            $message = $payload['edited_message'];
            $from = $message['from'] ?? null;
            $chat = $message['chat'] ?? null;
            $text = $message['text'] ?? null;
        } elseif (isset($payload['callback_query'])) {
            $updateType = 'callback_query';
            $callback = $payload['callback_query'];
            $message = $callback['message'] ?? null;
            $from = $callback['from'] ?? null;
            $chat = $message['chat'] ?? null;
            $text = $callback['data'] ?? null;
        }

        $attachments = [];
        if ($updateType !== 'callback_query') {
            if (!empty($message['document'])) {
                $doc = $message['document'];
                $attachments[] = [
                    'file_id' => $doc['file_id'] ?? null,
                    'file_name' => $doc['file_name'] ?? 'document',
                    'mime_type' => $doc['mime_type'] ?? null,
                    'type' => 'document',
                ];
            }

            if (!empty($message['photo']) && is_array($message['photo'])) {
                $photo = $message['photo'][count($message['photo']) - 1];
                $fileId = $photo['file_id'] ?? null;
                if ($fileId) {
                    $attachments[] = [
                        'file_id' => $fileId,
                        'file_name' => 'photo_' . $fileId . '.jpg',
                        'mime_type' => 'image/jpeg',
                        'type' => 'photo',
                    ];
                }
            }
        }

        return [
            'update_id' => $payload['update_id'] ?? null,
            'update_type' => $updateType,
            'chat_id' => $chat['id'] ?? null,
            'chat_type' => $chat['type'] ?? null,
            'telegram_user_id' => $from['id'] ?? null,
            'text' => is_string($text) ? trim($text) : null,
            'attachments' => $attachments,
            'callback_query_id' => data_get($payload, 'callback_query.id'),
        ];
    }

    private function logInbound(array $context, array $payload): void
    {
        AuditLog::create([
            'tenant_id' => null,
            'user_id' => null,
            'channel' => 'telegram',
            'action' => 'telegram.update',
            'payload_json' => [
                'update_id' => $context['update_id'],
                'update_type' => $context['update_type'],
                'chat_id' => $context['chat_id'],
                'telegram_user_id' => $context['telegram_user_id'],
                'has_attachments' => !empty($context['attachments']),
                'raw' => [
                    'message_id' => data_get($payload, 'message.message_id'),
                    'edited_message_id' => data_get($payload, 'edited_message.message_id'),
                    'callback_query_id' => data_get($payload, 'callback_query.id'),
                ],
            ],
            'created_at' => now(),
        ]);
    }
}
