<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Models\Telemetry;
use App\Models\Vehicle;
use App\Services\TelemetryIngestionService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class TelemetryController extends Controller
{
    protected $telemetryIngestionService;

    public function __construct(TelemetryIngestionService $telemetryIngestionService)
    {
        $this->telemetryIngestionService = $telemetryIngestionService;
    }

    public function ingest(Request $request): JsonResponse
    {
        $validatedData = $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'data' => 'required|array',
            'data.*.timestamp' => 'required|date',
            'data.*.usage_hours' => 'nullable|numeric',
            'data.*.asset_id' => 'nullable|integer',
        ]);

        $tenantId = $request->user()->tenant_id;
        $vehicle = Vehicle::where('tenant_id', $tenantId)->findOrFail($validatedData['vehicle_id']);
        $assetIds = collect($validatedData['data'])
            ->pluck('asset_id')
            ->filter()
            ->unique()
            ->values();

        if ($assetIds->isNotEmpty()) {
            $assetCount = Asset::where('tenant_id', $tenantId)
                ->whereIn('id', $assetIds)
                ->count();

            if ($assetCount !== $assetIds->count()) {
                return response()->json(['message' => 'Invalid asset reference'], 422);
            }
        }

        $telemetryData = $this->telemetryIngestionService->processTelemetryData(
            $vehicle->id,
            $validatedData['data'],
            $tenantId
        );

        return response()->json([
            'message' => 'Telemetry data ingested successfully.',
            'data' => $telemetryData,
        ], 201);
    }

    public function index(): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;
        $telemetries = Telemetry::where('tenant_id', $tenantId)->get();

        return response()->json($telemetries);
    }

    public function show($id): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;
        $telemetry = Telemetry::where('tenant_id', $tenantId)->findOrFail($id);

        return response()->json($telemetry);
    }
}
