<?php

namespace App\Http\Controllers;

use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class TenantController extends Controller
{
    public function index()
    {
        $tenantId = Auth::user()->tenant_id;
        $tenants = Tenant::where('id', $tenantId)->get();
        return response()->json($tenants);
    }

    public function show($id)
    {
        $tenantId = Auth::user()->tenant_id;
        if ((int) $id !== (int) $tenantId) {
            return response()->json(['message' => 'Tenant not found'], 404);
        }
        $tenant = Tenant::find($tenantId);
        return response()->json($tenant);
    }

    public function store(Request $request)
    {
        if (Auth::user()->tenant_id) {
            return response()->json(['message' => 'Tenant already assigned'], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:tenants,email',
            'subdomain' => 'required|string|max:255|unique:tenants,subdomain',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $data = $request->all();
        if (!empty($data['password'])) {
            $data['password'] = bcrypt($data['password']);
        }

        $tenant = Tenant::create($data);
        return response()->json($tenant, 201);
    }

    public function update(Request $request, $id)
    {
        $tenantId = Auth::user()->tenant_id;
        if ((int) $id !== (int) $tenantId) {
            return response()->json(['message' => 'Tenant not found'], 404);
        }

        $tenant = Tenant::find($tenantId);
        if (!$tenant) {
            return response()->json(['message' => 'Tenant not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|email|unique:tenants,email,' . $tenant->id,
            'subdomain' => 'sometimes|required|string|max:255|unique:tenants,subdomain,' . $tenant->id,
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $data = $request->all();
        if (!empty($data['password'])) {
            $data['password'] = bcrypt($data['password']);
        }

        $tenant->update($data);
        return response()->json($tenant);
    }

    public function destroy($id)
    {
        $tenantId = Auth::user()->tenant_id;
        if ((int) $id !== (int) $tenantId) {
            return response()->json(['message' => 'Tenant not found'], 404);
        }

        $tenant = Tenant::find($tenantId);
        if (!$tenant) {
            return response()->json(['message' => 'Tenant not found'], 404);
        }

        $tenant->delete();
        return response()->json(['message' => 'Tenant deleted successfully']);
    }
}
