<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Alert;
use App\Models\BehaviorMetric;
use App\Models\House;
use App\Models\SensorReading;
use App\Models\WeightSample;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $houses = House::query()->get();
        $cards = $houses->map(function (House $house) {
            return [
                'house' => $house,
                'temp' => $this->latestSensorValue($house, 'temp_c'),
                'humidity' => $this->latestSensorValue($house, 'humidity_pct'),
                'water_flow' => $this->latestSensorValue($house, 'water_flow_lpm'),
                'feed_level' => $this->latestSensorValue($house, 'feed_level_pct'),
                'weight' => WeightSample::where('house_id', $house->id)->orderByDesc('ts')->value('avg_weight_g'),
                'behavior' => BehaviorMetric::where('house_id', $house->id)
                    ->where('ts', '>=', Carbon::now('UTC')->subDay())
                    ->avg('abnormal_score'),
                'open_alerts' => Alert::where('house_id', $house->id)->where('status', 'open')->count(),
            ];
        });

        return view('poultry.dashboard', [
            'cards' => $cards,
        ]);
    }

    private function latestSensorValue(House $house, string $type): ?float
    {
        $value = SensorReading::query()
            ->join('sensors', 'sensors.id', '=', 'sensor_readings.sensor_id')
            ->join('devices', 'devices.id', '=', 'sensors.device_id')
            ->where('devices.house_id', $house->id)
            ->where('sensors.type', $type)
            ->orderByDesc('sensor_readings.ts')
            ->value('sensor_readings.value');

        return $value !== null ? (float) $value : null;
    }
}
