<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\AiReport;
use App\Models\Alert;
use App\Models\BehaviorMetric;
use App\Models\FeedLog;
use App\Models\House;
use App\Models\MortalityLog;
use App\Models\SensorReading;
use App\Models\WaterLog;
use App\Models\WeightSample;
use App\Services\Ai\PoultryAiService;
use Carbon\Carbon;
use Illuminate\Http\Request;

class HouseController extends Controller
{
    public function show(House $house)
    {
        $now = Carbon::now('UTC');
        $since = $now->copy()->subDay();

        $readings = SensorReading::query()
            ->select('sensors.type', 'sensor_readings.ts', 'sensor_readings.value')
            ->join('sensors', 'sensors.id', '=', 'sensor_readings.sensor_id')
            ->join('devices', 'devices.id', '=', 'sensors.device_id')
            ->where('devices.house_id', $house->id)
            ->whereBetween('sensor_readings.ts', [$since, $now])
            ->orderBy('sensor_readings.ts')
            ->get();

        $environment = $this->buildChartData($readings, ['temp_c', 'humidity_pct'], $house->timezone);
        $waterFeed = $this->buildChartData($readings, ['water_flow_lpm', 'feed_level_pct'], $house->timezone);

        $weightSamples = WeightSample::where('house_id', $house->id)
            ->where('ts', '>=', $now->copy()->subDays(7))
            ->orderBy('ts')
            ->get();
        $weightLabels = $weightSamples->map(fn ($sample) => $sample->ts->setTimezone($house->timezone)->toDateString());
        $weightValues = $weightSamples->map(fn ($sample) => (float) $sample->avg_weight_g);

        $behaviorSamples = BehaviorMetric::where('house_id', $house->id)
            ->whereBetween('ts', [$since, $now])
            ->orderBy('ts')
            ->get();
        $behaviorLabels = $behaviorSamples->map(fn ($metric) => $metric->ts->setTimezone($house->timezone)->toDateTimeString());
        $behaviorValues = $behaviorSamples->map(fn ($metric) => (float) $metric->abnormal_score);

        $feedLogs = FeedLog::where('house_id', $house->id)
            ->orderByDesc('date')
            ->limit(7)
            ->get();
        $waterLogs = WaterLog::where('house_id', $house->id)
            ->orderByDesc('date')
            ->limit(7)
            ->get();
        $mortalityLogs = MortalityLog::where('house_id', $house->id)
            ->orderByDesc('date')
            ->limit(7)
            ->get();

        $alerts = Alert::where('house_id', $house->id)
            ->orderByDesc('created_at')
            ->limit(10)
            ->get();

        $latestReport = AiReport::where('house_id', $house->id)
            ->where('report_type', 'poultry_ops')
            ->orderByDesc('created_at')
            ->first();

        return view('poultry.house', [
            'house' => $house,
            'environment' => $environment,
            'waterFeed' => $waterFeed,
            'weightLabels' => $weightLabels,
            'weightValues' => $weightValues,
            'behaviorLabels' => $behaviorLabels,
            'behaviorValues' => $behaviorValues,
            'feedLogs' => $feedLogs,
            'waterLogs' => $waterLogs,
            'mortalityLogs' => $mortalityLogs,
            'alerts' => $alerts,
            'latestReport' => $latestReport,
        ]);
    }

    public function generateReport(Request $request, House $house, PoultryAiService $service)
    {
        $period = $request->input('period', 'daily');
        $report = $service->generateReport($house, $period === 'weekly' ? 'weekly' : 'daily');

        return redirect()
            ->route('poultry.houses.show', $house)
            ->with('status', 'AI summary generated (Report #' . $report->id . ').');
    }

    private function buildChartData($readings, array $types, string $timezone): array
    {
        $series = [];
        $labels = [];

        foreach ($types as $type) {
            $points = $readings->where('type', $type)->map(function ($row) {
                return [
                    'ts' => Carbon::parse($row->ts)->setTimezone($timezone)->toDateTimeString(),
                    'value' => (float) $row->value,
                ];
            })->values();
            $series[$type] = $points;
            $labels = array_merge($labels, $points->pluck('ts')->all());
        }

        $labels = collect($labels)->unique()->sort()->values();
        $datasets = [];

        foreach ($types as $type) {
            $map = collect($series[$type])->keyBy('ts');
            $datasets[$type] = $labels->map(function ($label) use ($map) {
                return $map[$label]['value'] ?? null;
            });
        }

        return [
            'labels' => $labels,
            'datasets' => $datasets,
        ];
    }
}
