<?php

namespace App\Http\Controllers;

use App\Models\WorkOrder;
use App\Services\AuditService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class WorkOrderController extends Controller
{
    public function index()
    {
        return WorkOrder::with(['asset', 'assignedTo'])->latest()->get();
    }

    public function store(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $data = $request->validate([
            'asset_id' => [
                'required',
                Rule::exists('assets', 'id')->where('tenant_id', $tenantId),
            ],
            'checklist_id' => [
                'nullable',
                Rule::exists('checklists', 'id')->where('tenant_id', $tenantId),
            ],
            'downtime_reason_id' => [
                'nullable',
                Rule::exists('downtime_reason_codes', 'id')->where('tenant_id', $tenantId),
            ],
            'type' => 'nullable|string',
            'status' => 'nullable|string',
            'priority' => 'nullable|string',
            'description' => 'nullable|string',
            'reported_by' => [
                'nullable',
                Rule::exists('users', 'id')->where('tenant_id', $tenantId),
            ],
            'assigned_to' => [
                'nullable',
                Rule::exists('users', 'id')->where('tenant_id', $tenantId),
            ],
            'meter_reading' => 'nullable|numeric',
            'due_at' => 'nullable|date',
        ]);

        $workOrder = WorkOrder::create($data);
        app(AuditService::class)->log(
            'work_order.created',
            WorkOrder::class,
            $workOrder->id,
            $workOrder->toArray(),
            $request->user()->tenant_id,
            $request->user()->id,
            $request
        );

        return response()->json($workOrder, 201);
    }

    public function show(WorkOrder $workOrder)
    {
        return $workOrder->load(['asset', 'assignedTo', 'costs', 'parts']);
    }

    public function update(Request $request, WorkOrder $workOrder)
    {
        $tenantId = $request->user()->tenant_id;
        $data = $request->validate([
            'status' => 'nullable|string',
            'priority' => 'nullable|string',
            'description' => 'nullable|string',
            'cause' => 'nullable|string',
            'resolution' => 'nullable|string',
            'assigned_to' => [
                'nullable',
                Rule::exists('users', 'id')->where('tenant_id', $tenantId),
            ],
            'started_at' => 'nullable|date',
            'completed_at' => 'nullable|date',
        ]);

        $workOrder->update($data);
        app(AuditService::class)->log(
            'work_order.updated',
            WorkOrder::class,
            $workOrder->id,
            $workOrder->getChanges(),
            $request->user()->tenant_id,
            $request->user()->id,
            $request
        );

        return response()->json($workOrder);
    }
}
