<?php

namespace App\Jobs;

use App\Models\Device;
use App\Models\GpsPoint;
use App\Models\MeterReading;
use App\Models\SensorEventRaw;
use App\Models\SensorMetric;
use App\Services\RulesEngineService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;

class NormalizeSensorEvent implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(public int $eventId)
    {
    }

    public function handle(RulesEngineService $rulesEngine): void
    {
        $event = SensorEventRaw::find($this->eventId);
        if (!$event) {
            return;
        }

        $device = Device::find($event->device_id);
        if ($device) {
            $device->update(['last_seen_at' => $event->occurred_at]);
        }

        $payload = $event->payload ?? [];
        $assetId = $device?->asset_id ?? ($payload['asset_id'] ?? null);

        $metrics = $payload['metrics'] ?? $payload['readings'] ?? $payload;
        if (is_array($metrics)) {
            foreach ($metrics as $metric => $value) {
                if (!is_numeric($value) || $assetId === null) {
                    continue;
                }

                SensorMetric::create([
                    'tenant_id' => $event->tenant_id,
                    'asset_id' => $assetId,
                    'metric' => (string) $metric,
                    'value' => (float) $value,
                    'unit' => null,
                    'source' => 'device',
                    'occurred_at' => $event->occurred_at,
                ]);

                $meterType = $this->meterTypeForMetric((string) $metric);
                if ($meterType) {
                    MeterReading::create([
                        'tenant_id' => $event->tenant_id,
                        'asset_id' => $assetId,
                        'meter_type' => $meterType,
                        'value' => (float) $value,
                        'source' => 'device',
                        'recorded_at' => $event->occurred_at,
                    ]);
                }

                $rulesEngine->evaluateMetric(
                    $event->tenant_id,
                    $assetId,
                    (string) $metric,
                    (float) $value,
                    Carbon::parse($event->occurred_at)
                );
            }
        }

        $this->handleGps($event, $payload, $assetId);
    }

    protected function handleGps(SensorEventRaw $event, array $payload, ?int $assetId): void
    {
        $gps = $payload['gps'] ?? null;
        $latitude = $gps['lat'] ?? $gps['latitude'] ?? ($payload['latitude'] ?? null);
        $longitude = $gps['lon'] ?? $gps['longitude'] ?? ($payload['longitude'] ?? null);

        if ($latitude === null || $longitude === null) {
            return;
        }

        GpsPoint::create([
            'tenant_id' => $event->tenant_id,
            'asset_id' => $assetId,
            'device_id' => $event->device_id,
            'latitude' => (float) $latitude,
            'longitude' => (float) $longitude,
            'speed' => isset($gps['speed']) ? (float) $gps['speed'] : null,
            'heading' => isset($gps['heading']) ? (float) $gps['heading'] : null,
            'occurred_at' => $event->occurred_at,
        ]);
    }

    protected function meterTypeForMetric(string $metric): ?string
    {
        $metric = strtolower($metric);
        if (in_array($metric, ['engine_hours', 'runtime_hours', 'hours', 'usage_hours'], true)) {
            return 'hours';
        }

        if (in_array($metric, ['odometer', 'distance', 'km'], true)) {
            return 'km';
        }

        return null;
    }
}
