<?php

namespace App\Jobs;

use App\Domain\Telegram\TelegramNotifier;
use App\Models\NotificationOutbox;
use App\Models\TelegramGroup;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessNotificationOutbox implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function handle(TelegramNotifier $notifier): void
    {
        $batch = NotificationOutbox::where('status', 'queued')
            ->where(function ($query) {
                $query->whereNull('available_at')->orWhere('available_at', '<=', now());
            })
            ->limit(25)
            ->get();

        foreach ($batch as $item) {
            try {
                if ($item->channel === 'telegram') {
                    $this->dispatchTelegram($item, $notifier);
                }

                $item->update([
                    'status' => 'sent',
                    'sent_at' => now(),
                ]);
            } catch (\Throwable $e) {
                $item->update([
                    'status' => 'failed',
                    'failed_at' => now(),
                    'error_message' => $e->getMessage(),
                ]);
            }
        }
    }

    private function dispatchTelegram(NotificationOutbox $item, TelegramNotifier $notifier): void
    {
        $recipients = $item->recipients ?? [];
        if (!$recipients) {
            return;
        }

        foreach ($recipients as $recipient) {
            if (is_numeric($recipient)) {
                $notifier->notifyUser((int) $recipient, $item->message);
                continue;
            }

            if (is_string($recipient)) {
                $groups = TelegramGroup::where('tenant_id', $item->tenant_id)
                    ->whereIn('name', config('telegram_events.groups.' . $recipient, [$recipient]))
                    ->where('enabled', true)
                    ->get();
                foreach ($groups as $group) {
                    $notifier->notifyChat($group->chat_id, $item->message);
                }
            }
        }
    }
}
