<?php

namespace App\Services\Ai;

use App\Models\AiReport;
use App\Models\Alert;
use App\Models\BehaviorMetric;
use App\Models\ExpectedGrowthCurve;
use App\Models\FeedLog;
use App\Models\House;
use App\Models\MortalityLog;
use App\Models\SensorReading;
use App\Models\WaterLog;
use App\Models\WeightSample;
use App\Services\Notify\TelegramNotifier;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class PoultryAiService
{
    public function __construct(private TelegramNotifier $telegramNotifier)
    {
    }

    public function generateReport(House $house, string $period = 'daily'): AiReport
    {
        $end = Carbon::now('UTC');
        $start = $period === 'weekly' ? $end->copy()->subDays(7) : $end->copy()->subDay();

        $metrics = $this->collectMetrics($house, $start, $end);
        $summary = $this->generateAiSummary($house, $metrics, $start, $end, $period);

        return AiReport::create([
            'tenant_id' => $house->tenant_id,
            'house_id' => $house->id,
            'report_type' => 'poultry_ops',
            'period' => $period,
            'period_start' => $start->toDateString(),
            'period_end' => $end->toDateString(),
            'content' => $summary,
            'report_text' => $summary,
            'metrics_json' => $metrics,
        ]);
    }

    public function evaluateAlerts(House $house): array
    {
        $alerts = [];
        $now = Carbon::now('UTC');

        $range = $this->tempHumidityRange($house->age_days);
        if ($range) {
            $stats = $this->sensorStats($house, $now->copy()->subHours(6), $now);
            $temp = $stats['temp_c']['avg'] ?? null;
            $humidity = $stats['humidity_pct']['avg'] ?? null;

            if ($temp !== null && ($temp < $range['temp_min'] || $temp > $range['temp_max'])) {
                $severity = $this->severityFromDelta($temp, $range['temp_min'], $range['temp_max']);
                $alerts[] = $this->createAlertIfNew(
                    $house,
                    'environment',
                    'Temperature out of range',
                    sprintf('Average temperature %.1fC is outside %.1f%.1fC.', $temp, $range['temp_min'], $range['temp_max']),
                    $severity,
                    ['Adjust ventilation or heating to stabilize temperature.']
                );
            }

            if ($humidity !== null && ($humidity < $range['humidity_min'] || $humidity > $range['humidity_max'])) {
                $severity = $this->severityFromDelta($humidity, $range['humidity_min'], $range['humidity_max']);
                $alerts[] = $this->createAlertIfNew(
                    $house,
                    'environment',
                    'Humidity out of range',
                    sprintf('Average humidity %.1f%% is outside %.1f%.1f%%.', $humidity, $range['humidity_min'], $range['humidity_max']),
                    $severity,
                    ['Check ventilation, litter moisture, and drinker leaks.']
                );
            }
        }

        $alerts[] = $this->evaluateWaterFlow($house, $now);
        $alerts[] = $this->evaluateFeedIntake($house, $now);
        $alerts[] = $this->evaluateMortality($house, $now);
        $alerts[] = $this->evaluateBehavior($house, $now);
        $alerts[] = $this->evaluateWeightGain($house, $now);

        $alerts = array_filter($alerts);
        foreach ($alerts as $alert) {
            if ($alert->severity === 'critical') {
                $this->telegramNotifier->sendAlert($alert);
            }
        }

        return $alerts;
    }

    public function inferBehaviorFromFrame(House $house, int $deviceId, Carbon $ts, array $meta, string $path): BehaviorMetric
    {
        $seed = crc32($path . '|' . json_encode($meta));
        $activity = ($seed % 1000) / 1000;
        $clustering = (int) (floor($seed / 1000) % 1000) / 1000;
        $abnormal = min(1, max(0, ($activity * 0.6) + ($clustering * 0.4)));

        return BehaviorMetric::updateOrCreate([
            'house_id' => $house->id,
            'device_id' => $deviceId,
            'ts' => $ts->toDateTimeString(),
        ], [
            'activity_score' => round($activity, 3),
            'clustering_score' => round($clustering, 3),
            'abnormal_score' => round($abnormal, 3),
            'meta_json' => array_merge($meta, ['inference' => 'server_stub']),
        ]);
    }

    public function collectMetrics(House $house, Carbon $start, Carbon $end): array
    {
        $sensorStats = $this->sensorStats($house, $start, $end);
        $weightSamples = WeightSample::where('house_id', $house->id)
            ->whereBetween('ts', [$start, $end])
            ->orderBy('ts')
            ->get();

        $weightStart = $weightSamples->first()?->avg_weight_g;
        $weightEnd = $weightSamples->last()?->avg_weight_g;
        $weightGainG = $weightStart && $weightEnd ? max(0, $weightEnd - $weightStart) : null;
        $days = max(1, $start->diffInDays($end));
        $gainPerDay = $weightGainG !== null ? $weightGainG / $days : null;

        $feedKg = FeedLog::where('house_id', $house->id)
            ->whereBetween('date', [$start->toDateString(), $end->toDateString()])
            ->sum('feed_kg');
        $waterL = WaterLog::where('house_id', $house->id)
            ->whereBetween('date', [$start->toDateString(), $end->toDateString()])
            ->sum('water_l');
        $mortality = MortalityLog::where('house_id', $house->id)
            ->whereBetween('date', [$start->toDateString(), $end->toDateString()])
            ->sum('deaths');
        $birds = max(1, $house->capacity);

        $totalGainKg = $weightGainG !== null ? ($weightGainG / 1000) * $birds : null;
        $fcr = $totalGainKg && $totalGainKg > 0 ? $feedKg / $totalGainKg : null;
        $waterFeedRatio = $feedKg > 0 ? $waterL / $feedKg : null;
        $mortalityPerThousand = ($mortality / $birds) * 1000;

        $behaviorAvg = BehaviorMetric::where('house_id', $house->id)
            ->whereBetween('ts', [$start, $end])
            ->avg('abnormal_score');

        return [
            'period' => [
                'start' => $start->toDateTimeString(),
                'end' => $end->toDateTimeString(),
            ],
            'sensors' => $sensorStats,
            'weight' => [
                'start_g' => $weightStart,
                'end_g' => $weightEnd,
                'gain_g' => $weightGainG,
                'gain_per_day_g' => $gainPerDay,
            ],
            'feed_kg' => round($feedKg, 2),
            'water_l' => round($waterL, 2),
            'water_feed_ratio' => $waterFeedRatio ? round($waterFeedRatio, 3) : null,
            'mortality' => (int) $mortality,
            'mortality_per_1000' => round($mortalityPerThousand, 2),
            'fcr_estimate' => $fcr ? round($fcr, 2) : null,
            'behavior_abnormal_avg' => $behaviorAvg ? round($behaviorAvg, 3) : null,
        ];
    }

    private function sensorStats(House $house, Carbon $start, Carbon $end): array
    {
        $rows = SensorReading::query()
            ->select('sensors.type', DB::raw('avg(sensor_readings.value) as avg'), DB::raw('min(sensor_readings.value) as min'), DB::raw('max(sensor_readings.value) as max'), DB::raw('count(*) as count'))
            ->join('sensors', 'sensors.id', '=', 'sensor_readings.sensor_id')
            ->join('devices', 'devices.id', '=', 'sensors.device_id')
            ->where('devices.house_id', $house->id)
            ->whereBetween('sensor_readings.ts', [$start, $end])
            ->groupBy('sensors.type')
            ->get();

        $stats = [];
        foreach ($rows as $row) {
            $stats[$row->type] = [
                'avg' => $row->avg !== null ? round((float) $row->avg, 2) : null,
                'min' => $row->min !== null ? round((float) $row->min, 2) : null,
                'max' => $row->max !== null ? round((float) $row->max, 2) : null,
                'count' => (int) $row->count,
            ];
        }

        return $stats;
    }

    private function tempHumidityRange(int $ageDays): ?array
    {
        $ranges = config('poultry.temp_humidity_by_age', []);
        foreach ($ranges as $range) {
            if ($ageDays >= $range['min_day'] && $ageDays <= $range['max_day']) {
                return $range;
            }
        }

        return null;
    }

    private function evaluateWaterFlow(House $house, Carbon $now): ?Alert
    {
        $baselineStart = $now->copy()->subDays(3);
        $recentStart = $now->copy()->subDay();

        $baseline = $this->sensorAvg($house, 'water_flow_lpm', $baselineStart, $recentStart);
        $recent = $this->sensorAvg($house, 'water_flow_lpm', $recentStart, $now);

        if ($baseline === null || $recent === null || $baseline <= 0) {
            return null;
        }

        $delta = ($recent - $baseline) / $baseline;
        if (abs($delta) < config('poultry.water_flow_delta_threshold', 0.3)) {
            return null;
        }

        $severity = abs($delta) >= 0.5 ? 'critical' : 'warn';
        $direction = $delta > 0 ? 'spike' : 'drop';
        $message = sprintf('Water flow %s: recent avg %.2f LPM vs baseline %.2f LPM.', $direction, $recent, $baseline);

        return $this->createAlertIfNew(
            $house,
            'water',
            'Water flow anomaly',
            $message,
            $severity,
            ['Inspect drinker lines, pressure regulators, and leaks.']
        );
    }

    private function evaluateFeedIntake(House $house, Carbon $now): ?Alert
    {
        $today = $now->toDateString();
        $recent = FeedLog::where('house_id', $house->id)->where('date', $today)->value('feed_kg');

        $baseline = FeedLog::where('house_id', $house->id)
            ->whereBetween('date', [$now->copy()->subDays(3)->toDateString(), $now->copy()->subDay()->toDateString()])
            ->avg('feed_kg');

        if ($recent === null || $baseline === null || $baseline <= 0) {
            return null;
        }

        $delta = ($recent - $baseline) / $baseline;
        if (abs($delta) < config('poultry.feed_intake_delta_threshold', 0.25)) {
            return null;
        }

        $severity = abs($delta) >= 0.4 ? 'critical' : 'warn';
        $direction = $delta > 0 ? 'spike' : 'drop';
        $message = sprintf('Feed intake %s: today %.2f kg vs 3-day avg %.2f kg.', $direction, $recent, $baseline);

        return $this->createAlertIfNew(
            $house,
            'feed',
            'Feed intake anomaly',
            $message,
            $severity,
            ['Check feeders, feed formulation, and bird appetite trends.']
        );
    }

    private function evaluateMortality(House $house, Carbon $now): ?Alert
    {
        $today = $now->toDateString();
        $deaths = (int) (MortalityLog::where('house_id', $house->id)->where('date', $today)->value('deaths') ?? 0);
        if ($deaths <= 0) {
            return null;
        }

        $rate = ($deaths / max(1, $house->capacity)) * 1000;
        $threshold = config('poultry.mortality_threshold_per_1000', 5);
        if ($rate < $threshold) {
            return null;
        }

        $severity = $rate >= ($threshold * 2) ? 'critical' : 'warn';
        $message = sprintf('Mortality %.1f per 1000 birds today (%d deaths).', $rate, $deaths);

        return $this->createAlertIfNew(
            $house,
            'mortality',
            'Mortality spike',
            $message,
            $severity,
            ['Inspect flock, check environment, and consult a vet if symptoms persist.']
        );
    }

    private function evaluateBehavior(House $house, Carbon $now): ?Alert
    {
        $recentAvg = BehaviorMetric::where('house_id', $house->id)
            ->whereBetween('ts', [$now->copy()->subHours(6), $now])
            ->avg('abnormal_score');

        if ($recentAvg === null) {
            return null;
        }

        $threshold = config('poultry.behavior_abnormal_threshold', 0.7);
        if ($recentAvg < $threshold) {
            return null;
        }

        $severity = $recentAvg >= 0.85 ? 'critical' : 'warn';
        $message = sprintf('Behavior abnormality average %.2f over last 6h.', $recentAvg);

        return $this->createAlertIfNew(
            $house,
            'behavior',
            'Behavior anomaly',
            $message,
            $severity,
            ['Review camera footage, check stocking density, and verify ventilation.']
        );
    }

    private function evaluateWeightGain(House $house, Carbon $now): ?Alert
    {
        $latest = WeightSample::where('house_id', $house->id)->orderByDesc('ts')->first();
        if (!$latest) {
            return null;
        }

        $expected = ExpectedGrowthCurve::where('age_day', $house->age_days)->value('expected_weight_g');
        if (!$expected) {
            return null;
        }

        if ($latest->avg_weight_g >= $expected * 0.9) {
            return null;
        }

        $severity = $latest->avg_weight_g < $expected * 0.8 ? 'critical' : 'warn';
        $message = sprintf('Avg weight %.0fg below expected %.0fg for age day %d.', $latest->avg_weight_g, $expected, $house->age_days);

        return $this->createAlertIfNew(
            $house,
            'weight',
            'Weight gain lagging',
            $message,
            $severity,
            ['Check feed quality, adjust feeding schedule, and monitor uniformity.']
        );
    }

    private function createAlertIfNew(House $house, string $category, string $title, string $message, string $severity, array $actions): ?Alert
    {
        $existing = Alert::where('house_id', $house->id)
            ->where('category', $category)
            ->where('status', 'open')
            ->where('created_at', '>=', now('UTC')->subHours(12))
            ->first();

        if ($existing) {
            return null;
        }

        return Alert::create([
            'house_id' => $house->id,
            'severity' => $severity,
            'category' => $category,
            'title' => $title,
            'message' => $message,
            'status' => 'open',
            'suggested_actions_json' => $actions,
        ]);
    }

    private function generateAiSummary(House $house, array $metrics, Carbon $start, Carbon $end, string $period): string
    {
        $apiKey = config('services.openai.key');
        $endpoint = config('services.openai.endpoint');
        $model = config('services.openai.model', 'gpt-4o-mini');

        $prompt = $this->buildPrompt($house, $metrics, $start, $end, $period);
        if (!$apiKey || !$endpoint) {
            return $this->fallbackSummary($metrics, $period);
        }

        try {
            $headers = [];
            if (config('services.openai.organization')) {
                $headers['OpenAI-Organization'] = config('services.openai.organization');
            }
            if (config('services.openai.project')) {
                $headers['OpenAI-Project'] = config('services.openai.project');
            }

            $response = Http::withToken($apiKey)
                ->withHeaders($headers)
                ->timeout((int) config('services.openai.timeout', 20))
                ->post($endpoint, [
                    'model' => $model,
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are PoultryOps AI. Provide operational insights only. Do not diagnose diseases. Use cautious language.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'temperature' => 0.2,
                    'max_tokens' => 400,
                ]);

            if ($response->failed()) {
                return $this->fallbackSummary($metrics, $period);
            }

            $text = data_get($response->json(), 'choices.0.message.content');
            if (!is_string($text) || trim($text) === '') {
                return $this->fallbackSummary($metrics, $period);
            }

            return trim($text);
        } catch (\Throwable $e) {
            report($e);
            return $this->fallbackSummary($metrics, $period);
        }
    }

    private function buildPrompt(House $house, array $metrics, Carbon $start, Carbon $end, string $period): string
    {
        return implode("\n", [
            "House: {$house->name} (age {$house->age_days} days, capacity {$house->capacity})",
            "Period: {$start->toDateTimeString()} to {$end->toDateTimeString()} ({$period})",
            'Metrics JSON:',
            json_encode($metrics, JSON_PRETTY_PRINT),
            'Provide: 1) Possible operational factors (no diagnosis) 2) Recommended actions 3) What to monitor next.',
        ]);
    }

    private function fallbackSummary(array $metrics, string $period): string
    {
        $lines = [
            ucfirst($period) . ' poultry summary (fallback).',
            'Feed (kg): ' . ($metrics['feed_kg'] ?? 'n/a'),
            'Water (L): ' . ($metrics['water_l'] ?? 'n/a'),
            'Weight gain (g/day): ' . ($metrics['weight']['gain_per_day_g'] ?? 'n/a'),
            'Mortality per 1000: ' . ($metrics['mortality_per_1000'] ?? 'n/a'),
            'Behavior abnormal avg: ' . ($metrics['behavior_abnormal_avg'] ?? 'n/a'),
            'Actions: Verify ventilation, check drinker lines, and confirm feeder flow. Monitor temperatures closely.',
        ];

        return implode("\n", $lines);
    }

    private function sensorAvg(House $house, string $type, Carbon $start, Carbon $end): ?float
    {
        $avg = SensorReading::query()
            ->join('sensors', 'sensors.id', '=', 'sensor_readings.sensor_id')
            ->join('devices', 'devices.id', '=', 'sensors.device_id')
            ->where('devices.house_id', $house->id)
            ->where('sensors.type', $type)
            ->whereBetween('sensor_readings.ts', [$start, $end])
            ->avg('sensor_readings.value');

        return $avg !== null ? (float) $avg : null;
    }

    private function severityFromDelta(float $value, float $min, float $max): string
    {
        if ($value < $min) {
            $delta = $min - $value;
        } else {
            $delta = $value - $max;
        }

        return $delta >= 2 ? 'critical' : 'warn';
    }
}
