<?php

namespace App\Services;

use App\Models\AgentNotification;
use App\Models\Asset;
use App\Models\DowntimeLog;
use App\Models\FuelLog;
use App\Models\MeterReading;
use App\Models\PmSchedule;
use App\Models\Tenant;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Models\WorkOrderPart;
use Illuminate\Support\Facades\Cache;

class AiAgentService
{
    public function __construct(private TelegramEventService $events)
    {
    }

    public function runMaintenanceComplianceAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->runMaintenanceCompliance((int) $tenant->id);
            }
        });
    }

    public function runFailurePredictionAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->runFailurePrediction((int) $tenant->id);
            }
        });
    }

    public function runKpiAnalystAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->runKpiAnalyst((int) $tenant->id);
            }
        });
    }

    public function runCostControl(int $tenantId, string $source, ?int $sourceId = null): void
    {
        if ($this->isThrottled($tenantId, 'cost_control', 60)) {
            return;
        }

        $windowStart = now()->subDays(30);
        $todayStart = now()->startOfDay();

        $avgMaintenance = (float) WorkOrderCost::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $windowStart)
            ->sum('total_cost');
        $avgMaintenance = $avgMaintenance / max(1, 30);

        $todayMaintenance = (float) WorkOrderCost::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $todayStart)
            ->sum('total_cost');

        $avgFuel = (float) FuelLog::where('tenant_id', $tenantId)
            ->where('logged_at', '>=', $windowStart)
            ->sum('total_cost');
        $avgFuel = $avgFuel / max(1, 30);

        $todayFuel = (float) FuelLog::where('tenant_id', $tenantId)
            ->where('logged_at', '>=', $todayStart)
            ->sum('total_cost');

        $findings = [];
        $severity = 'warn';
        if ($avgMaintenance > 0 && $todayMaintenance > ($avgMaintenance * 1.2)) {
            $findings[] = 'Maintenance costs spiked to $' . number_format($todayMaintenance, 2) .
                ' (avg $' . number_format($avgMaintenance, 2) . ')';
            if ($todayMaintenance > ($avgMaintenance * 1.4)) {
                $severity = 'alarm';
            }
        }
        if ($avgFuel > 0 && $todayFuel > ($avgFuel * 1.2)) {
            $findings[] = 'Fuel costs spiked to $' . number_format($todayFuel, 2) .
                ' (avg $' . number_format($avgFuel, 2) . ')';
            if ($todayFuel > ($avgFuel * 1.4)) {
                $severity = 'alarm';
            }
        }

        if (!$findings) {
            return;
        }

        $message = implode("\n", array_map(fn ($line) => '- ' . $line, $findings));
        $summary = "Cost Control Alert\n{$message}\nRecommendation: Investigate recent jobs and fuel usage spikes.";

        $notification = $this->recordNotification(
            $tenantId,
            'cost_control',
            'Cost Control Agent',
            'Cost Control Alert',
            $summary,
            $severity,
            [
                'source' => $source,
                'source_id' => $sourceId,
                'findings' => $findings,
            ],
            ['maintenance', 'management']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.cost_control',
            $severity,
            'Cost Control Alert',
            $summary,
            [
                'details' => $findings,
                'title' => 'COST CONTROL',
            ]
        );

        if ($notification && $notification->severity === 'alarm') {
            $this->maybeNotifyOps($tenantId, $notification, $findings);
        }
    }

    public function runMaintenanceCompliance(int $tenantId): void
    {
        $schedules = PmSchedule::where('tenant_id', $tenantId)
            ->where('active', true)
            ->with('asset')
            ->get();

        $overdue = [];
        foreach ($schedules as $schedule) {
            $asset = $schedule->asset;
            if (!$asset) {
                continue;
            }

            $meterOverdue = null;
            if ($schedule->next_meter_reading !== null) {
                $latest = MeterReading::where('tenant_id', $tenantId)
                    ->where('asset_id', $schedule->asset_id)
                    ->where('meter_type', $schedule->meter_type)
                    ->orderByDesc('recorded_at')
                    ->first();

                if ($latest && $latest->value >= $schedule->next_meter_reading) {
                    $meterOverdue = round($latest->value - $schedule->next_meter_reading, 1);
                }
            }

            $daysOverdue = null;
            if ($schedule->next_due_at && now()->gt($schedule->next_due_at)) {
                $daysOverdue = $schedule->next_due_at->diffInDays(now());
            }

            if ($daysOverdue === null && $meterOverdue === null) {
                continue;
            }

            $riskScore = ($daysOverdue ?? 0) * 2 + ($meterOverdue ?? 0) / 10;
            $risk = $riskScore >= 14 ? 'High' : ($riskScore >= 6 ? 'Medium' : 'Low');

            $overdue[] = [
                'asset' => $asset,
                'days' => $daysOverdue,
                'meter' => $meterOverdue,
                'risk' => $risk,
                'score' => $riskScore,
            ];
        }

        if (!$overdue) {
            $this->recordNotification(
                $tenantId,
                'maintenance_compliance',
                'Maintenance Compliance Agent',
                'Maintenance Compliance',
                'No overdue assets detected.',
                'info',
                ['count' => 0],
                ['maintenance']
            );
            return;
        }

        usort($overdue, fn ($a, $b) => $b['score'] <=> $a['score']);
        $top = array_slice($overdue, 0, 8);

        $lines = [];
        foreach ($top as $row) {
            $label = $this->assetLabel($row['asset']);
            $overdueText = [];
            if ($row['days'] !== null) {
                $overdueText[] = $row['days'] . 'd overdue';
            }
            if ($row['meter'] !== null) {
                $unit = $row['asset']?->meter_type ?? $row['asset']?->meter_unit ?? 'hrs';
                $overdueText[] = $row['meter'] . $unit;
            }
            $lines[] = "{$label} | " . implode(', ', $overdueText) . " | Risk: {$row['risk']}";
        }

        $message = "Maintenance Compliance (14:00)\nOverdue assets:\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, $lines));

        $notification = $this->recordNotification(
            $tenantId,
            'maintenance_compliance',
            'Maintenance Compliance Agent',
            'Maintenance Compliance',
            $message,
            $this->severityFromRisk($top),
            ['overdue' => $lines, 'count' => count($overdue)],
            ['maintenance', 'management']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.maintenance_compliance',
            $this->severityFromRisk($top),
            'Maintenance Compliance',
            $message,
            [
                'details' => $lines,
                'title' => 'MAINTENANCE COMPLIANCE',
            ]
        );

        if ($notification && $notification->severity === 'alarm') {
            $this->maybeNotifyOps($tenantId, $notification, $lines);
        }
    }

    public function runFailurePrediction(int $tenantId): void
    {
        $windowStart = now()->subDays(30);

        $downtime = DowntimeLog::where('tenant_id', $tenantId)
            ->where('ended_at', '>=', $windowStart)
            ->selectRaw('asset_id, SUM(duration_minutes) as minutes')
            ->groupBy('asset_id')
            ->orderByDesc('minutes')
            ->with('asset')
            ->limit(5)
            ->get();

        $parts = WorkOrderPart::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $windowStart)
            ->selectRaw('asset_id, SUM(quantity * unit_cost) as spend')
            ->groupBy('asset_id')
            ->orderByDesc('spend')
            ->with('asset')
            ->limit(5)
            ->get();

        $fuelVariance = FuelLog::where('tenant_id', $tenantId)
            ->where('logged_at', '>=', $windowStart)
            ->whereNotNull('variance')
            ->orderByDesc('variance')
            ->with('asset')
            ->limit(5)
            ->get();

        $lines = [];
        $severity = 'warn';
        foreach ($downtime as $entry) {
            $hours = round(((float) $entry->minutes) / 60, 1);
            if ($hours >= 8) {
                $severity = 'alarm';
            }
            $lines[] = $this->assetLabel($entry->asset) . " downtime {$hours}h (30d)";
        }
        foreach ($parts as $entry) {
            if ((float) $entry->spend >= 5000) {
                $severity = 'alarm';
            }
            $lines[] = $this->assetLabel($entry->asset) . ' spares spend $' . number_format((float) $entry->spend, 2);
        }
        foreach ($fuelVariance as $entry) {
            if (abs((float) ($entry->variance ?? 0)) >= 20) {
                $severity = 'alarm';
            }
            $lines[] = $this->assetLabel($entry->asset) . ' fuel variance ' . ($entry->variance ?? 0);
        }

        if (!$lines) {
            $this->recordNotification(
                $tenantId,
                'failure_prediction',
                'Failure Prediction Agent',
                'Failure Prediction',
                'No abnormal trends detected in the last 30 days.',
                'info',
                ['count' => 0],
                ['maintenance', 'management']
            );
            return;
        }

        $recommendations = [
            'Schedule inspections for assets with repeated downtime.',
            'Review fuel usage variance for operator training or leaks.',
            'Adjust PM intervals for high spares spend assets.',
        ];

        $message = "Failure Prediction (Weekly)\nSignals:\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, array_slice($lines, 0, 8))) .
            "\nRecommendations:\n" . implode("\n", array_map(fn ($line) => '- ' . $line, $recommendations));

        $notification = $this->recordNotification(
            $tenantId,
            'failure_prediction',
            'Failure Prediction Agent',
            'Failure Prediction',
            $message,
            $severity,
            ['signals' => $lines, 'recommendations' => $recommendations],
            ['maintenance', 'management']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.failure_prediction',
            $severity,
            'Failure Prediction',
            $message,
            [
                'details' => array_slice($lines, 0, 8),
                'title' => 'FAILURE PREDICTION',
            ]
        );

        if ($notification && $notification->severity === 'alarm') {
            $this->maybeNotifyOps($tenantId, $notification, $lines);
        }
    }

    public function runKpiAnalyst(int $tenantId): void
    {
        $windowStart = now()->subDays(7)->startOfDay();
        $windowEnd = now()->endOfDay();

        $assetsCount = Asset::where('tenant_id', $tenantId)->count();
        $days = 7;
        $totalHours = $assetsCount * 24 * $days;

        $downtimeMinutes = DowntimeLog::where('tenant_id', $tenantId)
            ->whereBetween('ended_at', [$windowStart, $windowEnd])
            ->sum('duration_minutes');
        $downtimeHours = round(((float) $downtimeMinutes) / 60, 1);

        $runtimeHours = $this->sumRuntimeHours($tenantId, $windowStart, $windowEnd);
        $availability = $totalHours > 0 ? max(0, min(100, (1 - ($downtimeHours / $totalHours)) * 100)) : 0;
        $utilization = $totalHours > 0 ? max(0, min(100, ($runtimeHours / $totalHours) * 100)) : 0;

        $breakdowns = WorkOrder::where('tenant_id', $tenantId)
            ->whereBetween('created_at', [$windowStart, $windowEnd])
            ->where(function ($builder) {
                $builder->where('type', 'like', '%breakdown%')
                    ->orWhere('priority', 'high');
            })
            ->count();
        $mtbf = $breakdowns > 0 ? round($runtimeHours / $breakdowns, 1) : null;

        $fuelQuantity = FuelLog::where('tenant_id', $tenantId)
            ->whereBetween('logged_at', [$windowStart, $windowEnd])
            ->sum('quantity');
        $fuelEfficiency = $runtimeHours > 0 ? round(((float) $fuelQuantity) / $runtimeHours, 2) : null;

        $lines = [
            'Availability: ' . number_format($availability, 1) . '%',
            'Utilization: ' . number_format($utilization, 1) . '%',
            'MTBF: ' . ($mtbf !== null ? $mtbf . 'h' : 'N/A'),
            'Fuel efficiency: ' . ($fuelEfficiency !== null ? $fuelEfficiency . ' L/hr' : 'N/A'),
        ];

        $severity = 'info';
        if ($availability < 85 || $utilization < 60) {
            $severity = 'warn';
        }
        if ($availability < 75) {
            $severity = 'alarm';
        }

        $summary = "Weekly KPI Report (Week ending " . now()->toDateString() . ")\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, $lines)) .
            "\nRisks: " . ($downtimeHours > 0 ? 'Downtime ' . $downtimeHours . 'h this week.' : 'No major risks detected.') .
            "\nWins: " . ($availability >= 90 ? 'Availability above target.' : 'Track PM completion to lift availability.');

        $notification = $this->recordNotification(
            $tenantId,
            'kpi_analyst',
            'KPI Analyst Agent',
            'Weekly KPI Report',
            $summary,
            $severity,
            [
                'availability' => $availability,
                'utilization' => $utilization,
                'mtbf' => $mtbf,
                'fuel_efficiency' => $fuelEfficiency,
                'downtime_hours' => $downtimeHours,
            ],
            ['management']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.kpi_analyst',
            $severity,
            'Weekly KPI Report',
            $summary,
            [
                'details' => $lines,
                'title' => 'KPI REPORT',
            ]
        );

        if ($notification && $notification->severity === 'alarm') {
            $this->maybeNotifyOps($tenantId, $notification, $lines);
        }
    }

    private function sumRuntimeHours(int $tenantId, $start, $end): float
    {
        $readings = MeterReading::where('tenant_id', $tenantId)
            ->whereBetween('recorded_at', [$start, $end])
            ->orderBy('recorded_at')
            ->get()
            ->groupBy('asset_id');

        $total = 0.0;
        foreach ($readings as $assetReadings) {
            $first = $assetReadings->first();
            $last = $assetReadings->last();
            if ($first && $last) {
                $delta = (float) $last->value - (float) $first->value;
                if ($delta > 0) {
                    $total += $delta;
                }
            }
        }

        return $total;
    }

    private function recordNotification(
        int $tenantId,
        string $agentKey,
        string $agentName,
        string $title,
        string $summary,
        string $severity,
        array $payload,
        array $recipients
    ): AgentNotification {
        $prompt = config('ai_agents.prompts.' . $agentKey);

        return AgentNotification::create([
            'tenant_id' => $tenantId,
            'agent_key' => $agentKey,
            'agent_name' => $agentName,
            'title' => $title,
            'summary' => $summary,
            'severity' => $severity,
            'payload' => array_merge($payload, ['prompt' => $prompt]),
            'recipients' => $recipients,
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    private function sendTelegramEvent(
        int $tenantId,
        string $eventType,
        string $severity,
        string $title,
        string $message,
        array $payload
    ): void {
        $this->events->createEvent($tenantId, $eventType, $severity, array_merge($payload, [
            'details' => $payload['details'] ?? [],
            'title' => $title,
        ]), [
            'title' => $title,
            'message' => $message,
        ]);
    }

    private function maybeNotifyOps(int $tenantId, AgentNotification $notification, array $findings): void
    {
        $summary = "Ops Notifier\nCritical findings:\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, array_slice($findings, 0, 6)));

        app(\App\Services\OpsNotifierAgent::class)->queueFindingAlert($tenantId, $summary, [
            'source_agent' => $notification->agent_key,
            'findings' => $findings,
        ]);
    }

    private function severityFromRisk(array $rows): string
    {
        foreach ($rows as $row) {
            if (($row['risk'] ?? '') === 'High') {
                return 'alarm';
            }
        }

        return 'warn';
    }

    private function assetLabel(?Asset $asset): string
    {
        if (!$asset) {
            return '-';
        }

        $tag = $asset->asset_tag ?? '-';
        $name = $asset->name ?? null;
        return $name ? "{$tag} - {$name}" : $tag;
    }

    private function isThrottled(int $tenantId, string $agentKey, int $minutes): bool
    {
        $cacheKey = "agent:{$agentKey}:{$tenantId}";
        if (Cache::has($cacheKey)) {
            return true;
        }

        Cache::put($cacheKey, true, now()->addMinutes($minutes));
        return false;
    }
}
