<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class AiDocumentParserService
{
    public function parseText(string $text, ?string $docType = null): array
    {
        $instruction = $this->buildInstruction($docType);
        $messages = [
            ['role' => 'system', 'content' => $instruction],
            ['role' => 'user', 'content' => $text],
        ];

        return $this->requestCompletion($messages);
    }

    public function parseImage(string $base64, string $mimeType, ?string $docType = null): array
    {
        $instruction = $this->buildInstruction($docType);
        $messages = [[
            'role' => 'user',
            'content' => [
                ['type' => 'text', 'text' => $instruction],
                ['type' => 'image_url', 'image_url' => ['url' => "data:{$mimeType};base64,{$base64}"]],
            ],
        ]];

        return $this->requestCompletion($messages);
    }

    private function buildInstruction(?string $docType): string
    {
        $schema = config('ai_schema_registry.doc_types', []);
        $schemaText = [];
        foreach ($schema as $type => $definition) {
            $required = implode(', ', $definition['required'] ?? []);
            $optional = implode(', ', $definition['optional'] ?? []);
            $schemaText[] = "{$type}: required [{$required}] optional [{$optional}]";
        }

        $docHint = $docType ? "Preferred doc_type: {$docType}." : 'Detect doc_type based on content.';

        return "You are a document extraction assistant. {$docHint} " .
            "Return ONLY JSON with keys: doc_type, rows, warnings. " .
            "rows must be an array of objects with snake_case keys that match the schema registry. " .
            "If a field is missing, leave it null. " .
            "Schema registry:\n- " . implode("\n- ", $schemaText);
    }

    private function requestCompletion(array $messages): array
    {
        $key = config('services.openai.key');
        if (!$key) {
            return ['errors' => ['OpenAI API key missing.']];
        }

        $payload = [
            'model' => config('services.openai.model'),
            'messages' => $messages,
            'temperature' => 0.1,
            'max_tokens' => 800,
        ];

        $request = Http::withToken($key)->timeout((int) config('services.openai.timeout', 25));
        $verify = config('services.openai.verify_ssl', true);
        $verify = filter_var($verify, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $request = $request->withOptions([
            'verify' => $verify ?? true,
        ]);

        if (config('services.openai.organization')) {
            $request = $request->withHeaders([
                'OpenAI-Organization' => config('services.openai.organization'),
            ]);
        }
        if (config('services.openai.project')) {
            $request = $request->withHeaders([
                'OpenAI-Project' => config('services.openai.project'),
            ]);
        }

        try {
            $response = $request->post(config('services.openai.endpoint'), $payload);
        } catch (\Throwable $e) {
            return ['errors' => ['OpenAI request failed: ' . $e->getMessage()]];
        }

        if ($response->failed()) {
            return ['errors' => ['OpenAI request failed.']];
        }

        $content = data_get($response->json(), 'choices.0.message.content', '');
        $decoded = json_decode($content, true);

        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return $decoded;
        }

        return ['errors' => ['Failed to parse AI response.'], 'raw' => $content];
    }
}
