<?php

namespace App\Services;

use App\Models\AiInsight;
use App\Models\Asset;
use App\Models\WorkOrder;
use Illuminate\Support\Carbon;

class AiMaintenanceInsightsService
{
    public function generateTenantInsights(int $tenantId): void
    {
        $assets = Asset::where('tenant_id', $tenantId)->get();
        foreach ($assets as $asset) {
            $this->analyzeAsset($asset->id, $tenantId);
        }
    }

    public function analyzeAsset(int $assetId, int $tenantId): void
    {
        $asset = Asset::where('tenant_id', $tenantId)->find($assetId);
        if (!$asset) {
            return;
        }

        $windowStart = now()->subDays(90);

        $workOrders = WorkOrder::where('tenant_id', $tenantId)
            ->where('asset_id', $assetId)
            ->whereNotNull('completed_at')
            ->where('completed_at', '>=', $windowStart)
            ->get();

        $this->suggestRootCause($tenantId, $asset, $workOrders);
        $this->suggestPmInterval($tenantId, $asset, $workOrders, $windowStart);
    }

    protected function suggestRootCause(int $tenantId, Asset $asset, $workOrders): void
    {
        $grouped = $workOrders->groupBy(function ($order) {
            return $order->cause ?: $order->description;
        })->filter(function ($items, $key) {
            return $key !== null && $key !== '' && $items->count() >= 2;
        });

        if ($grouped->isEmpty()) {
            return;
        }

        $top = $grouped->sortByDesc(fn ($items) => $items->count())->first();
        $cause = $top->first()->cause ?: $top->first()->description;

        AiInsight::updateOrCreate(
            [
                'tenant_id' => $tenantId,
                'asset_id' => $asset->id,
                'type' => 'root_cause_suggestion',
            ],
            [
                'payload' => [
                    'asset' => $asset->name,
                    'cause' => $cause,
                    'occurrences' => $top->count(),
                    'window_days' => 90,
                ],
                'confidence' => 0.72,
            ]
        );
    }

    protected function suggestPmInterval(int $tenantId, Asset $asset, $workOrders, Carbon $windowStart): void
    {
        $correctiveCount = $workOrders->where('type', 'corrective')->count();
        $interval = $asset->maintenance_interval;

        if (!$interval) {
            return;
        }

        if ($correctiveCount >= 3) {
            $recommended = max(50, (int) round($interval * 0.8));
            $rationale = 'Repeated corrective work orders suggest reducing PM interval.';
        } elseif ($correctiveCount === 0) {
            $recommended = (int) round($interval * 1.1);
            $rationale = 'No corrective work orders detected; consider extending PM interval.';
        } else {
            return;
        }

        AiInsight::updateOrCreate(
            [
                'tenant_id' => $tenantId,
                'asset_id' => $asset->id,
                'type' => 'pm_interval_recommendation',
            ],
            [
                'payload' => [
                    'asset' => $asset->name,
                    'current_interval' => $interval,
                    'recommended_interval' => $recommended,
                    'window_start' => $windowStart->toDateString(),
                    'corrective_count' => $correctiveCount,
                    'rationale' => $rationale,
                ],
                'confidence' => 0.66,
            ]
        );
    }
}
