<?php

namespace App\Services;

use App\Models\AiReport;
use App\Models\Alarm;
use App\Models\Asset;
use App\Models\DowntimeLog;
use App\Models\InventoryItem;
use App\Models\Tenant;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use Illuminate\Support\Carbon;

class AiReportService
{
    public function generateWeeklyReports(): void
    {
        $end = Carbon::now()->startOfWeek();
        $start = $end->copy()->subWeek();

        Tenant::all()->each(function (Tenant $tenant) use ($start, $end) {
            app(AiMaintenanceInsightsService::class)->generateTenantInsights($tenant->id);
            $content = $this->buildWeeklySummary($tenant->id, $start, $end);

            AiReport::create([
                'tenant_id' => $tenant->id,
                'report_type' => 'weekly_kpi',
                'period_start' => $start->toDateString(),
                'period_end' => $end->toDateString(),
                'content' => $content,
            ]);
        });
    }

    protected function buildWeeklySummary(int $tenantId, Carbon $start, Carbon $end): string
    {
        $assets = Asset::where('tenant_id', $tenantId)->count();
        $openWorkOrders = WorkOrder::where('tenant_id', $tenantId)->where('status', 'open')->count();
        $openAlarms = Alarm::where('tenant_id', $tenantId)->where('status', 'open')->count();

        $periodMinutes = max(1, $start->diffInMinutes($end));
        $totalCapacityMinutes = max(1, $assets * $periodMinutes);
        $downtimeMinutes = DowntimeLog::where('tenant_id', $tenantId)
            ->whereBetween('started_at', [$start, $end])
            ->sum('duration_minutes');
        $availability = 1 - ($downtimeMinutes / $totalCapacityMinutes);

        $usageHours = (float) \App\Models\Telemetry::where('tenant_id', $tenantId)
            ->whereBetween('timestamp', [$start, $end])
            ->sum('usage_hours');
        $periodHours = max(1, $start->diffInHours($end));
        $utilization = $assets > 0 ? min(1, $usageHours / ($assets * $periodHours)) : 0;

        $downtimeReasons = DowntimeLog::select('downtime_reason_codes.label')
            ->join('downtime_reason_codes', 'downtime_reason_codes.id', '=', 'downtime_logs.reason_code_id')
            ->where('downtime_logs.tenant_id', $tenantId)
            ->whereBetween('downtime_logs.started_at', [$start, $end])
            ->groupBy('downtime_reason_codes.label')
            ->orderByRaw('count(*) desc')
            ->limit(3)
            ->pluck('downtime_reason_codes.label')
            ->all();

        $costDrivers = WorkOrderCost::select('type')
            ->where('tenant_id', $tenantId)
            ->whereBetween('created_at', [$start, $end])
            ->groupBy('type')
            ->orderByRaw('sum(total_cost) desc')
            ->limit(3)
            ->pluck('type')
            ->all();

        $lowStock = InventoryItem::where('tenant_id', $tenantId)
            ->whereNotNull('reorder_point')
            ->whereColumn('quantity', '<=', 'reorder_point')
            ->count();

        $actions = [];
        if ($openAlarms > 0) {
            $actions[] = 'Resolve ' . $openAlarms . ' open alarms.';
        }
        if ($lowStock > 0) {
            $actions[] = 'Reorder ' . $lowStock . ' low-stock items.';
        }
        if ($openWorkOrders > 0) {
            $actions[] = 'Clear ' . $openWorkOrders . ' open work orders.';
        }
        if (empty($actions)) {
            $actions[] = 'Maintain current operating cadence.';
        }

        return implode("\n", [
            'Weekly Ops Report',
            'Period: ' . $start->toDateString() . ' to ' . $end->toDateString(),
            'Availability: ' . round($availability * 100, 1) . '%',
            'Utilization: ' . round($utilization * 100, 1) . '%',
            'Top downtime reasons: ' . (empty($downtimeReasons) ? 'None' : implode(', ', $downtimeReasons)),
            'Cost drivers: ' . (empty($costDrivers) ? 'None' : implode(', ', $costDrivers)),
            'Next week actions: ' . implode(' ', $actions),
        ]);
    }
}
