<?php

namespace App\Services;

use App\Models\Alarm;
use App\Models\AlarmRule;
use App\Models\WorkOrder;
use Illuminate\Support\Carbon;

class AlarmService
{
    public function trigger(AlarmRule $rule, array $context): Alarm
    {
        $alarm = Alarm::create([
            'tenant_id' => $rule->tenant_id,
            'asset_id' => $context['asset_id'] ?? null,
            'rule_id' => $rule->id,
            'severity' => $rule->severity ?? 'warning',
            'status' => 'open',
            'message' => $this->buildMessage($rule, $context),
            'triggered_at' => $context['occurred_at'] ?? Carbon::now(),
        ]);

        if (!empty($rule->actions['auto_work_order'])) {
            $workOrder = WorkOrder::create([
                'tenant_id' => $rule->tenant_id,
                'asset_id' => $context['asset_id'] ?? null,
                'alarm_id' => $alarm->id,
                'type' => 'corrective',
                'status' => 'open',
                'priority' => $rule->severity === 'critical' ? 'high' : 'medium',
                'description' => $alarm->message,
                'auto_generated' => true,
            ]);

            $alarm->update(['work_order_id' => $workOrder->id]);
        }

        app(NotificationService::class)->dispatchEvent($rule->tenant_id, 'alarm.triggered', [
            'alarm_id' => $alarm->id,
            'asset_id' => $alarm->asset_id,
            'severity' => $alarm->severity,
            'message' => $alarm->message,
        ]);

        return $alarm;
    }

    protected function buildMessage(AlarmRule $rule, array $context): string
    {
        $metric = $context['metric'] ?? 'metric';
        $value = $context['value'] ?? 'n/a';
        return sprintf('%s threshold breached: %s = %s', $rule->name, $metric, $value);
    }
}
