<?php

namespace App\Services;

use App\Models\Asset;

class AssetMatcherService
{
    public function findMatches(int $tenantId, ?string $assetTag, ?string $serialNumber, ?string $name): array
    {
        $matches = [];

        if ($assetTag) {
            $asset = Asset::where('tenant_id', $tenantId)
                ->where('asset_tag', $assetTag)
                ->first();
            if ($asset) {
                return [[
                    'asset_id' => $asset->id,
                    'asset_tag' => $asset->asset_tag,
                    'name' => $asset->name,
                    'confidence' => 1.0,
                    'match' => 'asset_tag',
                ]];
            }
        }

        if ($serialNumber) {
            $asset = Asset::where('tenant_id', $tenantId)
                ->where('serial_number', $serialNumber)
                ->first();
            if ($asset) {
                return [[
                    'asset_id' => $asset->id,
                    'asset_tag' => $asset->asset_tag,
                    'name' => $asset->name,
                    'confidence' => 0.92,
                    'match' => 'serial_number',
                ]];
            }
        }

        if ($name) {
            $candidates = Asset::where('tenant_id', $tenantId)
                ->where('name', 'like', '%' . $name . '%')
                ->limit(5)
                ->get();

            foreach ($candidates as $asset) {
                $confidence = $this->similarity($name, $asset->name);
                $matches[] = [
                    'asset_id' => $asset->id,
                    'asset_tag' => $asset->asset_tag,
                    'name' => $asset->name,
                    'confidence' => $confidence,
                    'match' => 'name',
                ];
            }
        }

        usort($matches, fn ($a, $b) => $b['confidence'] <=> $a['confidence']);
        return array_slice($matches, 0, 3);
    }

    private function similarity(string $needle, string $haystack): float
    {
        $needle = strtolower($needle);
        $haystack = strtolower($haystack);

        similar_text($needle, $haystack, $percent);
        return round($percent / 100, 2);
    }
}
