<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\AssetSpare;
use App\Models\ImportBatch;
use App\Models\ImportRow;
use App\Models\InventoryItem;
use App\Models\InventoryLocation;
use App\Models\Part;
use App\Models\Site;
use App\Models\Vehicle;
use App\Support\XlsxReader;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class AssetRegisterImportService
{
    public function import(string $path, int $tenantId, ?int $userId = null, ?string $defaultLocation = null, ?string $sheetName = null): array
    {
        if (!is_file($path)) {
            return ['errors' => 1, 'message' => 'Asset register file not found.'];
        }

        $batch = ImportBatch::create([
            'tenant_id' => $tenantId,
            'created_by' => $userId,
            'status' => 'processing',
            'file_path' => $path,
            'file_type' => pathinfo($path, PATHINFO_EXTENSION),
            'import_type' => 'asset_register',
        ]);

        $summary = [
            'assets_created' => 0,
            'assets_updated' => 0,
            'assets_duplicates' => 0,
            'parts_created' => 0,
            'parts_updated' => 0,
            'asset_spares_created' => 0,
            'asset_spares_updated' => 0,
            'asset_spares_skipped' => 0,
            'errors' => 0,
        ];

        $result = DB::transaction(function () use ($path, $tenantId, $batch, $defaultLocation, $sheetName, &$summary) {
            $assetsSheet = XlsxReader::readSheetRows($path, $sheetName ?: 'Asset Register');
            if (!$assetsSheet) {
                $assetsSheet = XlsxReader::readSheetRows($path, null);
            }
            $sparesSheet = XlsxReader::readSheetRows($path, 'Spares Register');
            $linksSheet = XlsxReader::readSheetRows($path, 'Asset-Spare Links');

            $assetMap = $this->importAssets($assetsSheet, $tenantId, $batch, $summary, $defaultLocation);
            $partMap = $this->importSpares($sparesSheet, $tenantId, $summary);

            if ($linksSheet) {
                $this->importAssetSpareLinks($linksSheet, $tenantId, $assetMap, $partMap, $summary);
            }

            return $summary;
        });

        $batch->update([
            'status' => 'completed',
            'summary' => $result,
            'ai_summary' => sprintf(
                'Asset register import completed. Assets: %d created, %d updated. Parts: %d created, %d updated.',
                $summary['assets_created'],
                $summary['assets_updated'],
                $summary['parts_created'],
                $summary['parts_updated']
            ),
            'processed_at' => now(),
        ]);

        return $summary;
    }

    private function importAssets(array $rows, int $tenantId, ImportBatch $batch, array &$summary, ?string $defaultLocation = null): array
    {
        $assetMap = [];

        foreach ($rows as $row) {
            $assetName = $this->stringValue($row['Asset Name'] ?? null)
                ?? $this->stringValue($row['Asset Description'] ?? null);
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $vehicleId = $this->stringValue($row['Vehicle ID'] ?? null);
            if (!$assetName && !$assetId && !$vehicleId) {
                continue;
            }

            $locationName = $defaultLocation
                ?? $this->stringValue($row['Location'] ?? null)
                ?? $this->stringValue($row['Allocation'] ?? null);
            $siteId = $locationName ? $this->resolveSite($tenantId, $locationName) : null;
            $assetTag = $assetId ?: ($vehicleId ?: $assetName);

            $make = $this->stringValue($row['Make'] ?? null);
            [$fallbackMake, $fallbackModel] = $this->parseMakeModel($this->stringValue($row['Make/Model/Spec'] ?? null));
            $model = $this->stringValue($row['Type'] ?? null) ?? $fallbackModel;
            if (!$make) {
                $make = $fallbackMake;
            }
            $status = $this->normalizeStatus($this->stringValue($row['Status'] ?? null));
            $hasPlate = $this->stringValue($row['Plate'] ?? null);
            $assetGroup = $this->stringValue($row['Asset Group'] ?? null);
            $assetType = $hasPlate ? 'vehicle' : $this->inferAssetType($assetGroup);

            $data = [
                'tenant_id' => $tenantId,
                'site_id' => $siteId,
                'asset_tag' => $assetTag,
                'name' => $assetName ?: $assetTag,
                'category' => $assetGroup,
                'asset_type' => $assetType,
                'location' => $locationName,
                'make' => $make,
                'model' => $model,
                'serial_number' => $this->stringValue($row['Serial Number'] ?? null) ?? $vehicleId,
                'status' => $status ?: 'active',
                'lifecycle_status' => $status ?: 'active',
                'fit_for_purpose' => $this->boolValue($row['Fit for Purpose'] ?? null),
                'plant_restrictions' => $this->stringValue($row['Plant Restrictions'] ?? null),
                'comments' => $this->stringValue($row['Comments'] ?? null) ?? $this->stringValue($row['Allocation'] ?? null),
                'service_frequency' => $this->stringValue($row['Service Frequency'] ?? null),
                'maintenance_interval' => $this->numericValue($row['Service Frequency'] ?? null),
                'next_review_at' => $this->dateValue($row['Next Review Date'] ?? null),
                'source' => $this->stringValue($row['Source'] ?? null),
                'spare_tags' => $this->stringValue($row['Spare Tags'] ?? null),
                'top_spares' => $this->stringValue($row['Top Spares (indicative)'] ?? null),
                'yearly_budget' => $this->numericValue($row['Yearly Budget (USD)'] ?? null),
                'daily_budget' => $this->numericValue($row['Daily Budget (USD)'] ?? null),
                'weekly_budget' => $this->numericValue($row['Weekly Budget (USD)'] ?? null),
                'monthly_budget' => $this->numericValue($row['Monthly Budget (USD)'] ?? null),
                'purchase_date' => $this->dateValue($row['Date of Purchase'] ?? null),
                'current_value' => $this->numericValue($row['Purchase Cost (USD)'] ?? null),
                'description' => $this->stringValue($row['Asset Description'] ?? null) ?? $this->stringValue($row['Comments'] ?? null),
            ];

            $asset = Asset::where('tenant_id', $tenantId)
                ->when($assetTag, fn ($query) => $query->where('asset_tag', $assetTag))
                ->first();

            if ($asset) {
                $asset->fill($data);
                if ($asset->isDirty()) {
                    $asset->save();
                    $summary['assets_updated']++;
                    $rowStatus = 'updated';
                } else {
                    $summary['assets_duplicates']++;
                    $rowStatus = 'duplicates';
                }
            } else {
                $asset = Asset::create($data);
                $summary['assets_created']++;
                $rowStatus = 'created';
            }

            ImportRow::create([
                'tenant_id' => $tenantId,
                'batch_id' => $batch->id,
                'raw' => $row,
                'mapped' => $data,
                'status' => $rowStatus,
            ]);

            $this->syncVehicle($asset, $row);

            if ($asset->asset_tag) {
                $assetMap[strtolower($asset->asset_tag)] = $asset;
            }
            $assetMap[strtolower($asset->name)] = $asset;
        }

        return $assetMap;
    }

    private function importSpares(array $rows, int $tenantId, array &$summary): array
    {
        $partMap = [];
        if (!$rows) {
            return $partMap;
        }

        $location = InventoryLocation::firstOrCreate(
            ['tenant_id' => $tenantId, 'name' => 'Main Store'],
            ['code' => 'MAIN', 'site_id' => null]
        );

        foreach ($rows as $row) {
            $name = $this->stringValue($row['Spare Part'] ?? null);
            if (!$name) {
                continue;
            }

            $partData = [
                'tenant_id' => $tenantId,
                'name' => $name,
                'sku' => $this->makeSku($name),
                'category' => null,
                'stage_of_use' => $this->stringValue($row['Stage of Use'] ?? null),
                'replacement_frequency' => $this->stringValue($row['Replacement Frequency'] ?? null),
                'estimated_replacements_per_year' => $this->numericValue($row['Estimated Replacements per Year'] ?? null),
                'cost_per_year' => $this->numericValue($row['Cost per Year (USD)'] ?? null),
                'unit' => 'pcs',
            ];

            $part = Part::where('tenant_id', $tenantId)
                ->where('name', $name)
                ->first();

            if ($part) {
                $part->fill($partData);
                if ($part->isDirty()) {
                    $part->save();
                    $summary['parts_updated']++;
                }
            } else {
                $part = Part::create($partData);
                $summary['parts_created']++;
            }

            $quantity = $this->numericValue($row['Quantity'] ?? null);
            $unitCost = $this->numericValue($row['Unit Cost (USD)'] ?? null);
            $totalCost = $this->numericValue($row['Total Cost (USD)'] ?? null);
            if ($totalCost === null && $quantity !== null && $unitCost !== null) {
                $totalCost = $quantity * $unitCost;
            }

            InventoryItem::updateOrCreate(
                [
                    'tenant_id' => $tenantId,
                    'part_id' => $part->id,
                    'location_id' => $location->id,
                ],
                [
                    'quantity' => $quantity ?? 0,
                    'unit_cost' => $unitCost,
                    'total_cost' => $totalCost,
                ]
            );

            $partMap[strtolower($name)] = $part;
        }

        return $partMap;
    }

    private function importAssetSpareLinks(array $rows, int $tenantId, array $assetMap, array $partMap, array &$summary): void
    {
        foreach ($rows as $row) {
            $assetKeyValue = $this->stringValue($row['Asset ID'] ?? null) ?: $this->stringValue($row['Asset Name'] ?? null);
            $partKeyValue = $this->stringValue($row['Spare Part'] ?? null);
            $assetKey = $assetKeyValue ? strtolower($assetKeyValue) : '';
            $partKey = $partKeyValue ? strtolower($partKeyValue) : '';

            if (!$assetKey || !$partKey || !isset($assetMap[$assetKey]) || !isset($partMap[$partKey])) {
                $summary['asset_spares_skipped']++;
                continue;
            }

            $asset = $assetMap[$assetKey];
            $part = $partMap[$partKey];

            $data = [
                'tenant_id' => $tenantId,
                'asset_id' => $asset->id,
                'part_id' => $part->id,
                'quantity' => $this->numericValue($row['Quantity'] ?? null),
                'unit_cost' => $this->numericValue($row['Unit Cost (USD)'] ?? null),
                'replacement_frequency' => $this->stringValue($row['Replacement Frequency'] ?? null),
                'estimated_replacements_per_year' => $this->numericValue($row['Estimated Replacements per Year'] ?? null),
                'cost_per_year' => $this->numericValue($row['Cost per Year (USD)'] ?? null),
            ];

            $link = AssetSpare::where('tenant_id', $tenantId)
                ->where('asset_id', $asset->id)
                ->where('part_id', $part->id)
                ->first();

            if ($link) {
                $link->fill($data);
                if ($link->isDirty()) {
                    $link->save();
                    $summary['asset_spares_updated']++;
                }
            } else {
                AssetSpare::create($data);
                $summary['asset_spares_created']++;
            }
        }
    }

    private function syncVehicle(Asset $asset, array $row): void
    {
        $plate = $this->stringValue($row['Plate'] ?? null);
        if (!$plate) {
            return;
        }

        $vehicle = Vehicle::where('tenant_id', $asset->tenant_id)
            ->where(function ($query) use ($asset, $plate) {
                $query->where('asset_id', $asset->id)
                    ->orWhere('license_plate', $plate);
            })
            ->first();

        $payload = [
            'tenant_id' => $asset->tenant_id,
            'asset_id' => $asset->id,
            'site_id' => $asset->site_id,
            'name' => $asset->name,
            'type' => $this->stringValue($row['Type'] ?? null) ?? $asset->category,
            'license_plate' => $plate,
            'status' => 'active',
        ];

        if ($vehicle) {
            $vehicle->fill($payload);
            $vehicle->save();
            return;
        }

        Vehicle::create($payload);
    }

    private function resolveSite(int $tenantId, string $locationName): int
    {
        $site = Site::firstOrCreate(
            ['tenant_id' => $tenantId, 'name' => $locationName],
            [
                'code' => $this->siteCode($locationName),
                'type' => 'site',
                'address' => null,
                'active' => true,
            ]
        );

        return $site->id;
    }

    private function siteCode(string $name): string
    {
        $code = strtoupper(Str::substr(Str::slug($name, ''), 0, 6));
        return $code ?: 'SITE';
    }

    private function inferAssetType(?string $assetGroup): string
    {
        $group = strtolower((string) $assetGroup);
        if (str_contains($group, 'truck') || str_contains($group, 'vehicle') || str_contains($group, 'fleet')) {
            return 'vehicle';
        }
        if (str_contains($group, 'mobile')) {
            return 'mobile';
        }
        if (str_contains($group, 'stationary') || str_contains($group, 'fixed') || str_contains($group, 'plant')) {
            return 'fixed';
        }
        return 'asset';
    }

    private function parseMakeModel(?string $value): array
    {
        $value = $this->stringValue($value);
        if (!$value) {
            return [null, null];
        }

        $parts = preg_split('/\s+/', $value, 2);
        $make = $parts[0] ?? null;
        $model = $parts[1] ?? null;

        return [$make, $model];
    }

    private function normalizeStatus(?string $status): ?string
    {
        $value = strtolower((string) $status);
        if ($value === '') {
            return null;
        }

        if (str_contains($value, 'current') || str_contains($value, 'active') || str_contains($value, 'running')) {
            return 'active';
        }
        if (str_contains($value, 'standby') || str_contains($value, 'idle')) {
            return 'standby';
        }
        if (str_contains($value, 'down') || str_contains($value, 'break') || str_contains($value, 'repair')) {
            return 'down';
        }
        if (str_contains($value, 'retire')) {
            return 'retired';
        }

        return $value;
    }

    private function stringValue(mixed $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        return $value === '' ? null : $value;
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    private function boolValue(mixed $value): ?bool
    {
        if ($value === null || $value === '') {
            return null;
        }

        $value = strtolower(trim((string) $value));
        return in_array($value, ['yes', 'y', 'true', '1'], true);
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateString();
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function makeSku(string $name): string
    {
        $sku = strtoupper(Str::slug($name, ''));
        $sku = substr($sku, 0, 12);

        return $sku ?: strtoupper(Str::random(6));
    }
}
