<?php

namespace App\Services;

use App\Models\AiToolLog;
use App\Models\Asset;
use App\Models\DocumentIntakeBatch;
use App\Models\DocumentIntakeRow;
use App\Models\FuelLog;
use App\Models\GoodsReceipt;
use App\Models\GoodsReceiptItem;
use App\Models\InventoryItem;
use App\Models\MeterReading;
use App\Models\Part;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestItem;
use App\Models\NotificationLog;
use App\Models\PurchaseOrder;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Models\Maintenance;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class DocumentIntakeApplyService
{
    public function applyBatch(DocumentIntakeBatch $batch, ?int $userId = null): array
    {
        $summary = [
            'applied' => 0,
            'skipped' => 0,
            'errors' => 0,
        ];

        $rows = $batch->rows()->where('status', 'ready')->get();
        foreach ($rows as $row) {
            $result = match ($batch->doc_type) {
                'purchase_request' => $this->applyPurchaseRequestRow($batch, $row, $userId),
                'asset_register' => $this->applyAssetRow($batch, $row, $userId),
                'service_sheet' => $this->applyServiceSheetRow($batch, $row, $userId),
                'invoice' => $this->applyInvoiceRow($batch, $row, $userId),
                'fuel_sheet' => $this->applyFuelRow($batch, $row, $userId),
                'meter_readings' => $this->applyMeterReadingRow($batch, $row, $userId),
                'grn' => $this->applyGoodsReceiptRow($batch, $row, $userId),
                default => ['status' => 'skipped', 'message' => 'Unsupported doc type'],
            };

            if ($result['status'] === 'applied') {
                $summary['applied']++;
                $row->update(['status' => 'applied']);
            } elseif ($result['status'] === 'skipped') {
                $summary['skipped']++;
            } else {
                $summary['errors']++;
                $row->update(['status' => 'error']);
            }
        }

        $batch->update([
            'status' => 'applied',
            'summary' => array_merge($batch->summary ?? [], $summary),
        ]);

        return $summary;
    }

    private function applyPurchaseRequestRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $request = PurchaseRequest::firstOrCreate(
            [
                'tenant_id' => $batch->tenant_id,
                'request_code' => 'PR-AI-' . $batch->id,
            ],
            [
                'site_id' => null,
                'title' => 'Chatbot Intake',
                'status' => 'draft',
                'priority' => 'medium',
                'currency' => 'USD',
                'notes' => 'AI intake from chat',
                'source_file' => $batch->source_upload_ids ? implode(',', $batch->source_upload_ids) : null,
                'requested_by' => $userId,
            ]
        );

        $part = $this->findPart($batch->tenant_id, $data['sku'] ?? null, $data['item_ordered'] ?? null);

        PurchaseRequestItem::create([
            'tenant_id' => $batch->tenant_id,
            'purchase_request_id' => $request->id,
            'line_number' => $row->row_index,
            'item_name' => $data['item_ordered'] ?? null,
            'description' => $data['description'] ?? null,
            'quantity' => $this->numericValue($data['quantity'] ?? null),
            'unit' => $data['unit'] ?? null,
            'part_id' => $part?->id,
            'sku' => $part?->sku ?? ($data['sku'] ?? null),
            'est_unit_cost' => $this->numericValue($data['est_unit_cost'] ?? null),
            'purpose_cost_center' => $data['purpose_cost_centre'] ?? null,
            'date_initiated' => $this->dateValue($data['date_initiated'] ?? null),
            'supplier_name' => $data['suppliers'] ?? null,
            'quote_number' => $data['quote_number'] ?? null,
            'quote_amount_usd' => $this->numericValue($data['quote_amount_usd'] ?? null),
            'quote_amount_zwl' => $this->numericValue($data['quote_amount_zwl'] ?? null),
            'preferred_vendor' => $data['preferred_vendor'] ?? null,
            'selected_supplier' => $data['selected_supplier'] ?? null,
            'comments' => $data['comments'] ?? null,
        ]);

        $this->logTool($batch, $userId, 'purchase_request', 'create', $row->data, ['request_id' => $request->id]);

        return ['status' => 'applied'];
    }

    private function applyAssetRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $assetTag = $data['asset_tag'] ?? ($data['asset_id'] ?? null);
        $name = $data['name'] ?? ($data['asset_name'] ?? null);

        if (!$assetTag && !$name) {
            return ['status' => 'error', 'message' => 'Missing asset identifier'];
        }

        $asset = Asset::updateOrCreate(
            [
                'tenant_id' => $batch->tenant_id,
                'asset_tag' => $assetTag,
            ],
            [
                'name' => $name ?: $assetTag,
                'make' => $data['make'] ?? null,
                'model' => $data['model'] ?? null,
                'category' => $data['category'] ?? null,
                'location' => $data['location'] ?? null,
                'serial_number' => $data['serial_number'] ?? null,
                'purchase_date' => $this->dateValue($data['purchase_date'] ?? null),
                'current_value' => $this->numericValue($data['current_value'] ?? null),
                'status' => 'active',
                'lifecycle_status' => 'active',
            ]
        );

        $this->logTool($batch, $userId, 'asset', 'create_or_update', $row->data, ['asset_id' => $asset->id]);

        return ['status' => 'applied'];
    }

    private function applyServiceSheetRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $asset = $this->resolveAsset($batch->tenant_id, $data);
        if (!$asset) {
            return ['status' => 'error', 'message' => 'Asset not found'];
        }

        $workOrder = $this->findOrCreateWorkOrder($batch->tenant_id, $asset->id, $data['work_order_ref'] ?? null, $data['description'] ?? 'Service sheet');

        Maintenance::create([
            'tenant_id' => $batch->tenant_id,
            'asset_id' => $asset->id,
            'scheduled_date' => now()->subDays(7),
            'completed_date' => $this->dateValue($data['completed_at'] ?? null) ?? now(),
            'status' => 'completed',
            'description' => $data['description'] ?? null,
            'usage_hours' => $this->numericValue($data['meter_reading'] ?? null),
            'meter_reading' => $this->numericValue($data['meter_reading'] ?? null),
        ]);

        $this->applyCosts($batch->tenant_id, $workOrder->id, $data);

        $this->logTool($batch, $userId, 'service_sheet', 'apply', $row->data, ['work_order_id' => $workOrder->id]);

        return ['status' => 'applied'];
    }

    private function applyInvoiceRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $asset = $this->resolveAsset($batch->tenant_id, $data);
        $workOrder = $this->findOrCreateWorkOrder($batch->tenant_id, $asset?->id, $data['work_order_ref'] ?? null, $data['description'] ?? 'Invoice');

        if (!$workOrder) {
            return ['status' => 'error', 'message' => 'Work order not found'];
        }

        $amount = $this->numericValue($data['amount'] ?? null);
        WorkOrderCost::create([
            'tenant_id' => $batch->tenant_id,
            'work_order_id' => $workOrder->id,
            'type' => $data['cost_type'] ?? 'contractor',
            'description' => $data['description'] ?? ($data['invoice_number'] ?? 'Invoice'),
            'quantity' => 1,
            'unit_cost' => $amount,
            'total_cost' => $amount,
        ]);

        $this->logTool($batch, $userId, 'invoice', 'apply', $row->data, ['work_order_id' => $workOrder->id]);

        return ['status' => 'applied'];
    }

    private function applyFuelRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $asset = $this->resolveAsset($batch->tenant_id, $data);
        if (!$asset) {
            return ['status' => 'error', 'message' => 'Asset not found'];
        }

        FuelLog::create([
            'tenant_id' => $batch->tenant_id,
            'asset_id' => $asset->id,
            'source' => $data['source'] ?? null,
            'fuel_type' => $data['fuel_type'] ?? null,
            'quantity' => $this->numericValue($data['quantity'] ?? null),
            'unit_cost' => $this->numericValue($data['unit_cost'] ?? null),
            'total_cost' => $this->numericValue($data['total_cost'] ?? null),
            'logged_at' => $this->dateValue($data['logged_at'] ?? null),
            'notes' => 'AI intake',
        ]);

        $this->queueNotification($batch->tenant_id, 'Fuel log ingested for ' . $asset->name);

        $this->logTool($batch, $userId, 'fuel_log', 'create', $row->data, ['asset_id' => $asset->id]);

        return ['status' => 'applied'];
    }

    private function applyMeterReadingRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $asset = $this->resolveAsset($batch->tenant_id, $data);
        if (!$asset) {
            return ['status' => 'error', 'message' => 'Asset not found'];
        }

        MeterReading::create([
            'tenant_id' => $batch->tenant_id,
            'asset_id' => $asset->id,
            'meter_type' => $data['meter_type'] ?? null,
            'value' => $this->numericValue($data['reading'] ?? null),
            'source' => 'ai_intake',
            'recorded_at' => $this->dateValue($data['reading_at'] ?? null) ?? now(),
        ]);

        $this->queueNotification($batch->tenant_id, 'Meter reading ingested for ' . $asset->name);

        $this->logTool($batch, $userId, 'meter_reading', 'create', $row->data, ['asset_id' => $asset->id]);

        return ['status' => 'applied'];
    }

    private function applyGoodsReceiptRow(DocumentIntakeBatch $batch, DocumentIntakeRow $row, ?int $userId): array
    {
        $data = $row->data ?? [];
        $reference = $data['reference'] ?? ($data['grn_number'] ?? null);
        if (!$reference) {
            return ['status' => 'error', 'message' => 'Missing GRN reference'];
        }

        $receipt = GoodsReceipt::firstOrCreate(
            [
                'tenant_id' => $batch->tenant_id,
                'reference' => $reference,
            ],
            [
                'purchase_order_id' => $this->resolvePurchaseOrderId($batch->tenant_id, $data),
                'received_at' => $this->dateTimeValue($data['received_at'] ?? null) ?? now(),
                'received_by' => $userId,
                'notes' => 'AI intake',
            ]
        );

        $part = $this->findPart($batch->tenant_id, $data['sku'] ?? null, $data['description'] ?? null);
        $quantity = $this->numericValue($data['quantity'] ?? null);

        GoodsReceiptItem::create([
            'tenant_id' => $batch->tenant_id,
            'goods_receipt_id' => $receipt->id,
            'part_id' => $part?->id,
            'sku' => $part?->sku ?? ($data['sku'] ?? null),
            'description' => $data['description'] ?? $data['item'] ?? null,
            'quantity' => $quantity,
            'unit' => $data['unit'] ?? null,
            'unit_cost' => $this->numericValue($data['unit_cost'] ?? null),
        ]);

        if ($part && $quantity !== null) {
            $inventory = InventoryItem::firstOrCreate(
                [
                    'tenant_id' => $batch->tenant_id,
                    'part_id' => $part->id,
                ],
                [
                    'quantity' => 0,
                ]
            );
            $inventory->update(['quantity' => (float) $inventory->quantity + (float) $quantity]);
        }

        $this->logTool($batch, $userId, 'grn', 'create', $row->data, ['receipt_id' => $receipt->id]);

        return ['status' => 'applied'];
    }

    private function resolveAsset(int $tenantId, array $data): ?Asset
    {
        $assetTag = $data['asset_tag'] ?? null;
        if ($assetTag) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetTag)->first();
            if ($asset) {
                return $asset;
            }
        }

        $assetName = $data['asset_name'] ?? $data['name'] ?? null;
        if ($assetName) {
            return Asset::where('tenant_id', $tenantId)->where('name', $assetName)->first();
        }

        return null;
    }

    private function findOrCreateWorkOrder(int $tenantId, ?int $assetId, ?string $reference, string $description): ?WorkOrder
    {
        if ($reference) {
            $existing = WorkOrder::where('tenant_id', $tenantId)->where('reference_code', $reference)->first();
            if ($existing) {
                return $existing;
            }
        }

        if (!$assetId) {
            return null;
        }

        $reference = $reference ?: strtoupper('WO-AI-' . Str::random(6));

        return WorkOrder::create([
            'tenant_id' => $tenantId,
            'asset_id' => $assetId,
            'reference_code' => $reference,
            'source' => 'ai_intake',
            'type' => 'corrective',
            'status' => 'completed',
            'description' => $description,
            'completed_at' => now(),
        ]);
    }

    private function applyCosts(int $tenantId, int $workOrderId, array $data): void
    {
        $costs = [
            'labor_cost' => 'labor',
            'parts_cost' => 'parts',
            'contractor_cost' => 'contractor',
        ];

        foreach ($costs as $field => $type) {
            $amount = $this->numericValue($data[$field] ?? null);
            if ($amount === null) {
                continue;
            }

            WorkOrderCost::create([
                'tenant_id' => $tenantId,
                'work_order_id' => $workOrderId,
                'type' => $type,
                'description' => ucfirst($type) . ' from intake',
                'quantity' => 1,
                'unit_cost' => $amount,
                'total_cost' => $amount,
            ]);
        }
    }

    private function logTool(DocumentIntakeBatch $batch, ?int $userId, string $tool, string $action, array $input, array $output): void
    {
        AiToolLog::create([
            'tenant_id' => $batch->tenant_id,
            'user_id' => $userId,
            'conversation_id' => $batch->conversation_id,
            'upload_id' => null,
            'tool_name' => $tool,
            'action' => $action,
            'target_type' => $tool,
            'target_id' => $output['id'] ?? ($output['asset_id'] ?? null),
            'input' => $input,
            'output' => $output,
            'status' => 'success',
            'source' => 'ai_intake',
        ]);
    }

    private function queueNotification(int $tenantId, string $message): void
    {
        NotificationLog::create([
            'tenant_id' => $tenantId,
            'channel' => 'in_app',
            'recipient' => 'system',
            'payload' => ['message' => $message],
            'status' => 'queued',
            'sent_at' => now(),
        ]);
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    private function dateTimeValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateTimeString();
        }

        try {
            return Carbon::parse($value)->toDateTimeString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function resolvePurchaseOrderId(int $tenantId, array $data): ?int
    {
        $poNumber = $data['po_number'] ?? $data['po'] ?? null;
        if (!$poNumber) {
            return null;
        }

        $order = PurchaseOrder::where('tenant_id', $tenantId)
            ->where('po_number', $poNumber)
            ->first();

        return $order?->id;
    }

    private function findPart(int $tenantId, ?string $sku, ?string $name): ?Part
    {
        $sku = $sku ? trim($sku) : null;
        if ($sku) {
            $part = Part::where('tenant_id', $tenantId)->where('sku', $sku)->first();
            if ($part) {
                return $part;
            }
        }

        $name = $name ? trim($name) : null;
        if ($name) {
            return Part::where('tenant_id', $tenantId)
                ->where('name', 'like', '%' . $name . '%')
                ->first();
        }

        return null;
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateString();
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }
}
