<?php

namespace App\Services;

use App\Models\ChatUpload;
use App\Models\DocumentIntakeBatch;
use App\Models\DocumentIntakeRow;
use App\Support\XlsxReader;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;

class DocumentIntakeService
{
    public function intakeUploads(array $uploads, int $tenantId, ?int $conversationId, ?int $userId = null): DocumentIntakeBatch
    {
        $batch = DocumentIntakeBatch::create([
            'tenant_id' => $tenantId,
            'conversation_id' => $conversationId,
            'user_id' => $userId,
            'status' => 'review',
            'schema_version' => config('ai_schema_registry.version', 'v1'),
            'source_upload_ids' => collect($uploads)->pluck('id')->all(),
        ]);

        $summary = [
            'rows' => 0,
            'errors' => 0,
            'warnings' => 0,
            'doc_type' => null,
        ];

        $docType = null;
        $rowIndex = 1;
        $warnings = [];

        foreach ($uploads as $upload) {
            $result = $this->extractRowsFromUpload($upload);
            $rows = $result['rows'] ?? [];
            $uploadType = $result['doc_type'] ?? null;
            if ($uploadType && !$docType) {
                $docType = $uploadType;
            } elseif ($uploadType && $docType && $uploadType !== $docType) {
                $docType = 'mixed';
            }

            if (!empty($result['warnings'])) {
                $warnings = array_merge($warnings, $result['warnings']);
            }

            foreach ($rows as $row) {
                $normalized = $this->normalizeRow($row);
                $validation = $this->validateRow($docType, $normalized, $tenantId);

                DocumentIntakeRow::create([
                    'tenant_id' => $tenantId,
                    'batch_id' => $batch->id,
                    'row_index' => $rowIndex++,
                    'data' => $normalized,
                    'errors' => $validation['errors'],
                    'warnings' => $validation['warnings'],
                    'matches' => $validation['matches'],
                    'status' => $validation['errors'] ? 'error' : 'ready',
                ]);

                $summary['rows']++;
                $summary['errors'] += count($validation['errors']);
                $summary['warnings'] += count($validation['warnings']);
            }
        }

        $summary['doc_type'] = $docType;

        $batch->update([
            'doc_type' => $docType,
            'summary' => $summary,
            'warnings' => $warnings,
            'processed_at' => now(),
        ]);

        return $batch;
    }

    private function extractRowsFromUpload(ChatUpload $upload): array
    {
        $path = Storage::path($upload->storage_path);
        $extension = strtolower(pathinfo($upload->original_name, PATHINFO_EXTENSION));
        $rows = [];
        $warnings = [];
        $docType = null;

        if (in_array($extension, ['csv'], true)) {
            $rows = $this->parseCsv($path);
        } elseif (in_array($extension, ['xlsx', 'xls'], true)) {
            $rows = XlsxReader::readSheetRows($path, null);
        } elseif ($extension === 'docx') {
            $text = $this->extractDocxText($path);
            $aiResult = $this->parseTextWithAi($text, $upload, $warnings);
            $rows = $aiResult['rows'] ?? [];
            $docType = $aiResult['doc_type'] ?? null;
        } elseif ($extension === 'pdf') {
            $text = $this->extractPdfText($path);
            if ($text) {
                $aiResult = $this->parseTextWithAi($text, $upload, $warnings);
                $rows = $aiResult['rows'] ?? [];
                $docType = $aiResult['doc_type'] ?? null;
            } else {
                $warnings[] = 'PDF text extraction not available. Install pdftotext or enable OCR.';
            }
        } elseif (in_array($extension, ['png', 'jpg', 'jpeg'], true)) {
            $aiResult = $this->parseImageWithAi($path, $upload, $warnings);
            $rows = $aiResult['rows'] ?? [];
            $docType = $aiResult['doc_type'] ?? null;
        } else {
            $warnings[] = 'Unsupported file type: ' . $extension;
        }

        $docType = $docType ?: ($rows ? $this->detectDocType($rows) : null);
        if ($docType) {
            $upload->update(['doc_type' => $docType]);
        }

        return ['rows' => $rows, 'doc_type' => $docType, 'warnings' => $warnings];
    }

    private function parseTextWithAi(?string $text, ChatUpload $upload, array &$warnings): array
    {
        if (!$text) {
            $warnings[] = 'No text content extracted from ' . $upload->original_name;
            return [];
        }

        $parser = app(AiDocumentParserService::class);
        $result = $parser->parseText($text, null);

        if (!empty($result['errors'])) {
            $warnings[] = 'AI extraction failed for ' . $upload->original_name;
            return [];
        }

        return [
            'rows' => $result['rows'] ?? [],
            'doc_type' => $result['doc_type'] ?? null,
        ];
    }

    private function parseImageWithAi(string $path, ChatUpload $upload, array &$warnings): array
    {
        $contents = @file_get_contents($path);
        if (!$contents) {
            $warnings[] = 'Unable to read image ' . $upload->original_name;
            return [];
        }

        $parser = app(AiDocumentParserService::class);
        $result = $parser->parseImage(base64_encode($contents), $upload->mime_type ?: 'image/png', null);

        if (!empty($result['errors'])) {
            $warnings[] = 'AI OCR failed for ' . $upload->original_name;
            return [];
        }

        return [
            'rows' => $result['rows'] ?? [],
            'doc_type' => $result['doc_type'] ?? null,
        ];
    }

    private function parseCsv(string $path): array
    {
        $handle = fopen($path, 'r');
        if (!$handle) {
            return [];
        }

        $headers = [];
        $rows = [];

        while (($data = fgetcsv($handle, 0, ',')) !== false) {
            if (empty($headers)) {
                $headers = $data;
                continue;
            }

            $row = [];
            foreach ($headers as $index => $header) {
                $row[$header] = $data[$index] ?? null;
            }
            $rows[] = $row;
        }

        fclose($handle);
        return $rows;
    }

    private function extractDocxText(string $path): ?string
    {
        $zip = new \ZipArchive();
        if ($zip->open($path) !== true) {
            return null;
        }

        $xml = $zip->getFromName('word/document.xml');
        $zip->close();

        if (!$xml) {
            return null;
        }

        $text = strip_tags($xml);
        return trim($text);
    }

    private function extractPdfText(string $path): ?string
    {
        $binary = trim((string) shell_exec('where pdftotext'));
        if (!$binary) {
            return null;
        }

        $temp = tempnam(sys_get_temp_dir(), 'pdf_text_');
        $command = '"' . $binary . "\" \"{$path}\" \"{$temp}\"";
        @shell_exec($command);

        $text = @file_get_contents($temp);
        @unlink($temp);

        return $text ? trim($text) : null;
    }

    private function normalizeRow(array $row): array
    {
        $normalized = [];
        foreach ($row as $header => $value) {
            $key = $this->normalizeHeader((string) $header);
            $normalized[$key] = $value;
        }
        return $normalized;
    }

    private function normalizeHeader(string $header): string
    {
        $header = strtolower(trim($header));
        $header = preg_replace('/[^a-z0-9]+/', '_', $header);
        return trim($header, '_');
    }

    private function detectDocType(array $rows): ?string
    {
        $headers = array_keys($rows[0] ?? []);
        $normalized = array_map(fn ($header) => $this->normalizeHeader((string) $header), $headers);
        $routing = config('ai_schema_registry.routing', []);

        $bestType = null;
        $bestScore = 0;

        foreach ($routing as $type => $keywords) {
            $score = 0;
            foreach ($keywords as $keyword) {
                $needle = $this->normalizeHeader($keyword);
                foreach ($normalized as $header) {
                    if (str_contains($header, $needle)) {
                        $score++;
                        break;
                    }
                }
            }
            if ($score > $bestScore) {
                $bestScore = $score;
                $bestType = $type;
            }
        }

        return $bestType;
    }

    private function validateRow(?string $docType, array $row, int $tenantId): array
    {
        $errors = [];
        $warnings = [];
        $matches = [];

        $schema = config('ai_schema_registry.doc_types.' . $docType, []);
        $required = $schema['required'] ?? [];

        foreach ($required as $field) {
            $value = $row[$field] ?? null;
            if ($value === null || $value === '') {
                $errors[] = 'Missing ' . $field;
            }
        }

        $assetFields = $schema['asset_fields'] ?? [];
        if ($assetFields) {
            $assetTag = $row['asset_tag'] ?? null;
            $serial = $row['serial_number'] ?? null;
            $name = $row['asset_name'] ?? ($row['name'] ?? null);

            if (!$assetTag && !$serial && !$name) {
                $warnings[] = 'Asset reference missing';
            } else {
                $matcher = app(AssetMatcherService::class);
                $matches = $matcher->findMatches($tenantId, $assetTag, $serial, $name);
                if (!$matches) {
                    $warnings[] = 'No asset match found';
                }
            }
        }

        if (isset($row['date_initiated'])) {
            $row['date_initiated'] = $this->dateValue($row['date_initiated']);
        }

        return ['errors' => $errors, 'warnings' => $warnings, 'matches' => $matches];
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateString();
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }
}
