<?php

namespace App\Services;

use App\Models\AgentNotification;
use App\Models\Asset;
use App\Models\DocumentIntakeRow;
use App\Models\FuelLog;
use App\Models\GoodsReceipt;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestItem;
use App\Models\Tenant;
use App\Models\WorkOrderCost;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;

class FinanceOversightAgent
{
    public function __construct(private TelegramEventService $events)
    {
    }

    public function runAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->run((int) $tenant->id);
            }
        });
    }

    public function run(int $tenantId): void
    {
        $metrics = $this->buildMetrics($tenantId);
        $findings = $this->buildFindings($tenantId, $metrics);

        $summary = $this->summarize(
            $metrics,
            $findings,
            config('ai_agents.prompts.finance_oversight')
        );

        $severity = $this->highestSeverity($findings);
        $suggestedAction = $this->suggestedNextAction($findings);

        $payload = [
            'metrics' => $metrics,
            'findings' => $findings,
            'suggested_next_action' => $suggestedAction,
        ];

        $this->recordNotification(
            $tenantId,
            'finance_oversight',
            'Finance Oversight Agent',
            'Finance Oversight Digest',
            $summary,
            $severity,
            $payload,
            ['management', 'procurement']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.finance_oversight',
            $severity,
            'Finance Oversight Digest',
            $summary,
            [
                'details' => $this->formatFindingsList($findings, 8),
                'title' => 'FINANCE OVERSIGHT',
            ]
        );

        foreach ($findings as $finding) {
            if (($finding['severity'] ?? '') !== 'alarm') {
                continue;
            }

            $alert = "Finance Oversight ALERT\n" .
                '- ' . ($finding['title'] ?? 'Alert') . "\n" .
                ($finding['ref'] ? 'Ref: ' . $finding['ref'] . "\n" : '') .
                ($finding['vendor'] ? 'Vendor: ' . $finding['vendor'] . "\n" : '') .
                ($finding['recommendation'] ? 'Recommendation: ' . $finding['recommendation'] : '');

            $this->recordNotification(
                $tenantId,
                'finance_oversight',
                'Finance Oversight Agent',
                'Finance Oversight Alert',
                $alert,
                'alarm',
                ['finding' => $finding],
                ['management', 'procurement']
            );

            $this->sendTelegramEvent(
                $tenantId,
                'agent.finance_oversight',
                'alarm',
                'Finance Oversight Alert',
                $alert,
                [
                    'details' => [$finding['title'] ?? 'Alert'],
                    'title' => 'FINANCE ALERT',
                ]
            );
        }
    }

    private function buildMetrics(int $tenantId): array
    {
        $prAging = $this->averageAgeDays(
            PurchaseRequest::where('tenant_id', $tenantId)
                ->whereIn('status', ['submitted', 'pending_approval'])
                ->get(),
            'submitted_at'
        );

        $poAging = $this->averageAgeDays(
            PurchaseOrder::where('tenant_id', $tenantId)
                ->whereNotIn('status', ['received', 'closed'])
                ->get(),
            'sent_at'
        );

        $invoiceRows = $this->invoiceRows($tenantId);
        $invoiceAging = $this->invoiceAging($invoiceRows);
        $unpaidTotal = $this->invoiceUnpaidTotal($invoiceRows);

        $planned = (float) Asset::where('tenant_id', $tenantId)->sum('monthly_budget');
        $actual = $this->monthToDateSpend($tenantId);

        return [
            'pr_aging_days' => $prAging,
            'po_aging_days' => $poAging,
            'invoice_aging_days' => $invoiceAging,
            'unpaid_total' => $unpaidTotal,
            'planned_cashflow' => $planned,
            'month_to_date_spend' => $actual,
            'spend_vs_budget' => $planned > 0 ? round(($actual / $planned) * 100, 1) : null,
        ];
    }

    private function buildFindings(int $tenantId, array $metrics): array
    {
        $findings = [];

        $submitted = PurchaseRequest::where('tenant_id', $tenantId)
            ->whereIn('status', ['submitted', 'pending_approval'])
            ->with('approvals')
            ->limit(200)
            ->get();

        foreach ($submitted as $request) {
            $approvals = $request->approvals;
            $missingApproval = $approvals->isEmpty() || $approvals->where('status', 'approved')->count() < 2;
            if ($missingApproval) {
                $findings[] = $this->finding(
                    'warn',
                    'PR submitted without required approvals',
                    $request->request_code,
                    null,
                    'Request remaining approvals.'
                );
            }

            if (!$request->requested_by) {
                $findings[] = $this->finding(
                    'alarm',
                    'PR submitted without requester',
                    $request->request_code,
                    null,
                    'Verify requester identity and approvals.'
                );
            }

            if (in_array(strtolower((string) $request->priority), ['high', 'urgent'], true) && empty($request->notes)) {
                $findings[] = $this->finding(
                    'warn',
                    'Urgent PR missing justification',
                    $request->request_code,
                    null,
                    'Capture urgency justification.'
                );
            }
        }

        $items = PurchaseRequestItem::where('tenant_id', $tenantId)
            ->with('request')
            ->limit(300)
            ->get();

        foreach ($items as $item) {
            $request = $item->request;
            if (!$request || !$request->cost_center || !$item->purpose_cost_center) {
                continue;
            }

            if (strtolower($request->cost_center) !== strtolower($item->purpose_cost_center)) {
                $findings[] = $this->finding(
                    'alarm',
                    'PR cost center mismatch',
                    $request->request_code,
                    null,
                    'Align PR and line item cost centers.'
                );
            }
        }

        $splitFindings = $this->detectSplitPurchaseRequests($tenantId);
        $findings = array_merge($findings, $splitFindings);

        $overduePos = PurchaseOrder::where('tenant_id', $tenantId)
            ->whereNotIn('status', ['received', 'closed'])
            ->whereNotNull('delivery_due_at')
            ->where('delivery_due_at', '<', now()->subDays(7))
            ->limit(50)
            ->get();

        foreach ($overduePos as $order) {
            $findings[] = $this->finding(
                'warn',
                'PO overdue delivery',
                $order->po_number,
                $order->vendor_name,
                'Follow up supplier delivery timeline.'
            );
        }

        $orphanGrn = GoodsReceipt::where('tenant_id', $tenantId)
            ->whereNull('purchase_order_id')
            ->limit(50)
            ->get();
        foreach ($orphanGrn as $grn) {
            $findings[] = $this->finding(
                'alarm',
                'GRN missing PO link',
                $grn->reference,
                null,
                'Link GRN to a purchase order.'
            );
        }

        $invoiceRows = $this->invoiceRows($tenantId);
        $findings = array_merge($findings, $this->invoiceFindings($tenantId, $invoiceRows));

        $findings = array_merge($findings, $this->priceVarianceFindings($tenantId));

        $findings = array_merge($findings, $this->paymentRiskFindings($tenantId, $invoiceRows));

        $findings = array_merge($findings, $this->budgetFindings($metrics));

        return $findings;
    }

    private function detectSplitPurchaseRequests(int $tenantId): array
    {
        $windowStart = now()->subHours(48);
        $items = PurchaseRequestItem::where('tenant_id', $tenantId)
            ->whereHas('request', function ($q) use ($windowStart) {
                $q->where('created_at', '>=', $windowStart);
            })
            ->with('request')
            ->limit(300)
            ->get();

        $grouped = [];
        foreach ($items as $item) {
            $request = $item->request;
            if (!$request) {
                continue;
            }

            $vendor = $item->selected_supplier ?? $item->preferred_vendor ?? $item->supplier_name;
            if (!$vendor) {
                continue;
            }

            $key = strtolower(trim($vendor));
            $grouped[$key][] = [
                'request' => $request->request_code,
                'sku' => $item->sku,
                'desc' => $item->item_name ?? $item->description,
            ];
        }

        $findings = [];
        foreach ($grouped as $vendor => $entries) {
            $requests = collect($entries)->groupBy('request');
            if ($requests->count() < 2) {
                continue;
            }

            $duplicates = $this->findSimilarItems($entries);
            if ($duplicates) {
                $findings[] = $this->finding(
                    'warn',
                    'Possible split PRs to vendor within 48h',
                    implode(', ', array_keys($requests->toArray())),
                    $vendor,
                    'Review PRs for consolidation.'
                );
            }
        }

        return $findings;
    }

    private function invoiceFindings(int $tenantId, Collection $rows): array
    {
        $findings = [];
        $byVendorInvoice = [];

        foreach ($rows as $row) {
            $data = $row->data ?? [];
            $invoiceNumber = $this->stringValue($data['invoice_number'] ?? null);
            $vendor = $this->stringValue($data['vendor'] ?? $data['supplier'] ?? $data['supplier_name'] ?? null);
            $poNumber = $this->stringValue($data['po_number'] ?? null);
            $grnRef = $this->stringValue($data['grn_reference'] ?? $data['grn'] ?? null);

            if ($invoiceNumber && !$poNumber && !$grnRef) {
                $findings[] = $this->finding(
                    'alarm',
                    'Invoice missing PO/GRN link',
                    $invoiceNumber,
                    $vendor,
                    'Attach PO/GRN reference to invoice.'
                );
            }

            if ($invoiceNumber && $vendor) {
                $key = strtolower($vendor) . '|' . strtolower($invoiceNumber);
                $byVendorInvoice[$key][] = $row;
            }
        }

        foreach ($byVendorInvoice as $key => $group) {
            if (count($group) < 2) {
                continue;
            }

            $parts = explode('|', $key);
            $findings[] = $this->finding(
                'alarm',
                'Duplicate invoice detected',
                $parts[1] ?? 'Invoice',
                $parts[0] ?? null,
                'Hold payment and confirm duplicate invoice.'
            );
        }

        return $findings;
    }

    private function priceVarianceFindings(int $tenantId): array
    {
        $findings = [];
        $recentStart = now()->subDays(7);
        $historyStart = now()->subDays(90);

        $recentItems = PurchaseOrderItem::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $recentStart)
            ->with('order')
            ->limit(200)
            ->get();

        foreach ($recentItems as $item) {
            $vendor = $item->order?->vendor_name;
            $sku = $item->sku;
            $unitCost = (float) ($item->unit_cost ?? 0);
            if (!$vendor || !$sku || $unitCost <= 0) {
                continue;
            }

            $avg = PurchaseOrderItem::where('tenant_id', $tenantId)
                ->where('created_at', '>=', $historyStart)
                ->where('created_at', '<', $recentStart)
                ->where('sku', $sku)
                ->whereHas('order', fn ($q) => $q->where('vendor_name', $vendor))
                ->avg('unit_cost');

            if (!$avg || $avg <= 0) {
                continue;
            }

            $variance = ($unitCost - $avg) / $avg;
            if (abs($variance) > 0.15) {
                $findings[] = $this->finding(
                    'warn',
                    'Unit price variance > 15%',
                    $sku,
                    $vendor,
                    'Validate pricing against historical average.'
                );
            }
        }

        return $findings;
    }

    private function paymentRiskFindings(int $tenantId, Collection $rows): array
    {
        $findings = [];
        $overdue = $this->overdueInvoices($tenantId, $rows);

        foreach ($overdue as $row) {
            $data = $row['data'];
            $invoiceNumber = $this->stringValue($data['invoice_number'] ?? null);
            $vendor = $this->stringValue($data['vendor'] ?? $data['supplier'] ?? null);
            $findings[] = $this->finding(
                'alarm',
                'Approved invoice unpaid beyond terms',
                $invoiceNumber,
                $vendor,
                'Review payment status and clear overdue invoices.'
            );
        }

        $exposure = $this->supplierExposure($rows);
        if ($exposure['top_vendors']) {
            $findings[] = $this->finding(
                'warn',
                'Supplier exposure concentration',
                implode(', ', $exposure['top_vendors']),
                null,
                'Review aging buckets and plan settlements.'
            );
        }

        return $findings;
    }

    private function budgetFindings(array $metrics): array
    {
        $planned = (float) ($metrics['planned_cashflow'] ?? 0);
        $actual = (float) ($metrics['month_to_date_spend'] ?? 0);
        if ($planned <= 0) {
            return [];
        }

        $ratio = $actual / $planned;
        if ($ratio > 1.35) {
            return [
                $this->finding(
                    'alarm',
                    'Spend exceeds planned cashflow by > 35%',
                    null,
                    null,
                    'Review cashflow forecast and approvals.'
                ),
            ];
        }

        if ($ratio > 1.2) {
            return [
                $this->finding(
                    'warn',
                    'Spend exceeds planned cashflow by > 20%',
                    null,
                    null,
                    'Monitor discretionary spend and approvals.'
                ),
            ];
        }

        return [];
    }

    private function invoiceRows(int $tenantId): Collection
    {
        return DocumentIntakeRow::where('tenant_id', $tenantId)
            ->whereHas('batch', fn ($q) => $q->where('doc_type', 'invoice'))
            ->get();
    }

    private function invoiceAging(Collection $rows): ?float
    {
        $ages = [];
        foreach ($rows as $row) {
            $data = $row->data ?? [];
            $issuedAt = $this->dateValue($data['invoice_date'] ?? $data['issued_at'] ?? null);
            if (!$issuedAt) {
                continue;
            }
            $ages[] = $issuedAt->diffInDays(now());
        }

        return $ages ? round(array_sum($ages) / count($ages), 1) : null;
    }

    private function invoiceUnpaidTotal(Collection $rows): float
    {
        $total = 0.0;
        foreach ($rows as $row) {
            $data = $row->data ?? [];
            if (!empty($data['paid_at'])) {
                continue;
            }
            $amount = $this->numericValue($data['amount'] ?? null);
            $total += $amount ?? 0;
        }

        return round($total, 2);
    }

    private function overdueInvoices(int $tenantId, Collection $rows): array
    {
        $overdue = [];
        foreach ($rows as $row) {
            $data = $row->data ?? [];
            if (!empty($data['paid_at'])) {
                continue;
            }

            $due = $this->dateValue($data['due_date'] ?? null);
            if (!$due) {
                $invoiceDate = $this->dateValue($data['invoice_date'] ?? $data['issued_at'] ?? null);
                $termsDays = $this->termsDays($this->invoicePoTerms($tenantId, $data));
                if ($invoiceDate) {
                    $due = $invoiceDate->copy()->addDays($termsDays);
                }
            }

            if ($due && $due->isPast()) {
                $overdue[] = ['row' => $row, 'data' => $data, 'due_at' => $due];
            }
        }

        return $overdue;
    }

    private function supplierExposure(Collection $rows): array
    {
        $totals = [];
        $buckets = [
            '0-30' => 0.0,
            '31-60' => 0.0,
            '61-90' => 0.0,
            '90+' => 0.0,
        ];

        foreach ($rows as $row) {
            $data = $row->data ?? [];
            if (!empty($data['paid_at'])) {
                continue;
            }
            $amount = (float) ($this->numericValue($data['amount'] ?? null) ?? 0);
            if ($amount <= 0) {
                continue;
            }

            $vendor = $this->stringValue($data['vendor'] ?? $data['supplier'] ?? $data['supplier_name'] ?? 'Unknown');
            $totals[$vendor] = ($totals[$vendor] ?? 0) + $amount;

            $issuedAt = $this->dateValue($data['invoice_date'] ?? $data['issued_at'] ?? null);
            $age = $issuedAt ? $issuedAt->diffInDays(now()) : 0;
            if ($age <= 30) {
                $buckets['0-30'] += $amount;
            } elseif ($age <= 60) {
                $buckets['31-60'] += $amount;
            } elseif ($age <= 90) {
                $buckets['61-90'] += $amount;
            } else {
                $buckets['90+'] += $amount;
            }
        }

        arsort($totals);
        $topVendors = array_slice(array_keys($totals), 0, 3);

        return [
            'top_vendors' => $topVendors,
            'aging_buckets' => $buckets,
        ];
    }

    private function monthToDateSpend(int $tenantId): float
    {
        $start = now()->startOfMonth();
        $maintenance = (float) WorkOrderCost::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $start)
            ->sum('total_cost');
        $fuel = (float) FuelLog::where('tenant_id', $tenantId)
            ->where('logged_at', '>=', $start)
            ->sum('total_cost');

        return round($maintenance + $fuel, 2);
    }

    private function averageAgeDays(Collection $rows, string $dateField): ?float
    {
        $ages = [];
        foreach ($rows as $row) {
            $date = $row->{$dateField} ?? $row->created_at ?? null;
            if (!$date) {
                continue;
            }
            $ages[] = Carbon::parse($date)->diffInDays(now());
        }

        return $ages ? round(array_sum($ages) / count($ages), 1) : null;
    }

    private function summarize(array $metrics, array $findings, ?string $prompt): string
    {
        $lines = [
            'PR aging: ' . ($metrics['pr_aging_days'] ?? 'N/A') . ' days',
            'PO aging: ' . ($metrics['po_aging_days'] ?? 'N/A') . ' days',
            'Invoice aging: ' . ($metrics['invoice_aging_days'] ?? 'N/A') . ' days',
            'Unpaid total: $' . number_format((float) ($metrics['unpaid_total'] ?? 0), 2),
            'Spend vs budget: ' . ($metrics['spend_vs_budget'] !== null ? $metrics['spend_vs_budget'] . '%' : 'N/A'),
        ];

        $findingLines = $this->formatFindingsList($findings, 8);
        $summary = "Finance Oversight Digest\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, $lines)) .
            ($findingLines ? "\nFindings:\n" . implode("\n", array_map(fn ($line) => '- ' . $line, $findingLines)) : "\nFindings: None");

        if (!$prompt) {
            return $summary;
        }

        $aiSummary = $this->summarizeWithAi($prompt, $metrics, $findings);
        return $aiSummary ?: $summary;
    }

    private function summarizeWithAi(string $prompt, array $metrics, array $findings): ?string
    {
        $key = config('services.openai.key');
        if (!$key) {
            return null;
        }

        $payload = [
            'model' => config('services.openai.model'),
            'messages' => [
                ['role' => 'system', 'content' => $prompt],
                ['role' => 'user', 'content' => json_encode(['metrics' => $metrics, 'findings' => $findings])],
            ],
            'temperature' => 0.2,
            'max_tokens' => 400,
        ];

        $verify = config('services.openai.verify_ssl', true);
        $verify = filter_var($verify, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $verify = $verify ?? true;

        try {
            $response = Http::withOptions(['verify' => $verify])
                ->withToken($key)
                ->timeout((int) config('services.openai.timeout', 20))
                ->post(config('services.openai.endpoint'), $payload);
        } catch (\Throwable $e) {
            return null;
        }

        if ($response->failed()) {
            return null;
        }

        return trim((string) data_get($response->json(), 'choices.0.message.content'));
    }

    private function highestSeverity(array $findings): string
    {
        $rank = ['info' => 1, 'warn' => 2, 'alarm' => 3];
        $highest = 'info';
        foreach ($findings as $finding) {
            $severity = $finding['severity'] ?? 'info';
            if (($rank[$severity] ?? 1) > ($rank[$highest] ?? 1)) {
                $highest = $severity;
            }
        }

        return $highest;
    }

    private function suggestedNextAction(array $findings): string
    {
        foreach ($findings as $finding) {
            if (($finding['severity'] ?? '') !== 'alarm') {
                continue;
            }

            $title = strtolower((string) ($finding['title'] ?? ''));
            if (str_contains($title, 'invoice') || str_contains($title, 'payment')) {
                return 'HOLD_PAYMENT';
            }
            if (str_contains($title, 'approval') || str_contains($title, 'unauthorized')) {
                return 'REQUEST_APPROVAL';
            }
            return 'NOTIFY';
        }

        return 'NONE';
    }

    private function recordNotification(
        int $tenantId,
        string $agentKey,
        string $agentName,
        string $title,
        string $summary,
        string $severity,
        array $payload,
        array $recipients
    ): AgentNotification {
        return AgentNotification::create([
            'tenant_id' => $tenantId,
            'agent_key' => $agentKey,
            'agent_name' => $agentName,
            'title' => $title,
            'summary' => $summary,
            'severity' => $severity,
            'payload' => array_merge($payload, ['prompt' => config('ai_agents.prompts.' . $agentKey)]),
            'recipients' => $recipients,
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    private function sendTelegramEvent(
        int $tenantId,
        string $eventType,
        string $severity,
        string $title,
        string $message,
        array $payload
    ): void {
        $this->events->createEvent($tenantId, $eventType, $severity, array_merge($payload, [
            'title' => $title,
        ]), [
            'title' => $title,
            'message' => $message,
        ]);
    }

    private function formatFindingsList(array $findings, int $limit): array
    {
        $lines = [];
        foreach (array_slice($findings, 0, $limit) as $finding) {
            $line = $finding['title'] ?? 'Finding';
            if (!empty($finding['ref'])) {
                $line .= ' (' . $finding['ref'] . ')';
            }
            $lines[] = $line;
        }

        return $lines;
    }

    private function finding(
        string $severity,
        string $title,
        ?string $ref,
        ?string $vendor,
        string $recommendation
    ): array {
        return [
            'severity' => $severity,
            'title' => $title,
            'ref' => $ref,
            'vendor' => $vendor,
            'recommendation' => $recommendation,
        ];
    }

    private function findSimilarItems(array $entries): bool
    {
        $seen = [];
        foreach ($entries as $entry) {
            $sku = $this->stringValue($entry['sku'] ?? null);
            $desc = $this->normalizeText($entry['desc'] ?? null);
            $key = $sku ?: $desc;
            if (!$key) {
                continue;
            }
            if (isset($seen[$key]) && $seen[$key] !== ($entry['request'] ?? null)) {
                return true;
            }
            $seen[$key] = $entry['request'] ?? null;
        }

        return false;
    }

    private function normalizeText(?string $value): ?string
    {
        if (!$value) {
            return null;
        }
        $value = strtolower($value);
        $value = preg_replace('/[^a-z0-9]+/', '', $value);
        return $value ?: null;
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }
        if (is_numeric($value)) {
            return (float) $value;
        }
        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        return $cleaned === '' ? null : (float) $cleaned;
    }

    private function stringValue(?string $value): ?string
    {
        $value = $value ? trim((string) $value) : null;
        return $value !== '' ? $value : null;
    }

    private function dateValue(mixed $value): ?Carbon
    {
        if ($value === null || $value === '') {
            return null;
        }
        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value);
        }
        try {
            return Carbon::parse($value);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function invoicePoTerms(int $tenantId, array $data): ?string
    {
        $poNumber = $this->stringValue($data['po_number'] ?? null);
        if (!$poNumber) {
            return null;
        }
        $po = PurchaseOrder::where('tenant_id', $tenantId)->where('po_number', $poNumber)->first();
        return $po?->terms;
    }

    private function termsDays(?string $terms): int
    {
        if (!$terms) {
            return 30;
        }
        if (preg_match('/(\d+)/', $terms, $matches)) {
            return (int) $matches[1];
        }
        return 30;
    }
}
