<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\ComplianceRecord;
use App\Models\InventoryItem;
use App\Models\InventoryLocation;
use App\Models\MaintenanceCashflowLine;
use App\Models\MaintenanceRiskHeatmap;
use App\Models\MaintenanceRiskRegister;
use App\Models\MaintenanceWeeklyRepair;
use App\Models\Part;
use App\Models\PmSchedule;
use App\Models\Site;
use App\Models\Tenant;
use App\Models\Vehicle;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class FourwaysMaintenanceDataImportService
{
    public function importAll(string $strategyPack, string $cashflowFile, string $concreteFile, Tenant $tenant): array
    {
        $summary = [
            'assets_created' => 0,
            'assets_updated' => 0,
            'vehicles_created' => 0,
            'vehicles_updated' => 0,
            'compliance_records' => 0,
            'pm_schedules' => 0,
            'parts_created' => 0,
            'parts_updated' => 0,
            'inventory_items' => 0,
            'cashflow_lines' => 0,
            'weekly_repairs' => 0,
            'risk_heatmap' => 0,
            'risk_register' => 0,
        ];

        DB::transaction(function () use ($strategyPack, $cashflowFile, $concreteFile, $tenant, &$summary) {
            $this->clearExistingBudgetData($tenant->id, $cashflowFile, $concreteFile);
            $this->importAssetsMaintRegister($strategyPack, $tenant, $summary);
            $this->importAssetsMerged($strategyPack, $tenant, $summary);
            $this->importAssetCriticality($strategyPack, $tenant, $summary);
            $this->importComplianceAll($strategyPack, $tenant, $summary);

            $templates = $this->loadPmTemplates($strategyPack);
            $this->importAssetPmMapping($strategyPack, $tenant, $templates, $summary);

            $this->importSparesMinMax($strategyPack, $tenant, $summary);

            $this->importMonthlyCashflow($cashflowFile, $tenant, $summary);
            $this->importBacklogWeekly($cashflowFile, $tenant, $summary);

            $this->importWeeklyRepairs($concreteFile, $tenant, $summary);
            $this->importRiskHeatMap($concreteFile, $tenant, $summary);
            $this->importRiskRegister($concreteFile, $tenant, $summary);
        });

        return $summary;
    }

    private function importAssetsMaintRegister(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Assets_Maint_Register');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $name = $this->stringValue($row['Asset Name'] ?? null);
            if (!$assetId && !$name) {
                continue;
            }

            $location = $this->stringValue($row['Location'] ?? null);
            $siteId = $location ? $this->resolveSite($tenant->id, $location) : null;
            $category = $this->stringValue($row['Asset Group'] ?? null);
            $status = $this->normalizeStatus($this->stringValue($row['Status'] ?? null));

            $data = [
                'tenant_id' => $tenant->id,
                'asset_tag' => $assetId ?: $name,
                'name' => $name ?: $assetId,
                'category' => $category,
                'location' => $location,
                'site_id' => $siteId,
                'status' => $status ?: 'active',
                'lifecycle_status' => $status ?: 'active',
                'service_frequency' => $this->stringValue($row['Service Frequency'] ?? null),
                'maintenance_interval' => $this->numericValue($row['Service Frequency'] ?? null),
                'yearly_budget' => $this->numericValue($row['Yearly Budget (USD)'] ?? null),
                'monthly_budget' => $this->numericValue($row['Monthly Budget (USD)'] ?? null),
            ];

            $asset = Asset::where('tenant_id', $tenant->id)
                ->where('asset_tag', $data['asset_tag'])
                ->first();

            if ($asset) {
                $asset->fill($data);
                if ($asset->isDirty()) {
                    $asset->save();
                    $summary['assets_updated']++;
                }
            } else {
                Asset::create($data);
                $summary['assets_created']++;
            }
        }
    }

    private function importAssetsMerged(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Assets_Merged');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $name = $this->stringValue($row['Asset Description'] ?? null);
            if (!$assetId && !$name) {
                continue;
            }

            $category = $this->stringValue($row['Asset Group'] ?? null);
            $location = $this->stringValue($row['Allocation'] ?? null);
            $siteId = $location ? $this->resolveSite($tenant->id, $location) : null;
            $plate = $this->stringValue($row['Plate'] ?? null);
            $type = $this->stringValue($row['Type'] ?? null);

            $data = [
                'tenant_id' => $tenant->id,
                'asset_tag' => $assetId,
                'name' => $name ?: $assetId,
                'category' => $category,
                'asset_type' => $plate ? 'vehicle' : $this->inferAssetType($category),
                'location' => $location,
                'site_id' => $siteId,
                'purchase_date' => $this->dateValue($row['Date of Purchase'] ?? null),
                'current_value' => $this->numericValue($row['Purchase Cost (USD)'] ?? null),
                'description' => $name,
                'serial_number' => $plate,
            ];

            $asset = Asset::where('tenant_id', $tenant->id)
                ->where('asset_tag', $assetId)
                ->first();

            if ($asset) {
                $asset->fill($data);
                if ($asset->isDirty()) {
                    $asset->save();
                    $summary['assets_updated']++;
                }
            } else {
                $asset = Asset::create($data);
                $summary['assets_created']++;
            }

            if ($plate) {
                $vehicle = Vehicle::where('tenant_id', $tenant->id)
                    ->where(function ($query) use ($asset, $plate) {
                        $query->where('asset_id', $asset->id)
                            ->orWhere('license_plate', $plate);
                    })
                    ->first();

                $payload = [
                    'tenant_id' => $tenant->id,
                    'asset_id' => $asset->id,
                    'site_id' => $siteId,
                    'name' => $name ?: $asset->name,
                    'type' => $type ?: $category,
                    'license_plate' => $plate,
                    'status' => 'active',
                ];

                if ($vehicle) {
                    $vehicle->fill($payload);
                    if ($vehicle->isDirty()) {
                        $vehicle->save();
                        $summary['vehicles_updated']++;
                    }
                } else {
                    Vehicle::create($payload);
                    $summary['vehicles_created']++;
                }
            }

            $this->syncComplianceDates($tenant, $asset, $plate, $row, $summary);
        }
    }

    private function importAssetCriticality(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Asset_Criticality');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $name = $this->stringValue($row['Asset'] ?? null);
            if (!$assetId && !$name) {
                continue;
            }

            $asset = $this->findAsset($tenant->id, $assetId, $name);
            if (!$asset) {
                continue;
            }

            $criticality = $this->stringValue($row['Criticality'] ?? null);
            if (!$criticality) {
                continue;
            }

            $note = 'Criticality: ' . $criticality;
            if (!$asset->comments || !str_contains((string) $asset->comments, 'Criticality:')) {
                $asset->comments = trim(($asset->comments ? $asset->comments . ' | ' : '') . $note);
                $asset->save();
            }
        }
    }

    private function importComplianceAll(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Compliance_All');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $name = $this->stringValue($row['Asset'] ?? null);
            $requirement = $this->stringValue($row['Requirement'] ?? null);
            if (!$requirement) {
                continue;
            }

            $asset = $this->findAsset($tenant->id, $assetId, $name);
            if (!$asset) {
                continue;
            }

            $expiry = $this->dateValue($row['Due Date'] ?? null);
            $status = $this->normalizeComplianceStatus($this->stringValue($row['Status'] ?? null));

            $this->upsertCompliance($tenant->id, $asset->id, null, $requirement, $expiry, $status);
            $summary['compliance_records']++;
        }
    }

    private function loadPmTemplates(string $path): array
    {
        $rows = $this->readRows($path, 'PM_Templates');
        $templates = [];
        foreach ($rows as $row) {
            $code = $this->stringValue($row['Template Code'] ?? null);
            if (!$code) {
                continue;
            }

            $templates[$code] = [
                'trigger' => $this->stringValue($row['Trigger'] ?? null),
                'interval' => $this->stringValue($row['Interval'] ?? null),
                'description' => $this->stringValue($row['Description'] ?? null),
            ];
        }

        return $templates;
    }

    private function importAssetPmMapping(string $path, Tenant $tenant, array $templates, array &$summary): void
    {
        $rows = $this->readRows($path, 'Asset_PM_Mapping');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $assetId = $this->stringValue($row['Asset ID'] ?? null);
            $name = $this->stringValue($row['Asset'] ?? null);
            $templateCode = $this->stringValue($row['Template Code'] ?? null);
            if (!$templateCode) {
                continue;
            }

            $asset = $this->findAsset($tenant->id, $assetId, $name);
            if (!$asset) {
                continue;
            }

            $template = $templates[$templateCode] ?? [];
            [$intervalValue, $intervalUnit] = $this->parseInterval($template['interval'] ?? null);
            $trigger = strtolower((string) ($template['trigger'] ?? ''));

            $scheduleType = str_contains($trigger, 'meter') || $intervalUnit === 'hours'
                ? 'meter'
                : 'calendar';

            $meterType = $scheduleType === 'meter' ? 'hours' : null;

            $payload = [
                'tenant_id' => $tenant->id,
                'asset_id' => $asset->id,
                'name' => $templateCode,
                'schedule_type' => $scheduleType,
                'interval_value' => $intervalValue,
                'interval_unit' => $intervalUnit,
                'meter_type' => $meterType,
                'active' => true,
            ];

            PmSchedule::updateOrCreate(
                [
                    'tenant_id' => $tenant->id,
                    'asset_id' => $asset->id,
                    'name' => $templateCode,
                ],
                $payload
            );

            $summary['pm_schedules']++;
        }
    }

    private function importSparesMinMax(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Spares_MinMax');
        if (!$rows) {
            return;
        }

        $location = InventoryLocation::firstOrCreate(
            ['tenant_id' => $tenant->id, 'name' => 'Main Store'],
            ['code' => 'MAIN', 'site_id' => null]
        );

        foreach ($rows as $row) {
            $name = $this->stringValue($row['Spare Part'] ?? null);
            if (!$name) {
                continue;
            }

            $partData = [
                'tenant_id' => $tenant->id,
                'name' => $name,
                'sku' => $this->makeSku($name),
                'stage_of_use' => $this->stringValue($row['Stage of Use'] ?? null),
                'estimated_replacements_per_year' => $this->numericValue($row['annual_qty'] ?? null),
                'cost_per_year' => $this->numericValue($row['annual_cost'] ?? null),
                'unit' => 'pcs',
            ];

            $part = Part::where('tenant_id', $tenant->id)->where('name', $name)->first();
            if ($part) {
                $part->fill($partData);
                if ($part->isDirty()) {
                    $part->save();
                    $summary['parts_updated']++;
                }
            } else {
                $part = Part::create($partData);
                $summary['parts_created']++;
            }

            InventoryItem::updateOrCreate(
                [
                    'tenant_id' => $tenant->id,
                    'part_id' => $part->id,
                    'location_id' => $location->id,
                ],
                [
                    'quantity' => 0,
                    'unit_cost' => $this->numericValue($row['unit_cost'] ?? null),
                    'min_quantity' => $this->numericValue($row['suggested_min'] ?? null),
                    'max_quantity' => $this->numericValue($row['suggested_max'] ?? null),
                    'reorder_point' => $this->numericValue($row['reorder_point'] ?? null),
                ]
            );

            $summary['inventory_items']++;
        }
    }

    private function importMonthlyCashflow(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Monthly Cashflow', 3);
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $month = $this->stringValue($row['Month'] ?? null);
            if (!$month) {
                continue;
            }

            foreach ($row as $label => $value) {
                if ($label === 'Month') {
                    continue;
                }

                $amount = $this->numericValue($value);
                if ($amount === null) {
                    continue;
                }

                MaintenanceCashflowLine::create([
                    'tenant_id' => $tenant->id,
                    'period_type' => 'monthly',
                    'period_label' => $month,
                    'category' => $label,
                    'amount_usd' => $amount,
                    'source' => basename($path),
                ]);

                $summary['cashflow_lines']++;
            }
        }
    }

    private function importBacklogWeekly(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Backlog Weekly (Start 19 Jan)');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $week = $this->stringValue($row['Week'] ?? null);
            if (!$week) {
                continue;
            }

            MaintenanceCashflowLine::create([
                'tenant_id' => $tenant->id,
                'period_type' => 'backlog_weekly',
                'period_label' => $week,
                'period_start' => $this->dateValue($row['Start'] ?? null),
                'period_end' => $this->dateValue($row['End'] ?? null),
                'category' => 'Backlog',
                'amount_usd' => $this->numericValue($row['Planned Total (USD)'] ?? null),
                'notes' => $this->stringValue($row['Notes'] ?? null),
                'source' => basename($path),
            ]);

            $summary['cashflow_lines']++;
        }
    }

    private function importWeeklyRepairs(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Weekly Tracker');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $week = $this->stringValue($row['Week'] ?? null);
            $asset = $this->stringValue($row['Asset'] ?? null);
            if (!$week || !$asset) {
                continue;
            }

            $fxKey = $this->findKey($row, ['ZAR', 'USD']);
            MaintenanceWeeklyRepair::create([
                'tenant_id' => $tenant->id,
                'week' => $week,
                'priority' => (int) $this->numericValue($row['Priority'] ?? null),
                'asset_ref' => $asset,
                'category' => $this->stringValue($row['Category'] ?? null),
                'scope' => $this->stringValue($row['Scope'] ?? null),
                'original_usd' => $this->numericValue($row['Original (USD)'] ?? null),
                'original_zar' => $this->numericValue($row['Original (ZAR)'] ?? null),
                'fx_rate' => $fxKey ? $this->numericValue($row[$fxKey] ?? null) : null,
                'additionals_usd' => $this->numericValue($row['Additionals (USD)'] ?? null),
                'planned_total_usd' => $this->numericValue($row['Planned Total (USD)'] ?? null),
                'source' => basename($path),
            ]);

            $summary['weekly_repairs']++;
        }
    }

    private function importRiskHeatMap(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Weekly Risk Heat Map');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $week = $this->stringValue($row['Week'] ?? null);
            if (!$week) {
                continue;
            }

            MaintenanceRiskHeatmap::create([
                'tenant_id' => $tenant->id,
                'week' => $week,
                'overall_risk' => $this->stringValue($row['Overall Risk'] ?? null),
                'safety' => $this->stringValue($row['Safety'] ?? null),
                'compliance' => $this->stringValue($row['Compliance'] ?? null),
                'production' => $this->stringValue($row['Production'] ?? null),
                'financial' => $this->stringValue($row['Financial'] ?? null),
                'reputation' => $this->stringValue($row['Reputation'] ?? null),
                'source' => basename($path),
            ]);

            $summary['risk_heatmap']++;
        }
    }

    private function importRiskRegister(string $path, Tenant $tenant, array &$summary): void
    {
        $rows = $this->readRows($path, 'Risk Register');
        if (!$rows) {
            return;
        }

        foreach ($rows as $row) {
            $riskId = $this->stringValue($row['Risk ID'] ?? null);
            if (!$riskId) {
                continue;
            }

            MaintenanceRiskRegister::create([
                'tenant_id' => $tenant->id,
                'risk_code' => $riskId,
                'week' => $this->stringValue($row['Week'] ?? null),
                'assets' => $this->stringValue($row['Asset(s)'] ?? null),
                'description' => $this->stringValue($row['Risk Description'] ?? null),
                'category' => $this->stringValue($row['Risk Category'] ?? null),
                'likelihood' => $this->stringValue($row['Likelihood'] ?? null),
                'impact' => $this->stringValue($row['Impact'] ?? null),
                'rating' => $this->stringValue($row['Risk Rating'] ?? null),
                'mitigation' => $this->stringValue($row['Mitigation'] ?? null),
                'owner' => $this->stringValue($row['Owner'] ?? null),
                'status' => $this->stringValue($row['Status'] ?? null),
                'source' => basename($path),
            ]);

            $summary['risk_register']++;
        }
    }

    private function syncComplianceDates(Tenant $tenant, Asset $asset, ?string $plate, array $row, array &$summary): void
    {
        $fields = [
            'ZINARA' => $row['ZINARA'] ?? null,
            'INSURANCE' => $row['INSURANCE'] ?? null,
            'RADIO' => $row['RADIO'] ?? null,
            'REG BOOK' => $row['REG BOOK'] ?? null,
        ];

        foreach ($fields as $type => $value) {
            if ($value === null || $value === '') {
                $status = 'missing';
                $expiry = null;
            } else {
                $expiry = $this->dateValue($value);
                $status = $expiry ? 'active' : $this->stringValue($value);
            }

            $this->upsertCompliance($tenant->id, $asset->id, null, $type, $expiry, $status);
            $summary['compliance_records']++;
        }
    }

    private function upsertCompliance(int $tenantId, ?int $assetId, ?int $vehicleId, string $type, ?string $expiry, ?string $status): void
    {
        ComplianceRecord::withoutEvents(function () use ($tenantId, $assetId, $vehicleId, $type, $expiry, $status) {
            $record = ComplianceRecord::where('tenant_id', $tenantId)
                ->where('asset_id', $assetId)
                ->where('compliance_type', $type)
                ->first();

            if ($record) {
                $record->expiry_date = $expiry;
                $record->status = $status ?: $record->status;
                $record->save();
                return;
            }

            ComplianceRecord::create([
                'tenant_id' => $tenantId,
                'asset_id' => $assetId,
                'vehicle_id' => $vehicleId,
                'compliance_type' => $type,
                'expiry_date' => $expiry,
                'status' => $status ?: 'active',
            ]);
        });
    }

    private function findAsset(int $tenantId, ?string $assetId, ?string $name): ?Asset
    {
        if ($assetId) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $assetId)->first();
            if ($asset) {
                return $asset;
            }
        }

        if ($name) {
            return Asset::where('tenant_id', $tenantId)->where('name', $name)->first();
        }

        return null;
    }

    private function resolveSite(int $tenantId, string $locationName): int
    {
        $site = Site::firstOrCreate(
            ['tenant_id' => $tenantId, 'name' => $locationName],
            [
                'code' => $this->siteCode($locationName),
                'type' => 'site',
                'address' => null,
                'active' => true,
            ]
        );

        return $site->id;
    }

    private function siteCode(string $name): string
    {
        $code = strtoupper(Str::substr(Str::slug($name, ''), 0, 6));
        return $code ?: 'SITE';
    }

    private function inferAssetType(?string $assetGroup): string
    {
        $group = strtolower((string) $assetGroup);
        if (str_contains($group, 'truck') || str_contains($group, 'vehicle') || str_contains($group, 'fleet')) {
            return 'vehicle';
        }
        if (str_contains($group, 'mobile')) {
            return 'mobile';
        }
        if (str_contains($group, 'stationary') || str_contains($group, 'fixed') || str_contains($group, 'plant')) {
            return 'fixed';
        }
        return 'asset';
    }

    private function parseInterval(?string $value): array
    {
        $value = strtolower(trim((string) $value));
        if ($value === '') {
            return [null, null];
        }

        if (preg_match('/(\d+(?:\.\d+)?)\s*(hour|hours|hr|hrs|km|day|days|week|weeks|month|months)/', $value, $matches)) {
            $number = (float) $matches[1];
            $unit = $matches[2];
            $unit = match (true) {
                str_starts_with($unit, 'hr') || str_starts_with($unit, 'hour') => 'hours',
                str_starts_with($unit, 'day') => 'days',
                str_starts_with($unit, 'week') => 'weeks',
                str_starts_with($unit, 'month') => 'months',
                default => $unit,
            };
            return [$number, $unit];
        }

        return [$this->numericValue($value), null];
    }

    private function normalizeComplianceStatus(?string $value): string
    {
        $value = strtolower((string) $value);
        if ($value === '') {
            return 'active';
        }
        if (str_contains($value, 'overdue') || str_contains($value, 'expired')) {
            return 'expired';
        }
        if (str_contains($value, 'missing')) {
            return 'missing';
        }
        return $value;
    }

    private function normalizeStatus(?string $status): ?string
    {
        $value = strtolower((string) $status);
        if ($value === '') {
            return null;
        }

        if (str_contains($value, 'current') || str_contains($value, 'active') || str_contains($value, 'running')) {
            return 'active';
        }
        if (str_contains($value, 'standby') || str_contains($value, 'idle')) {
            return 'standby';
        }
        if (str_contains($value, 'down') || str_contains($value, 'break') || str_contains($value, 'repair')) {
            return 'down';
        }
        if (str_contains($value, 'retire')) {
            return 'retired';
        }

        return $value;
    }

    private function stringValue(mixed $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        return $value === '' ? null : $value;
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateString();
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function makeSku(string $name): string
    {
        $sku = strtoupper(Str::slug($name, ''));
        $sku = substr($sku, 0, 12);

        return $sku ?: strtoupper(Str::random(6));
    }

    private function readRows(string $path, string $sheetName, ?int $headerRow = null): array
    {
        if (!is_file($path)) {
            return [];
        }

        $reader = IOFactory::createReaderForFile($path);
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($path);
        $sheet = $spreadsheet->getSheetByName($sheetName);
        if (!$sheet) {
            return [];
        }

        $rows = $sheet->toArray(null, true, true, true);
        if (!$rows) {
            return [];
        }

        $headerIndex = $headerRow ?: $this->detectHeaderRow($rows);
        $headers = $rows[$headerIndex] ?? [];
        $result = [];

        foreach ($rows as $index => $row) {
            if ($index <= $headerIndex) {
                continue;
            }

            $record = [];
            foreach ($headers as $column => $header) {
                $header = $this->stringValue($header);
                if (!$header) {
                    continue;
                }
                $record[$header] = $row[$column] ?? null;
            }

            if (array_filter($record, fn ($value) => $value !== null && $value !== '')) {
                $result[] = $record;
            }
        }

        return $result;
    }

    private function detectHeaderRow(array $rows): int
    {
        $bestIndex = array_key_first($rows);
        $bestScore = -1;

        $scanned = 0;
        foreach ($rows as $index => $row) {
            $scanned++;
            if ($scanned > 25) {
                break;
            }

            $values = array_values($row);
            $nonEmpty = array_filter($values, fn ($value) => $value !== null && $value !== '');
            if (!$nonEmpty) {
                continue;
            }

            $stringCount = 0;
            foreach ($nonEmpty as $value) {
                if (is_string($value) && preg_match('/[A-Za-z]/', $value)) {
                    $stringCount++;
                }
            }

            if ($stringCount > $bestScore) {
                $bestScore = $stringCount;
                $bestIndex = $index;
            }
        }

        return (int) $bestIndex;
    }

    private function clearExistingBudgetData(int $tenantId, string $cashflowFile, string $concreteFile): void
    {
        $cashflowSource = basename($cashflowFile);
        $concreteSource = basename($concreteFile);

        MaintenanceCashflowLine::where('tenant_id', $tenantId)
            ->where('source', $cashflowSource)
            ->delete();

        MaintenanceWeeklyRepair::where('tenant_id', $tenantId)
            ->where('source', $concreteSource)
            ->delete();

        MaintenanceRiskHeatmap::where('tenant_id', $tenantId)
            ->where('source', $concreteSource)
            ->delete();

        MaintenanceRiskRegister::where('tenant_id', $tenantId)
            ->where('source', $concreteSource)
            ->delete();
    }

    private function findKey(array $row, array $needles): ?string
    {
        foreach (array_keys($row) as $key) {
            $haystack = strtolower((string) $key);
            $matched = true;
            foreach ($needles as $needle) {
                if (!str_contains($haystack, strtolower($needle))) {
                    $matched = false;
                    break;
                }
            }
            if ($matched) {
                return $key;
            }
        }

        return null;
    }
}
