<?php

namespace App\Services;

use App\Models\AgentNotification;
use App\Models\User;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Models\WorkOrderDocument;
use App\Models\Tenant;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;

class HrArtisanPerformanceAgent
{
    public function __construct(private TelegramEventService $events)
    {
    }

    public function runAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->run((int) $tenant->id);
            }
        });
    }

    public function run(int $tenantId): void
    {
        $range = $this->weekRange();
        $artisans = $this->artisanUsers($tenantId);
        if ($artisans->isEmpty()) {
            $this->recordNotification(
                $tenantId,
                'hr_artisan_performance',
                'HR Artisan Performance Agent',
                'Artisan Performance Summary',
                'No artisan users found for this tenant.',
                'info',
                ['metrics' => [], 'findings' => []],
                ['management', 'maintenance']
            );
            return;
        }

        $metrics = $this->buildMetrics($tenantId, $artisans, $range);
        $findings = $this->buildFindings($metrics, $range);
        $summary = $this->summarize(
            $metrics,
            $findings,
            config('ai_agents.prompts.hr_artisan_performance')
        );

        $severity = $this->highestSeverity($findings, 'warn');
        $suggestedAction = $this->suggestedNextAction($findings);

        $payload = [
            'metrics' => $metrics,
            'findings' => $findings,
            'suggested_next_action' => $suggestedAction,
        ];

        $this->recordNotification(
            $tenantId,
            'hr_artisan_performance',
            'HR Artisan Performance Agent',
            'Artisan Performance Summary',
            $summary,
            $severity,
            $payload,
            ['management', 'maintenance']
        );

        $this->sendTelegramEvent(
            $tenantId,
            'agent.hr_artisan_performance',
            $severity,
            'Artisan Performance Summary',
            $summary,
            [
                'details' => $this->formatFindingsList($findings, 8),
                'title' => 'ARTISAN PERFORMANCE',
            ]
        );

        foreach ($findings as $finding) {
            if (($finding['severity'] ?? '') !== 'alarm') {
                continue;
            }

            $alert = "Artisan Performance ALERT\n" .
                '- ' . ($finding['title'] ?? 'Alert') . "\n" .
                ($finding['ref'] ? 'Artisan: ' . $finding['ref'] . "\n" : '') .
                ($finding['recommendation'] ? 'Recommendation: ' . $finding['recommendation'] : '');

            $this->recordNotification(
                $tenantId,
                'hr_artisan_performance',
                'HR Artisan Performance Agent',
                'Artisan Performance Alert',
                $alert,
                'alarm',
                ['finding' => $finding],
                ['management', 'maintenance']
            );

            $this->sendTelegramEvent(
                $tenantId,
                'agent.hr_artisan_performance',
                'alarm',
                'Artisan Performance Alert',
                $alert,
                [
                    'details' => [$finding['title'] ?? 'Alert'],
                    'title' => 'ARTISAN ALERT',
                ]
            );
        }
    }

    private function buildMetrics(int $tenantId, Collection $artisans, array $range): array
    {
        $metrics = [];
        $availableHours = (float) config('hr_agent.available_hours_per_week', 40);
        $defaultHoursPerWo = (float) config('hr_agent.default_hours_per_wo', 2);

        foreach ($artisans as $artisan) {
            $assigned = WorkOrder::where('tenant_id', $tenantId)
                ->where('assigned_to', $artisan->id)
                ->whereBetween('created_at', [$range['start'], $range['end']])
                ->count();

            $completed = WorkOrder::where('tenant_id', $tenantId)
                ->where('assigned_to', $artisan->id)
                ->whereNotNull('completed_at')
                ->whereBetween('completed_at', [$range['start'], $range['end']])
                ->get();

            $avgClose = null;
            if ($completed->isNotEmpty()) {
                $durations = $completed->map(function ($order) {
                    return $order->created_at && $order->completed_at
                        ? $order->created_at->diffInHours($order->completed_at)
                        : null;
                })->filter();
                if ($durations->isNotEmpty()) {
                    $avgClose = round($durations->sum() / $durations->count(), 1);
                }
            }

            $laborHours = $this->laborHoursForOrders($tenantId, $completed->pluck('id')->all());
            if ($laborHours <= 0) {
                $laborHours = $completed->count() * $defaultHoursPerWo;
            }

            $utilization = $availableHours > 0 ? round($laborHours / $availableHours, 2) : null;

            $evidenceRate = $this->evidenceComplianceRate($tenantId, $completed);

            $metrics[] = [
                'artisan_id' => $artisan->id,
                'artisan_name' => $artisan->name ?? $artisan->email,
                'assigned' => $assigned,
                'completed' => $completed->count(),
                'avg_close_hours' => $avgClose,
                'utilization' => $utilization,
                'compliance_rate' => $evidenceRate,
                'repeat_failures' => $this->repeatFailures($tenantId, $artisan->id),
            ];
        }

        return $metrics;
    }

    private function buildFindings(array $metrics, array $range): array
    {
        $findings = [];
        $warnUtil = (float) config('hr_agent.utilization_warn', 0.5);
        $alarmUtil = (float) config('hr_agent.utilization_alarm', 0.3);
        $warnCompliance = (float) config('hr_agent.compliance_warn', 0.7);
        $alarmCompliance = (float) config('hr_agent.compliance_alarm', 0.5);
        $overload = (float) config('hr_agent.overload_threshold', 1.5);

        foreach ($metrics as $row) {
            $util = $row['utilization'];
            if ($util !== null && $util < $alarmUtil) {
                $findings[] = $this->finding(
                    'alarm',
                    'Low utilization (<30%)',
                    $row['artisan_name'],
                    'Review workload and attendance.'
                );
            } elseif ($util !== null && $util < $warnUtil) {
                $findings[] = $this->finding(
                    'warn',
                    'Low utilization (<50%)',
                    $row['artisan_name'],
                    'Rebalance assignments or review leave status.'
                );
            }

            $compliance = $row['compliance_rate'];
            if ($compliance !== null && $compliance < $alarmCompliance) {
                $findings[] = $this->finding(
                    'alarm',
                    'Checklist/evidence compliance below 50%',
                    $row['artisan_name'],
                    'Require evidence uploads for closed work orders.'
                );
            } elseif ($compliance !== null && $compliance < $warnCompliance) {
                $findings[] = $this->finding(
                    'warn',
                    'Checklist/evidence compliance below 70%',
                    $row['artisan_name'],
                    'Remind to attach job cards and photos.'
                );
            }

            if ($row['repeat_failures'] >= 2) {
                $findings[] = $this->finding(
                    'alarm',
                    'Repeat failure pattern detected',
                    $row['artisan_name'],
                    'Review root causes and reinforce QA checks.'
                );
            } elseif ($row['repeat_failures'] === 1) {
                $findings[] = $this->finding(
                    'warn',
                    'Repeat failure detected',
                    $row['artisan_name'],
                    'Schedule coaching or targeted inspection.'
                );
            }
        }

        $utilization = array_column($metrics, 'utilization');
        $minUtil = $utilization ? min(array_filter($utilization, fn ($value) => $value !== null)) : null;
        $maxUtil = $utilization ? max(array_filter($utilization, fn ($value) => $value !== null)) : null;

        if ($maxUtil !== null && $minUtil !== null && $maxUtil > $overload && $minUtil < $warnUtil) {
            $findings[] = $this->finding(
                'warn',
                'Work allocation imbalance',
                null,
                'Balance workloads to avoid overload and idle time.'
            );
        }

        return $findings;
    }

    private function artisanUsers(int $tenantId): Collection
    {
        $keywords = config('hr_agent.role_keywords', ['technician', 'artisan', 'mechanic', 'maintenance']);
        return User::where('tenant_id', $tenantId)
            ->whereHas('roles', function ($q) use ($keywords) {
                foreach ($keywords as $index => $keyword) {
                    if ($index === 0) {
                        $q->where('name', 'like', '%' . $keyword . '%');
                    } else {
                        $q->orWhere('name', 'like', '%' . $keyword . '%');
                    }
                }
            })
            ->get();
    }

    private function evidenceComplianceRate(int $tenantId, Collection $workOrders): ?float
    {
        if ($workOrders->isEmpty()) {
            return null;
        }

        $evidenceCount = 0;
        foreach ($workOrders as $order) {
            $docs = WorkOrderDocument::where('tenant_id', $tenantId)
                ->where('work_order_id', $order->id)
                ->count();

            if ($docs > 0 || $order->checklist_id) {
                $evidenceCount++;
            }
        }

        return round($evidenceCount / max(1, $workOrders->count()), 2);
    }

    private function laborHoursForOrders(int $tenantId, array $workOrderIds): float
    {
        if (!$workOrderIds) {
            return 0.0;
        }

        $hours = (float) WorkOrderCost::where('tenant_id', $tenantId)
            ->whereIn('work_order_id', $workOrderIds)
            ->where('type', 'labor')
            ->sum('quantity');

        return round($hours, 2);
    }

    private function repeatFailures(int $tenantId, int $artisanId): int
    {
        $windowStart = now()->subDays(30);
        $orders = WorkOrder::where('tenant_id', $tenantId)
            ->where('assigned_to', $artisanId)
            ->whereNotNull('completed_at')
            ->where('completed_at', '>=', $windowStart)
            ->whereNotNull('cause')
            ->get()
            ->groupBy(fn ($order) => $order->asset_id . '|' . strtolower($order->cause));

        $repeats = 0;
        foreach ($orders as $group) {
            if ($group->count() >= 2) {
                $repeats++;
            }
        }

        return $repeats;
    }

    private function weekRange(): array
    {
        $end = now();
        $start = now()->subDays(7)->startOfDay();

        return ['start' => $start, 'end' => $end];
    }

    private function summarize(array $metrics, array $findings, ?string $prompt): string
    {
        $lines = [];
        foreach ($metrics as $row) {
            $lines[] = sprintf(
                '%s: assigned %d, completed %d, utilization %s, compliance %s',
                $row['artisan_name'],
                $row['assigned'],
                $row['completed'],
                $row['utilization'] !== null ? round($row['utilization'] * 100) . '%' : 'N/A',
                $row['compliance_rate'] !== null ? round($row['compliance_rate'] * 100) . '%' : 'N/A'
            );
        }

        $summary = "Artisan Performance Summary\n" .
            implode("\n", array_map(fn ($line) => '- ' . $line, array_slice($lines, 0, 8)));

        $findingLines = $this->formatFindingsList($findings, 6);
        if ($findingLines) {
            $summary .= "\nFindings:\n" . implode("\n", array_map(fn ($line) => '- ' . $line, $findingLines));
        }

        if (!$prompt) {
            return $summary;
        }

        $aiSummary = $this->summarizeWithAi($prompt, $metrics, $findings);
        return $aiSummary ?: $summary;
    }

    private function summarizeWithAi(string $prompt, array $metrics, array $findings): ?string
    {
        $key = config('services.openai.key');
        if (!$key) {
            return null;
        }

        $payload = [
            'model' => config('services.openai.model'),
            'messages' => [
                ['role' => 'system', 'content' => $prompt],
                ['role' => 'user', 'content' => json_encode(['metrics' => $metrics, 'findings' => $findings])],
            ],
            'temperature' => 0.2,
            'max_tokens' => 400,
        ];

        $verify = config('services.openai.verify_ssl', true);
        $verify = filter_var($verify, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $verify = $verify ?? true;

        try {
            $response = Http::withOptions(['verify' => $verify])
                ->withToken($key)
                ->timeout((int) config('services.openai.timeout', 20))
                ->post(config('services.openai.endpoint'), $payload);
        } catch (\Throwable $e) {
            return null;
        }

        if ($response->failed()) {
            return null;
        }

        return trim((string) data_get($response->json(), 'choices.0.message.content'));
    }

    private function highestSeverity(array $findings, string $default = 'info'): string
    {
        $rank = ['info' => 1, 'warn' => 2, 'alarm' => 3];
        $highest = $default;
        foreach ($findings as $finding) {
            $severity = $finding['severity'] ?? $default;
            if (($rank[$severity] ?? 1) > ($rank[$highest] ?? 1)) {
                $highest = $severity;
            }
        }

        return $highest;
    }

    private function suggestedNextAction(array $findings): string
    {
        foreach ($findings as $finding) {
            if (($finding['severity'] ?? '') !== 'alarm') {
                continue;
            }
            return 'REVIEW_SUPERVISOR';
        }

        return 'NONE';
    }

    private function formatFindingsList(array $findings, int $limit): array
    {
        $lines = [];
        foreach (array_slice($findings, 0, $limit) as $finding) {
            $line = $finding['title'] ?? 'Finding';
            if (!empty($finding['ref'])) {
                $line .= ' (' . $finding['ref'] . ')';
            }
            $lines[] = $line;
        }

        return $lines;
    }

    private function finding(string $severity, string $title, ?string $ref, string $recommendation): array
    {
        return [
            'severity' => $severity,
            'title' => $title,
            'ref' => $ref,
            'recommendation' => $recommendation,
        ];
    }

    private function recordNotification(
        int $tenantId,
        string $agentKey,
        string $agentName,
        string $title,
        string $summary,
        string $severity,
        array $payload,
        array $recipients
    ): AgentNotification {
        return AgentNotification::create([
            'tenant_id' => $tenantId,
            'agent_key' => $agentKey,
            'agent_name' => $agentName,
            'title' => $title,
            'summary' => $summary,
            'severity' => $severity,
            'payload' => array_merge($payload, ['prompt' => config('ai_agents.prompts.' . $agentKey)]),
            'recipients' => $recipients,
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    private function sendTelegramEvent(
        int $tenantId,
        string $eventType,
        string $severity,
        string $title,
        string $message,
        array $payload
    ): void {
        $this->events->createEvent($tenantId, $eventType, $severity, array_merge($payload, [
            'title' => $title,
        ]), [
            'title' => $title,
            'message' => $message,
        ]);
    }
}
