<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\ImportBatch;
use App\Models\ImportRow;
use App\Models\Maintenance;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Support\XlsxReader;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Carbon;

class ImportService
{
    public function createBatch(UploadedFile $file, int $tenantId, ?int $userId = null, string $importType = 'assets'): ImportBatch
    {
        $path = $file->store('imports');

        return ImportBatch::create([
            'tenant_id' => $tenantId,
            'created_by' => $userId,
            'status' => 'uploaded',
            'file_path' => $path,
            'file_type' => $file->getClientOriginalExtension(),
            'import_type' => $importType,
        ]);
    }

    public function processBatch(ImportBatch $batch): ImportBatch
    {
        $batch->update(['status' => 'processing']);

        $summary = [
            'created' => 0,
            'updated' => 0,
            'duplicates' => 0,
            'errors' => 0,
        ];

        $path = $batch->file_path;
        if (!$path || !Storage::exists($path)) {
            $batch->update([
                'status' => 'failed',
                'summary' => ['errors' => 1, 'message' => 'Import file missing.'],
            ]);
            return $batch;
        }

        $extension = strtolower($batch->file_type ?? pathinfo($path, PATHINFO_EXTENSION));
        if (!in_array($extension, ['csv', 'xlsx'], true)) {
            $batch->update([
                'status' => 'failed',
                'summary' => ['errors' => 1, 'message' => 'Only CSV/XLSX is supported in this build.'],
            ]);
            return $batch;
        }

        $rows = $extension === 'csv'
            ? $this->parseCsv(Storage::path($path))
            : $this->parseXlsx(Storage::path($path), $batch->import_type);
        foreach ($rows as $row) {
            $result = $this->processRow($batch, $row);
            $summary[$result] = ($summary[$result] ?? 0) + 1;
        }

        $batch->update([
            'status' => 'completed',
            'summary' => $summary,
            'ai_summary' => $this->buildAiSummary($batch->import_type, $summary),
            'processed_at' => now(),
        ]);

        return $batch;
    }

    protected function parseCsv(string $path): array
    {
        $handle = fopen($path, 'r');
        if (!$handle) {
            return [];
        }

        $headers = [];
        $rows = [];

        while (($data = fgetcsv($handle, 0, ',')) !== false) {
            if (empty($headers)) {
                $headers = $this->normalizeHeaders($data);
                continue;
            }

            $row = [];
            foreach ($headers as $index => $header) {
                $row[$header] = $data[$index] ?? null;
            }
            $rows[] = $row;
        }

        fclose($handle);
        return $rows;
    }

    protected function parseXlsx(string $path, ?string $importType = null): array
    {
        $sheetName = match ($importType) {
            'assets' => 'Asset Register',
            'service_sheets' => 'Service Sheets',
            'invoices' => 'Invoices',
            default => null,
        };

        $rows = XlsxReader::readSheetRows($path, $sheetName);
        if (!$rows) {
            return [];
        }

        $normalized = [];
        foreach ($rows as $row) {
            $record = [];
            foreach ($row as $header => $value) {
                $record[$this->normalizeHeader((string) $header)] = $value;
            }
            $normalized[] = $record;
        }

        return $normalized;
    }

    protected function normalizeHeaders(array $headers): array
    {
        return array_map(function ($header) {
            return $this->normalizeHeader((string) $header);
        }, $headers);
    }

    protected function normalizeHeader(string $header): string
    {
        $header = strtolower(trim($header));
        $header = str_replace([' ', '-', '/'], '_', $header);
        return $header;
    }

    protected function mapRow(array $row): ?array
    {
        $map = [
            'asset_tag' => ['asset_tag', 'tag', 'asset_id'],
            'name' => ['name', 'asset_name'],
            'make' => ['make', 'brand'],
            'model' => ['model'],
            'serial_number' => ['serial_number', 'serial', 'serial_no'],
            'year' => ['year'],
            'category' => ['category', 'class', 'asset_group'],
            'asset_type' => ['asset_type', 'type'],
            'meter_type' => ['meter_type', 'meter'],
            'maintenance_interval' => ['maintenance_interval', 'pm_interval', 'service_frequency'],
        ];

        $mapped = [];
        foreach ($map as $target => $candidates) {
            foreach ($candidates as $candidate) {
                if (array_key_exists($candidate, $row) && $row[$candidate] !== null && $row[$candidate] !== '') {
                    $mapped[$target] = $this->cleanValue($target, $row[$candidate]);
                    break;
                }
            }
        }

        if (empty($mapped['name']) && empty($mapped['asset_tag'])) {
            return null;
        }

        return $mapped;
    }

    protected function upsertAsset(int $tenantId, array $data): string
    {
        $query = Asset::query()->where('tenant_id', $tenantId);
        if (!empty($data['asset_tag'])) {
            $query->where('asset_tag', $data['asset_tag']);
        } else {
            $query->where('name', $data['name']);
        }

        $asset = $query->first();
        if ($asset) {
            $asset->fill($data);
            if (!$asset->isDirty()) {
                return 'duplicates';
            }
            $asset->save();
            return 'updated';
        }

        $data['tenant_id'] = $tenantId;
        Asset::create($data);
        return 'created';
    }

    protected function processRow(ImportBatch $batch, array $row): string
    {
        $importType = $batch->import_type ?? 'assets';
        $mapped = match ($importType) {
            'service_sheets' => $this->mapServiceSheetRow($row),
            'invoices' => $this->mapInvoiceRow($row),
            default => $this->mapRow($row),
        };

        $rowStatus = $mapped ? 'mapped' : 'errors';

        $importRow = ImportRow::create([
            'tenant_id' => $batch->tenant_id,
            'batch_id' => $batch->id,
            'raw' => $row,
            'mapped' => $mapped,
            'status' => $rowStatus,
            'error_message' => $mapped ? null : 'Missing required fields.',
        ]);

        if (!$mapped) {
            return 'errors';
        }

        $result = match ($importType) {
            'service_sheets' => $this->createWorkOrderFromServiceSheet($batch->tenant_id, $mapped),
            'invoices' => $this->applyInvoiceCost($batch->tenant_id, $mapped),
            default => $this->upsertAsset($batch->tenant_id, $mapped),
        };

        $importRow->update(['status' => $result]);

        return $result;
    }

    protected function mapServiceSheetRow(array $row): ?array
    {
        $map = [
            'asset_tag' => ['asset_tag', 'tag', 'asset_id'],
            'asset_name' => ['asset_name', 'asset', 'name'],
            'description' => ['description', 'issue', 'work_done'],
            'completed_at' => ['completed_at', 'completed_date', 'service_date', 'date'],
            'meter_reading' => ['meter_reading', 'hours', 'km'],
            'labor_cost' => ['labor_cost', 'labour_cost'],
            'parts_cost' => ['parts_cost'],
            'contractor_cost' => ['contractor_cost'],
            'status' => ['status'],
            'work_order_ref' => ['work_order_ref', 'work_order', 'wo'],
            'root_cause' => ['root_cause', 'cause'],
        ];

        $mapped = [];
        foreach ($map as $target => $candidates) {
            foreach ($candidates as $candidate) {
                if (array_key_exists($candidate, $row) && $row[$candidate] !== null && $row[$candidate] !== '') {
                    $mapped[$target] = $this->cleanValue($target, $row[$candidate]);
                    break;
                }
            }
        }

        if (empty($mapped['asset_tag']) && empty($mapped['asset_name'])) {
            return null;
        }

        return $mapped;
    }

    protected function mapInvoiceRow(array $row): ?array
    {
        $map = [
            'asset_tag' => ['asset_tag', 'asset_id', 'tag'],
            'asset_name' => ['asset_name', 'asset', 'name'],
            'work_order_ref' => ['work_order_ref', 'work_order', 'wo'],
            'amount' => ['amount', 'total', 'cost'],
            'cost_type' => ['cost_type', 'type'],
            'description' => ['description', 'line_item', 'notes'],
            'invoice_number' => ['invoice_number', 'invoice', 'inv_no'],
        ];

        $mapped = [];
        foreach ($map as $target => $candidates) {
            foreach ($candidates as $candidate) {
                if (array_key_exists($candidate, $row) && $row[$candidate] !== null && $row[$candidate] !== '') {
                    $mapped[$target] = $this->cleanValue($target, $row[$candidate]);
                    break;
                }
            }
        }

        if (empty($mapped['work_order_ref']) && empty($mapped['asset_tag']) && empty($mapped['asset_name'])) {
            return null;
        }

        return $mapped;
    }

    protected function createWorkOrderFromServiceSheet(int $tenantId, array $data): string
    {
        $asset = $this->resolveAsset($tenantId, $data);
        if (!$asset) {
            return 'errors';
        }

        $reference = $data['work_order_ref'] ?? null;
        $workOrder = null;

        if ($reference) {
            $workOrder = WorkOrder::where('tenant_id', $tenantId)
                ->where('reference_code', $reference)
                ->first();
        }

        if (!$workOrder) {
            $workOrder = WorkOrder::create([
                'tenant_id' => $tenantId,
                'asset_id' => $asset->id,
                'reference_code' => $reference,
                'source' => 'service_sheet',
                'type' => 'corrective',
                'status' => 'completed',
                'description' => $data['description'] ?? null,
                'cause' => $data['root_cause'] ?? null,
                'completed_at' => $data['completed_at'] ? Carbon::parse($data['completed_at']) : now(),
            ]);
        } else {
            $workOrder->update([
                'description' => $data['description'] ?? $workOrder->description,
                'cause' => $data['root_cause'] ?? $workOrder->cause,
                'status' => $data['status'] ?? $workOrder->status,
            ]);
        }

        Maintenance::create([
            'tenant_id' => $tenantId,
            'asset_id' => $asset->id,
            'scheduled_date' => now()->subDays(7),
            'completed_date' => $data['completed_at'] ? Carbon::parse($data['completed_at']) : now(),
            'status' => 'completed',
            'description' => $data['description'] ?? null,
            'usage_hours' => $this->numericValue($data['meter_reading'] ?? null),
            'meter_reading' => $this->numericValue($data['meter_reading'] ?? null),
        ]);

        $this->applyCosts($tenantId, $workOrder, $data);
        app(\App\Services\AiMaintenanceInsightsService::class)->analyzeAsset($asset->id, $tenantId);

        return $reference ? 'updated' : 'created';
    }

    protected function applyInvoiceCost(int $tenantId, array $data): string
    {
        $asset = $this->resolveAsset($tenantId, $data);
        $workOrder = null;
        $reference = $data['work_order_ref'] ?? null;

        if ($reference) {
            $workOrder = WorkOrder::where('tenant_id', $tenantId)
                ->where('reference_code', $reference)
                ->first();
        }

        if (!$workOrder && $asset) {
            $workOrder = WorkOrder::create([
                'tenant_id' => $tenantId,
                'asset_id' => $asset->id,
                'reference_code' => $reference,
                'source' => 'invoice',
                'type' => 'corrective',
                'status' => 'completed',
                'description' => $data['description'] ?? 'Invoice import',
                'completed_at' => now(),
            ]);
        }

        if (!$workOrder) {
            return 'errors';
        }

        $amount = $this->numericValue($data['amount'] ?? null);
        WorkOrderCost::create([
            'tenant_id' => $tenantId,
            'work_order_id' => $workOrder->id,
            'type' => $data['cost_type'] ?? 'contractor',
            'description' => $data['description'] ?? ($data['invoice_number'] ?? 'Invoice'),
            'quantity' => 1,
            'unit_cost' => $amount,
            'total_cost' => $amount,
        ]);

        $this->refreshWorkOrderTotal($workOrder);
        app(\App\Services\AiMaintenanceInsightsService::class)->analyzeAsset($workOrder->asset_id, $tenantId);

        return $reference ? 'updated' : 'created';
    }

    protected function applyCosts(int $tenantId, WorkOrder $workOrder, array $data): void
    {
        $costs = [
            'labor_cost' => 'labor',
            'parts_cost' => 'parts',
            'contractor_cost' => 'contractor',
        ];

        foreach ($costs as $field => $type) {
            $amount = $this->numericValue($data[$field] ?? null);
            if ($amount === null) {
                continue;
            }

            WorkOrderCost::create([
                'tenant_id' => $tenantId,
                'work_order_id' => $workOrder->id,
                'type' => $type,
                'description' => ucfirst($type) . ' from service sheet',
                'quantity' => 1,
                'unit_cost' => $amount,
                'total_cost' => $amount,
            ]);
        }

        $this->refreshWorkOrderTotal($workOrder);
    }

    protected function refreshWorkOrderTotal(WorkOrder $workOrder): void
    {
        $total = WorkOrderCost::where('work_order_id', $workOrder->id)->sum('total_cost');
        $workOrder->update(['total_cost' => $total]);
    }

    protected function resolveAsset(int $tenantId, array $data): ?Asset
    {
        if (!empty($data['asset_tag'])) {
            $asset = Asset::where('tenant_id', $tenantId)
                ->where('asset_tag', $data['asset_tag'])
                ->first();
            if ($asset) {
                return $asset;
            }
        }

        if (!empty($data['asset_name'])) {
            return Asset::where('tenant_id', $tenantId)
                ->where('name', $data['asset_name'])
                ->first();
        }

        return null;
    }

    protected function cleanValue(string $field, mixed $value): mixed
    {
        if (is_string($value)) {
            $value = trim($value);
        }

        if (in_array($field, ['asset_tag', 'serial_number', 'work_order_ref', 'invoice_number'], true)) {
            return strtoupper((string) $value);
        }

        if (in_array($field, ['maintenance_interval', 'labor_cost', 'parts_cost', 'contractor_cost', 'amount', 'meter_reading'], true)) {
            return $this->numericValue($value);
        }

        return $value;
    }

    protected function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    protected function buildAiSummary(string $importType, array $summary): string
    {
        $label = match ($importType) {
            'service_sheets' => 'Service sheet ingestion',
            'invoices' => 'Invoice posting',
            default => 'Asset register import',
        };

        return sprintf(
            '%s completed. Created %d, updated %d, duplicates %d, errors %d.',
            $label,
            $summary['created'] ?? 0,
            $summary['updated'] ?? 0,
            $summary['duplicates'] ?? 0,
            $summary['errors'] ?? 0
        );
    }
}
