<?php

namespace App\Services;

use App\Models\DocumentIntakeRow;
use App\Models\GoodsReceipt;
use App\Models\GoodsReceiptItem;
use App\Models\NotificationOutbox;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestItem;
use App\Models\PmSchedule;
use App\Models\ReconciliationLink;
use App\Models\Tenant;
use App\Models\WorkOrder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class LinkingReconciliationService
{
    public function runAll(bool $dryRun = false): array
    {
        $summary = [];
        Tenant::query()->select('id')->chunk(100, function ($tenants) use (&$summary, $dryRun) {
            foreach ($tenants as $tenant) {
                $summary[$tenant->id] = $this->runForTenant((int) $tenant->id, $dryRun);
            }
        });

        return $summary;
    }

    public function runForTenant(int $tenantId, bool $dryRun = false): array
    {
        $summary = [
            'po_to_pr' => 0,
            'grn_to_po' => 0,
            'invoice_to_po' => 0,
            'pm_to_wo' => 0,
            'suggestions' => 0,
            'auto_applied' => 0,
        ];

        DB::transaction(function () use ($tenantId, $dryRun, &$summary) {
            $summary['po_to_pr'] += $this->linkPurchaseOrdersToRequests($tenantId, $dryRun, $summary);
            $summary['grn_to_po'] += $this->linkGoodsReceiptsToOrders($tenantId, $dryRun, $summary);
            $summary['invoice_to_po'] += $this->linkInvoicesToOrders($tenantId, $dryRun, $summary);
            $summary['pm_to_wo'] += $this->linkWorkOrdersToPmSchedules($tenantId, $dryRun, $summary);
        });

        return $summary;
    }

    public function applyLink(ReconciliationLink $link, ?int $userId = null): array
    {
        if ($link->status === 'applied') {
            return ['status' => 'already_applied'];
        }

        $result = ['status' => 'applied'];

        if ($link->source_type === 'purchase_order' && $link->target_type === 'purchase_request') {
            $order = PurchaseOrder::where('tenant_id', $link->tenant_id)->find($link->source_id);
            if ($order && !$order->purchase_request_id) {
                $order->purchase_request_id = $link->target_id;
                $order->save();
            }
        }

        if ($link->source_type === 'goods_receipt' && $link->target_type === 'purchase_order') {
            $receipt = GoodsReceipt::where('tenant_id', $link->tenant_id)->find($link->source_id);
            if ($receipt && !$receipt->purchase_order_id) {
                $receipt->purchase_order_id = $link->target_id;
                $receipt->save();
            }
        }

        $link->status = 'applied';
        $link->applied_by = $userId;
        $link->applied_at = now();
        $link->save();

        return $result;
    }

    private function linkPurchaseOrdersToRequests(int $tenantId, bool $dryRun, array &$summary): int
    {
        $orders = PurchaseOrder::where('tenant_id', $tenantId)
            ->whereNull('purchase_request_id')
            ->with('items')
            ->get();

        $requests = PurchaseRequest::where('tenant_id', $tenantId)
            ->with('items')
            ->get();

        $count = 0;
        foreach ($orders as $order) {
            $best = $this->bestRequestMatch($order, $requests);
            if (!$best) {
                continue;
            }

            [$request, $confidence, $reason, $meta] = $best;
            $link = $this->recordLink(
                $tenantId,
                'purchase_order',
                $order->id,
                'purchase_request',
                $request->id,
                $confidence,
                $reason,
                $meta,
                $dryRun
            );

            if ($link->status === 'applied') {
                $summary['auto_applied']++;
            } else {
                $summary['suggestions']++;
                $this->notifyIfNeeded($link);
            }

            if ($link->status === 'applied' && !$dryRun) {
                if (!$order->purchase_request_id) {
                    $order->purchase_request_id = $request->id;
                    $order->save();
                }
            }

            $count++;
        }

        return $count;
    }

    private function linkGoodsReceiptsToOrders(int $tenantId, bool $dryRun, array &$summary): int
    {
        $receipts = GoodsReceipt::where('tenant_id', $tenantId)
            ->whereNull('purchase_order_id')
            ->with('items')
            ->get();

        $orders = PurchaseOrder::where('tenant_id', $tenantId)
            ->with('items')
            ->get();

        $count = 0;
        foreach ($receipts as $receipt) {
            $best = $this->bestOrderMatchForReceipt($receipt, $orders);
            if (!$best) {
                continue;
            }

            [$order, $confidence, $reason, $meta] = $best;
            $link = $this->recordLink(
                $tenantId,
                'goods_receipt',
                $receipt->id,
                'purchase_order',
                $order->id,
                $confidence,
                $reason,
                $meta,
                $dryRun
            );

            if ($link->status === 'applied') {
                $summary['auto_applied']++;
            } else {
                $summary['suggestions']++;
                $this->notifyIfNeeded($link);
            }

            if ($link->status === 'applied' && !$dryRun) {
                if (!$receipt->purchase_order_id) {
                    $receipt->purchase_order_id = $order->id;
                    $receipt->save();
                }
            }

            $count++;
        }

        return $count;
    }

    private function linkInvoicesToOrders(int $tenantId, bool $dryRun, array &$summary): int
    {
        $rows = DocumentIntakeRow::where('tenant_id', $tenantId)
            ->whereHas('batch', fn ($query) => $query->where('doc_type', 'invoice'))
            ->get();

        if ($rows->isEmpty()) {
            return 0;
        }

        $orders = PurchaseOrder::where('tenant_id', $tenantId)->get();
        $receipts = GoodsReceipt::where('tenant_id', $tenantId)->get();

        $count = 0;
        foreach ($rows as $row) {
            $data = $row->data ?? [];
            $poNumber = $this->stringValue($data['po_number'] ?? $data['po'] ?? $data['purchase_order'] ?? null);
            $vendor = $this->stringValue($data['vendor'] ?? $data['supplier'] ?? null);
            $amount = $this->numericValue($data['amount'] ?? null);
            $grnRef = $this->stringValue($data['grn_ref'] ?? $data['grn'] ?? null);

            $orderMatch = null;
            if ($poNumber) {
                $orderMatch = $orders->first(fn ($order) => $this->same($order->po_number, $poNumber));
            }

            if (!$orderMatch && $vendor && $amount) {
                $orderMatch = $orders->first(function ($order) use ($vendor, $amount) {
                    if (!$order->vendor_name || !$order->total_estimated_cost) {
                        return false;
                    }
                    return $this->same($order->vendor_name, $vendor) && abs((float) $order->total_estimated_cost - $amount) < 1;
                });
            }

            if (!$orderMatch) {
                continue;
            }

            $confidence = $poNumber ? 0.95 : 0.7;
            $reason = $poNumber ? 'Invoice references PO number.' : 'Invoice vendor/amount matches PO.';
            $meta = [
                'invoice_number' => $data['invoice_number'] ?? null,
                'vendor' => $vendor,
                'amount' => $amount,
                'evidence' => $poNumber ? ['po_number' => $poNumber] : ['vendor' => $vendor, 'amount' => $amount],
            ];

            $link = $this->recordLink(
                $tenantId,
                'invoice_row',
                $row->id,
                'purchase_order',
                $orderMatch->id,
                $confidence,
                $reason,
                $meta,
                $dryRun
            );

            if ($link->status === 'applied') {
                $summary['auto_applied']++;
            } else {
                $summary['suggestions']++;
                $this->notifyIfNeeded($link);
            }

            if ($grnRef) {
                $receipt = $receipts->first(fn ($item) => $this->same($item->reference, $grnRef));
                if ($receipt) {
                    $this->recordLink(
                        $tenantId,
                        'invoice_row',
                        $row->id,
                        'goods_receipt',
                        $receipt->id,
                        0.9,
                        'Invoice references GRN.' ,
                        ['evidence' => ['grn_ref' => $grnRef]],
                        $dryRun
                    );
                }
            }

            $count++;
        }

        return $count;
    }

    private function linkWorkOrdersToPmSchedules(int $tenantId, bool $dryRun, array &$summary): int
    {
        $workOrders = WorkOrder::where('tenant_id', $tenantId)
            ->where(function ($query) {
                $query->where('source', 'pm')->orWhere('type', 'preventive');
            })
            ->get();

        if ($workOrders->isEmpty()) {
            return 0;
        }

        $schedules = PmSchedule::where('tenant_id', $tenantId)->get();
        if ($schedules->isEmpty()) {
            return 0;
        }

        $count = 0;
        foreach ($workOrders as $workOrder) {
            $best = $this->bestScheduleMatch($workOrder, $schedules);
            if (!$best) {
                continue;
            }

            [$schedule, $confidence, $reason, $meta] = $best;
            $link = $this->recordLink(
                $tenantId,
                'work_order',
                $workOrder->id,
                'pm_schedule',
                $schedule->id,
                $confidence,
                $reason,
                $meta,
                $dryRun
            );

            if ($link->status === 'applied') {
                $summary['auto_applied']++;
            } else {
                $summary['suggestions']++;
                $this->notifyIfNeeded($link);
            }

            $count++;
        }

        return $count;
    }

    private function bestRequestMatch(PurchaseOrder $order, Collection $requests): ?array
    {
        if ($requests->isEmpty()) {
            return null;
        }

        $orderSkus = $order->items->pluck('sku')->filter()->map(fn ($sku) => strtoupper((string) $sku))->unique()->values();
        $orderNames = $order->items->pluck('description')->filter()->map(fn ($desc) => strtoupper((string) $desc))->unique()->values();

        $best = null;
        $bestScore = 0;
        foreach ($requests as $request) {
            $reqSkus = $request->items->pluck('sku')->filter()->map(fn ($sku) => strtoupper((string) $sku))->unique();
            $reqNames = $request->items->pluck('item_name')->filter()->map(fn ($desc) => strtoupper((string) $desc))->unique();

            $skuOverlap = $this->overlapScore($orderSkus, $reqSkus);
            $nameOverlap = $this->overlapScore($orderNames, $reqNames);
            $score = max($skuOverlap, $nameOverlap);

            if ($order->vendor_name) {
                $preferred = $request->items->pluck('preferred_vendor')->filter()->unique();
                $selected = $request->items->pluck('selected_supplier')->filter()->unique();
                if ($preferred->contains(fn ($vendor) => $this->same($vendor, $order->vendor_name)) ||
                    $selected->contains(fn ($vendor) => $this->same($vendor, $order->vendor_name))) {
                    $score += 0.15;
                }
            }

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $request;
            }
        }

        if (!$best || $bestScore < (float) config('reconciliation.review_threshold', 0.6)) {
            return null;
        }

        $confidence = min(0.99, $bestScore);
        $meta = [
            'order_number' => $order->po_number,
            'request_code' => $best->request_code,
            'sku_overlap' => round($bestScore, 2),
        ];

        return [$best, $confidence, 'PO items match PR items.', $meta];
    }

    private function bestOrderMatchForReceipt(GoodsReceipt $receipt, Collection $orders): ?array
    {
        $best = null;
        $bestScore = 0;
        foreach ($orders as $order) {
            if ($receipt->reference && $this->same($receipt->reference, $order->po_number)) {
                return [$order, 0.95, 'GRN reference matches PO number.', ['reference' => $receipt->reference]];
            }

            $score = $this->receiptOrderOverlap($receipt, $order);
            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $order;
            }
        }

        if (!$best || $bestScore < (float) config('reconciliation.review_threshold', 0.6)) {
            return null;
        }

        $meta = [
            'receipt' => $receipt->reference,
            'score' => $bestScore,
        ];

        return [$best, min(0.99, $bestScore), 'GRN items overlap PO items.', $meta];
    }

    private function bestScheduleMatch(WorkOrder $workOrder, Collection $schedules): ?array
    {
        $candidates = $schedules->where('asset_id', $workOrder->asset_id);
        if ($candidates->isEmpty()) {
            return null;
        }

        $best = null;
        $bestScore = 0;
        foreach ($candidates as $schedule) {
            $score = 0.5;
            if ($schedule->name && $workOrder->description && str_contains(strtolower($workOrder->description), strtolower($schedule->name))) {
                $score += 0.3;
            }
            if ($schedule->next_due_at && $workOrder->created_at) {
                $diff = $schedule->next_due_at->diffInDays($workOrder->created_at, false);
                if (abs($diff) <= 7) {
                    $score += 0.2;
                }
            }

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $schedule;
            }
        }

        if (!$best || $bestScore < (float) config('reconciliation.review_threshold', 0.6)) {
            return null;
        }

        $meta = [
            'schedule_name' => $best->name,
            'trigger' => $best->schedule_type,
            'interval' => $best->interval_value ? $best->interval_value . ' ' . $best->interval_unit : null,
        ];

        return [$best, min(0.95, $bestScore), 'WO matches PM schedule on asset/time.', $meta];
    }

    private function receiptOrderOverlap(GoodsReceipt $receipt, PurchaseOrder $order): float
    {
        $receiptSkus = $receipt->items->pluck('sku')->filter()->map(fn ($sku) => strtoupper((string) $sku))->unique();
        $orderSkus = $order->items->pluck('sku')->filter()->map(fn ($sku) => strtoupper((string) $sku))->unique();
        $score = $this->overlapScore($receiptSkus, $orderSkus);

        if ($score > 0) {
            return $score;
        }

        $receiptDesc = $receipt->items->pluck('description')->filter()->map(fn ($desc) => strtoupper((string) $desc))->unique();
        $orderDesc = $order->items->pluck('description')->filter()->map(fn ($desc) => strtoupper((string) $desc))->unique();
        return $this->overlapScore($receiptDesc, $orderDesc);
    }

    private function overlapScore(Collection $left, Collection $right): float
    {
        if ($left->isEmpty() || $right->isEmpty()) {
            return 0;
        }

        $intersection = $left->intersect($right)->count();
        $max = max($left->count(), $right->count());

        return $max > 0 ? $intersection / $max : 0;
    }

    private function recordLink(int $tenantId, string $sourceType, int $sourceId, ?string $targetType, ?int $targetId, float $confidence, string $reason, array $meta, bool $dryRun): ReconciliationLink
    {
        $link = ReconciliationLink::firstOrCreate([
            'tenant_id' => $tenantId,
            'source_type' => $sourceType,
            'source_id' => $sourceId,
            'target_type' => $targetType,
            'target_id' => $targetId,
        ], [
            'confidence' => round($confidence, 2),
            'status' => 'suggested',
            'reason' => $reason,
            'meta_json' => $meta,
        ]);

        $threshold = (float) config('reconciliation.auto_apply_threshold', 0.9);
        if (!$dryRun && $confidence >= $threshold && $link->status === 'suggested') {
            $link->status = 'applied';
            $link->applied_at = now();
            $link->save();
        }

        return $link;
    }

    private function notifyIfNeeded(ReconciliationLink $link): void
    {
        if ($link->notified_at) {
            return;
        }

        $message = "Link suggestion (#{$link->id})\n" .
            "Source: {$link->source_type} {$link->source_id}\n" .
            "Target: {$link->target_type} {$link->target_id}\n" .
            "Confidence: {$link->confidence}\n" .
            "Reason: {$link->reason}\n" .
            "Reply: CONFIRM LINK {$link->id} or REJECT LINK {$link->id}";

        NotificationOutbox::create([
            'tenant_id' => $link->tenant_id,
            'channel' => config('reconciliation.telegram_channel', 'telegram'),
            'severity' => 'warn',
            'title' => 'Linking Suggestion',
            'message' => $message,
            'recipients' => config('reconciliation.review_recipients', ['management']),
            'status' => 'queued',
            'available_at' => now(),
        ]);

        $link->notified_at = now();
        $link->notified_channel = config('reconciliation.telegram_channel', 'telegram');
        $link->save();
    }

    private function stringValue(mixed $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        return $value === '' ? null : $value;
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    private function same(?string $left, ?string $right): bool
    {
        if (!$left || !$right) {
            return false;
        }

        return Str::lower(trim($left)) === Str::lower(trim($right));
    }
}
