<?php

namespace App\Services;

use App\Models\AiReport;
use App\Models\Alarm;
use App\Models\Asset;
use App\Models\FuelLog;
use App\Models\InventoryItem;
use App\Models\InventoryLocation;
use App\Models\Maintenance;
use App\Models\Part;
use App\Models\PurchaseOrder;
use App\Models\PurchaseRequest;
use App\Models\Site;
use App\Models\Telemetry;
use App\Models\Tenant;
use App\Models\Tyre;
use App\Models\User;
use App\Models\Vendor;
use App\Models\WorkOrder;
use Carbon\Carbon;
use Illuminate\Support\Str;
use RuntimeException;

class ModuleDataService
{
    public function __construct(private TelegramExportService $exportService)
    {
    }

    public function list(string $module, array $args, int $tenantId): array
    {
        $module = $this->normalizeModule($module);
        $limit = $this->limit($args);

        return match ($module) {
            'assets' => $this->listAssets($tenantId, $args, $limit),
            'workorders' => $this->listWorkOrders($tenantId, $args, $limit),
            'maintenance' => $this->listMaintenance($tenantId, $args, $limit),
            'inventory' => $this->listInventory($tenantId, $args, $limit),
            'fuel' => $this->listFuel($tenantId, $args, $limit),
            'tyres' => $this->listTyres($tenantId, $args, $limit),
            'telemetry' => $this->listTelemetry($tenantId, $args, $limit),
            'alerts' => $this->listAlerts($tenantId, $args, $limit),
            'kpi' => $this->listKpi($tenantId),
            'reports' => $this->listReports($tenantId, $args, $limit),
            'tenants' => $this->listTenants($args, $limit),
            'users' => $this->listUsers($tenantId, $args, $limit),
            'purchase_requests' => $this->listPurchaseRequests($tenantId, $args, $limit),
            'purchase_orders' => $this->listPurchaseOrders($tenantId, $args, $limit),
            'vendors' => $this->listVendors($tenantId, $args, $limit),
            'parts' => $this->listParts($tenantId, $args, $limit),
            'supply' => $this->listSupply($tenantId, $args),
            default => [
                'title' => 'Unknown Module',
                'headers' => [],
                'rows' => [],
                'message' => 'Unsupported module. Try /list module=assets.',
            ],
        };
    }

    public function get(string $module, array $args, int $tenantId): array
    {
        $module = $this->normalizeModule($module);

        return match ($module) {
            'assets' => $this->getAsset($tenantId, $args),
            'workorders' => $this->getWorkOrder($tenantId, $args),
            'maintenance' => $this->getMaintenance($tenantId, $args),
            'inventory' => $this->getInventoryItem($tenantId, $args),
            'fuel' => $this->getFuelLog($tenantId, $args),
            'tyres' => $this->getTyre($tenantId, $args),
            'telemetry' => $this->getTelemetry($tenantId, $args),
            'alerts' => $this->getAlert($tenantId, $args),
            'reports' => $this->getReport($tenantId, $args),
            'tenants' => $this->getTenant($args),
            'users' => $this->getUser($tenantId, $args),
            'purchase_requests' => $this->getPurchaseRequest($tenantId, $args),
            'purchase_orders' => $this->getPurchaseOrder($tenantId, $args),
            'vendors' => $this->getVendor($tenantId, $args),
            'parts' => $this->getPart($tenantId, $args),
            default => [
                'title' => 'Unknown Module',
                'headers' => [],
                'rows' => [],
                'message' => 'Unsupported module. Try /get module=assets tag=ME-EXC-01.',
            ],
        };
    }

    public function export(string $module, array $args, int $tenantId): array
    {
        $format = strtolower($args['format'] ?? 'pdf');
        $module = $this->normalizeModule($module);

        if (!in_array($format, ['pdf', 'xlsx'], true)) {
            return ['message' => 'Unsupported format. Use format=pdf or format=xlsx.'];
        }

        if ($module === 'purchase_requests' && $format === 'pdf') {
            $code = $args['code'] ?? $args['request_code'] ?? $args['pr'] ?? null;
            if ($code) {
                $request = PurchaseRequest::where('tenant_id', $tenantId)
                    ->where('request_code', $code)
                    ->first();
                if (!$request) {
                    return $this->purchaseRequestNotFoundSuggestion($tenantId, $code);
                }
                $tenant = Tenant::find($tenantId);
                if (!$tenant) {
                    return ['message' => 'Tenant not found for export.'];
                }
                $file = $this->exportService->exportPurchaseRequest($request, $tenant);
                return [
                    'message' => "Export ready: {$file['filename']}",
                    'file_path' => $file['path'],
                    'filename' => $file['filename'],
                ];
            }
        }

        if ($module === 'workorders' && $format === 'pdf') {
            $reference = $args['wo'] ?? $args['ref'] ?? $args['reference'] ?? null;
            if ($reference) {
                $order = WorkOrder::where('tenant_id', $tenantId)
                    ->where('reference_code', $reference)
                    ->first();
                if (!$order) {
                    return $this->workOrderNotFoundSuggestion($tenantId, $reference);
                }
                $tenant = Tenant::find($tenantId);
                if (!$tenant) {
                    return ['message' => 'Tenant not found for export.'];
                }
                $file = $this->exportService->exportWorkOrder($order, $tenant);
                return [
                    'message' => "Export ready: {$file['filename']}",
                    'file_path' => $file['path'],
                    'filename' => $file['filename'],
                ];
            }
        }

        $list = $this->list($module, $args + ['limit' => $args['limit'] ?? 200], $tenantId);
        $rows = $list['rows'] ?? [];
        $headers = $list['headers'] ?? [];
        $title = $list['title'] ?? ucfirst($module);

        if (!$rows || !$headers) {
            return ['message' => 'No data available to export. Try /list module=' . $module . ' or adjust filters.'];
        }

        $tenant = Tenant::find($tenantId);
        if (!$tenant) {
            return ['message' => 'Tenant not found for export.'];
        }

        try {
            $file = $this->exportService->exportTable($title, $headers, $rows, $tenant, $format);
        } catch (RuntimeException $e) {
            return ['message' => 'Unable to export. Use format=pdf or format=xlsx and check filters.'];
        }

        return [
            'message' => "Export ready: {$file['filename']}",
            'file_path' => $file['path'],
            'filename' => $file['filename'],
        ];
    }

    private function listAssets(int $tenantId, array $args, int $limit): array
    {
        $builder = Asset::where('tenant_id', $tenantId)->with('site');
        $tag = null;

        if (!empty($args['tag']) || !empty($args['asset_tag'])) {
            $tag = $args['tag'] ?? $args['asset_tag'];
            $builder->where('asset_tag', $tag);
        } elseif (!empty($args['q'])) {
            $query = $args['q'];
            $builder->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                    ->orWhere('asset_tag', 'like', "%{$query}%")
                    ->orWhere('category', 'like', "%{$query}%")
                    ->orWhere('location', 'like', "%{$query}%");
            });
        }

        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }

        if (!empty($args['site'])) {
            $site = Site::where('tenant_id', $tenantId)->where('name', $args['site'])->first();
            if (!$site) {
                return $this->siteNotFoundSuggestion($tenantId, $args['site']);
            }
            $builder->where('site_id', $site->id);
        }

        if (!empty($args['location'])) {
            $builder->where('location', 'like', '%' . $args['location'] . '%');
        }

        $assets = $builder->limit($limit)->get();
        if ($assets->isEmpty() && !empty($tag)) {
            return $this->assetNotFoundSuggestion($tenantId, $tag);
        }
        if ($assets->isEmpty() && !empty($args['location'])) {
            return $this->locationNotFoundSuggestion($tenantId, $args['location']);
        }
        $headers = ['Tag', 'Name', 'Status', 'Location'];
        $rows = $assets->map(fn ($asset) => [
            $asset->asset_tag ?? '-',
            $asset->name ?? '-',
            $asset->status ?? '-',
            $asset->location ?? ($asset->site?->name ?? '-'),
        ])->all();

        return $this->formatList('Assets', $headers, $rows);
    }

    private function listWorkOrders(int $tenantId, array $args, int $limit): array
    {
        $builder = WorkOrder::where('tenant_id', $tenantId)->with('asset');

        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }
        if (!empty($args['priority'])) {
            $builder->where('priority', $args['priority']);
        }

        $orders = $builder->limit($limit)->get();
        $headers = ['Ref', 'Asset', 'Status', 'Priority', 'Due'];
        $rows = $orders->map(function ($order) {
            return [
                $order->reference_code ?? ('WO-' . $order->id),
                $this->assetLabel($order->asset),
                $order->status ?? '-',
                $order->priority ?? '-',
                $order->due_at?->format('Y-m-d') ?? '-',
            ];
        })->all();

        return $this->formatList('Work Orders', $headers, $rows);
    }

    private function listMaintenance(int $tenantId, array $args, int $limit): array
    {
        $builder = Maintenance::where('tenant_id', $tenantId)->with('asset');
        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }

        $items = $builder->limit($limit)->get();
        $headers = ['Asset', 'Status', 'Scheduled', 'Completed'];
        $rows = $items->map(fn ($item) => [
            $this->assetLabel($item->asset),
            $item->status ?? '-',
            $item->scheduled_date?->format('Y-m-d') ?? '-',
            $item->completed_date?->format('Y-m-d') ?? '-',
        ])->all();

        return $this->formatList('Maintenance', $headers, $rows);
    }

    private function listInventory(int $tenantId, array $args, int $limit): array
    {
        $builder = InventoryItem::where('tenant_id', $tenantId)->with(['part', 'location']);
        if (!empty($args['q'])) {
            $query = $args['q'];
            $builder->whereHas('part', function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                    ->orWhere('sku', 'like', "%{$query}%");
            });
        }
        if (!empty($args['location'])) {
            $location = InventoryLocation::where('tenant_id', $tenantId)
                ->where('name', $args['location'])
                ->first();
            if (!$location) {
                return $this->inventoryLocationNotFoundSuggestion($tenantId, $args['location']);
            }
            $builder->where('location_id', $location->id);
        }

        $items = $builder->limit($limit)->get();
        $headers = ['SKU', 'Part', 'Location', 'Qty', 'Min', 'Max'];
        $rows = $items->map(fn ($item) => [
            $item->part?->sku ?? '-',
            $item->part?->name ?? '-',
            $item->location?->name ?? '-',
            $item->quantity ?? 0,
            $item->min_quantity ?? '-',
            $item->max_quantity ?? '-',
        ])->all();

        return $this->formatList('Inventory', $headers, $rows);
    }

    private function listFuel(int $tenantId, array $args, int $limit): array
    {
        $builder = FuelLog::where('tenant_id', $tenantId)->with(['asset', 'site']);
        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['site'])) {
            $site = Site::where('tenant_id', $tenantId)->where('name', $args['site'])->first();
            if (!$site) {
                return $this->siteNotFoundSuggestion($tenantId, $args['site']);
            }
            $builder->where('site_id', $site->id);
        }
        if (!empty($args['from']) || !empty($args['to'])) {
            $this->applyDateRange($builder, 'logged_at', $args['from'] ?? null, $args['to'] ?? null);
        }

        $logs = $builder->limit($limit)->get();
        $headers = ['Asset', 'Qty', 'Fuel Type', 'Cost', 'Date'];
        $rows = $logs->map(fn ($log) => [
            $this->assetLabel($log->asset),
            $log->quantity ?? 0,
            $log->fuel_type ?? '-',
            $log->total_cost ?? '-',
            $log->logged_at?->format('Y-m-d H:i') ?? '-',
        ])->all();

        return $this->formatList('Fuel Logs', $headers, $rows);
    }

    private function listTyres(int $tenantId, array $args, int $limit): array
    {
        $builder = Tyre::where('tenant_id', $tenantId)->with('asset');
        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }

        $tyres = $builder->limit($limit)->get();
        $headers = ['Asset', 'Position', 'Brand', 'Status', 'Tread'];
        $rows = $tyres->map(fn ($tyre) => [
            $this->assetLabel($tyre->asset),
            $tyre->position ?? '-',
            $tyre->brand ?? '-',
            $tyre->status ?? '-',
            $tyre->tread_depth ?? '-',
        ])->all();

        return $this->formatList('Tyres', $headers, $rows);
    }

    private function listTelemetry(int $tenantId, array $args, int $limit): array
    {
        $builder = Telemetry::where('tenant_id', $tenantId)->with('asset');
        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['from']) || !empty($args['to'])) {
            $this->applyDateRange($builder, 'timestamp', $args['from'] ?? null, $args['to'] ?? null);
        }

        $entries = $builder->orderByDesc('timestamp')->limit($limit)->get();
        $headers = ['Time', 'Asset', 'Hours'];
        $rows = $entries->map(fn ($entry) => [
            $entry->timestamp?->format('Y-m-d H:i') ?? '-',
            $this->assetLabel($entry->asset),
            $entry->usage_hours ?? '-',
        ])->all();

        return $this->formatList('Telemetry', $headers, $rows);
    }

    private function listAlerts(int $tenantId, array $args, int $limit): array
    {
        $builder = Alarm::where('tenant_id', $tenantId)->with('asset');
        if (!empty($args['asset'])) {
            $asset = Asset::where('tenant_id', $tenantId)->where('asset_tag', $args['asset'])->first();
            if (!$asset) {
                return $this->assetNotFoundSuggestion($tenantId, $args['asset']);
            }
            $builder->where('asset_id', $asset->id);
        }
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }
        if (!empty($args['severity'])) {
            $builder->where('severity', $args['severity']);
        }

        $alerts = $builder->orderByDesc('triggered_at')->limit($limit)->get();
        $headers = ['Asset', 'Severity', 'Status', 'Message', 'Time'];
        $rows = $alerts->map(fn ($alert) => [
            $this->assetLabel($alert->asset),
            $alert->severity ?? '-',
            $alert->status ?? '-',
            $alert->message ?? '-',
            $alert->triggered_at?->format('Y-m-d H:i') ?? '-',
        ])->all();

        return $this->formatList('Alerts', $headers, $rows);
    }

    private function listKpi(int $tenantId): array
    {
        $assets = Asset::where('tenant_id', $tenantId)->count();
        $activeAssets = Asset::where('tenant_id', $tenantId)->where('status', 'active')->count();
        $openWos = WorkOrder::where('tenant_id', $tenantId)->where('status', 'open')->count();
        $pendingMaintenance = Maintenance::where('tenant_id', $tenantId)->where('status', 'pending')->count();
        $openAlerts = Alarm::where('tenant_id', $tenantId)->where('status', 'open')->count();

        $headers = ['KPI', 'Value', 'Notes'];
        $rows = [
            ['Total Assets', $assets, 'All registered assets'],
            ['Active Assets', $activeAssets, 'Status = active'],
            ['Open Work Orders', $openWos, 'Status = open'],
            ['Pending Maintenance', $pendingMaintenance, 'Status = pending'],
            ['Open Alerts', $openAlerts, 'Status = open'],
        ];

        return $this->formatList('KPI Summary', $headers, $rows);
    }

    private function listReports(int $tenantId, array $args, int $limit): array
    {
        $builder = AiReport::where('tenant_id', $tenantId);
        if (!empty($args['type'])) {
            $builder->where('report_type', $args['type']);
        }

        $reports = $builder->orderByDesc('period_start')->limit($limit)->get();
        $headers = ['Type', 'Period', 'Summary'];
        $rows = $reports->map(function ($report) {
            $period = trim(($report->period_start?->format('Y-m-d') ?? '') . ' - ' . ($report->period_end?->format('Y-m-d') ?? ''));
            return [
                $report->report_type ?? '-',
                $period ?: '-',
                Str::limit(strip_tags($report->content ?? ''), 80),
            ];
        })->all();

        return $this->formatList('Reports', $headers, $rows);
    }

    private function listTenants(array $args, int $limit): array
    {
        $builder = Tenant::query();
        if (!empty($args['q'])) {
            $query = $args['q'];
            $builder->where('name', 'like', "%{$query}%")
                ->orWhere('email', 'like', "%{$query}%");
        }

        $tenants = $builder->limit($limit)->get();
        $headers = ['ID', 'Name', 'Status', 'Email'];
        $rows = $tenants->map(fn ($tenant) => [
            $tenant->id,
            $tenant->name ?? '-',
            $tenant->status ?? '-',
            $tenant->email ?? '-',
        ])->all();

        return $this->formatList('Tenants', $headers, $rows);
    }

    private function listUsers(int $tenantId, array $args, int $limit): array
    {
        $builder = User::where('tenant_id', $tenantId)->with('roles');
        if (!empty($args['q'])) {
            $query = $args['q'];
            $builder->where(function ($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                    ->orWhere('email', 'like', "%{$query}%");
            });
        }
        if (!empty($args['role'])) {
            $builder->whereHas('roles', function ($q) use ($args) {
                $q->where('name', $args['role']);
            });
        }

        $users = $builder->limit($limit)->get();
        $headers = ['Name', 'Email', 'Role'];
        $rows = $users->map(function ($user) {
            $role = $user->roles->first()?->name ?? '-';
            return [$user->name ?? '-', $user->email ?? '-', $role];
        })->all();

        return $this->formatList('Users', $headers, $rows);
    }

    private function listPurchaseRequests(int $tenantId, array $args, int $limit): array
    {
        $builder = PurchaseRequest::where('tenant_id', $tenantId)->with('site');
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }
        if (!empty($args['priority'])) {
            $builder->where('priority', $args['priority']);
        }
        if (!empty($args['department'])) {
            $builder->where('department', 'like', '%' . $args['department'] . '%');
        }
        if (!empty($args['cost_center'])) {
            $builder->where('cost_center', 'like', '%' . $args['cost_center'] . '%');
        }
        if (!empty($args['site'])) {
            $site = Site::where('tenant_id', $tenantId)->where('name', $args['site'])->first();
            if ($site) {
                $builder->where('site_id', $site->id);
            }
        }
        if (!empty($args['code'])) {
            $builder->where('request_code', $args['code']);
        }
        if (!empty($args['request_code'])) {
            $builder->where('request_code', $args['request_code']);
        }
        if (!empty($args['needed_by'])) {
            $builder->whereDate('needed_by', Carbon::parse($args['needed_by']));
        }

        $requests = $builder->limit($limit)->get();
        $headers = ['Code', 'Title', 'Site', 'Status', 'Priority'];
        $rows = $requests->map(fn ($request) => [
            $request->request_code ?? '-',
            $request->title ?? '-',
            $request->site?->name ?? '-',
            $request->status ?? '-',
            $request->priority ?? '-',
        ])->all();

        return $this->formatList('Purchase Requests', $headers, $rows);
    }

    private function listPurchaseOrders(int $tenantId, array $args, int $limit): array
    {
        $builder = PurchaseOrder::where('tenant_id', $tenantId);
        if (!empty($args['status'])) {
            $builder->where('status', $args['status']);
        }
        if (!empty($args['vendor'])) {
            $builder->where('vendor_name', 'like', '%' . $args['vendor'] . '%');
        }
        if (!empty($args['po'])) {
            $builder->where('po_number', $args['po']);
        }

        $orders = $builder->limit($limit)->get();
        $headers = ['PO', 'Vendor', 'Status', 'Total', 'Created'];
        $rows = $orders->map(fn ($order) => [
            $order->po_number ?? '-',
            $order->vendor_name ?? '-',
            $order->status ?? '-',
            $order->total_estimated_cost ?? '-',
            $order->created_at?->format('Y-m-d') ?? '-',
        ])->all();

        return $this->formatList('Purchase Orders', $headers, $rows);
    }

    private function listVendors(int $tenantId, array $args, int $limit): array
    {
        $builder = Vendor::where('tenant_id', $tenantId);
        if (!empty($args['q'])) {
            $query = $args['q'];
            $builder->where('name', 'like', "%{$query}%");
        }

        $vendors = $builder->limit($limit)->get();
        $headers = ['Vendor', 'Contact', 'Phone', 'Email'];
        $rows = $vendors->map(fn ($vendor) => [
            $vendor->name ?? '-',
            $vendor->contact_name ?? '-',
            $vendor->phone ?? '-',
            $vendor->email ?? '-',
        ])->all();

        return $this->formatList('Vendors', $headers, $rows);
    }

    private function listParts(int $tenantId, array $args, int $limit): array
    {
        $builder = Part::where('tenant_id', $tenantId);
        if (!empty($args['q'])) {
            $query = $args['q'];
            $builder->where('name', 'like', "%{$query}%")
                ->orWhere('sku', 'like', "%{$query}%");
        }

        $parts = $builder->limit($limit)->get();
        $headers = ['SKU', 'Name', 'Category', 'Unit'];
        $rows = $parts->map(fn ($part) => [
            $part->sku ?? '-',
            $part->name ?? '-',
            $part->category ?? '-',
            $part->unit ?? '-',
        ])->all();

        return $this->formatList('Parts', $headers, $rows);
    }

    private function listSupply(int $tenantId, array $args): array
    {
        $type = strtolower($args['type'] ?? '');
        if ($type) {
            return $this->list($type, $args, $tenantId);
        }

        $headers = ['Module', 'Count', 'Hint'];
        $rows = [
            ['Purchase Requests', PurchaseRequest::where('tenant_id', $tenantId)->count(), 'Use type=purchase_requests'],
            ['Purchase Orders', PurchaseOrder::where('tenant_id', $tenantId)->count(), 'Use type=purchase_orders'],
            ['Inventory Items', InventoryItem::where('tenant_id', $tenantId)->count(), 'Use type=inventory'],
            ['Vendors', Vendor::where('tenant_id', $tenantId)->count(), 'Use type=vendors'],
        ];

        return $this->formatList('Supply Chain', $headers, $rows);
    }

    private function getAsset(int $tenantId, array $args): array
    {
        $asset = $this->findAsset($tenantId, $args);
        if (!$asset) {
            $needle = $args['tag'] ?? $args['asset_tag'] ?? null;
            return $this->assetNotFoundSuggestion($tenantId, $needle);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Tag', $asset->asset_tag ?? '-'],
            ['Name', $asset->name ?? '-'],
            ['Status', $asset->status ?? '-'],
            ['Category', $asset->category ?? '-'],
            ['Location', $asset->location ?? '-'],
            ['Meter Type', $asset->meter_type ?? '-'],
            ['Meter Reading', $asset->meter_reading ?? '-'],
        ];

        return $this->formatList('Asset Detail', $headers, $rows);
    }

    private function getWorkOrder(int $tenantId, array $args): array
    {
        $reference = $args['ref'] ?? $args['wo'] ?? null;
        $query = WorkOrder::where('tenant_id', $tenantId)->with('asset');
        if ($reference) {
            $query->where('reference_code', $reference);
        } elseif (!empty($args['id'])) {
            $query->where('id', $args['id']);
        }

        $order = $query->first();
        if (!$order) {
            return $this->workOrderNotFoundSuggestion($tenantId, $reference);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Ref', $order->reference_code ?? ('WO-' . $order->id)],
            ['Asset', $this->assetLabel($order->asset)],
            ['Status', $order->status ?? '-'],
            ['Priority', $order->priority ?? '-'],
            ['Due', $order->due_at?->format('Y-m-d') ?? '-'],
            ['Description', $order->description ?? '-'],
        ];

        return $this->formatList('Work Order Detail', $headers, $rows);
    }

    private function getMaintenance(int $tenantId, array $args): array
    {
        $item = Maintenance::where('tenant_id', $tenantId)->with('asset')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$item) {
            return ['message' => 'Maintenance record not found.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Asset', $this->assetLabel($item->asset)],
            ['Status', $item->status ?? '-'],
            ['Scheduled', $item->scheduled_date?->format('Y-m-d') ?? '-'],
            ['Completed', $item->completed_date?->format('Y-m-d') ?? '-'],
            ['Notes', $item->notes ?? '-'],
        ];

        return $this->formatList('Maintenance Detail', $headers, $rows);
    }

    private function getInventoryItem(int $tenantId, array $args): array
    {
        $builder = InventoryItem::where('tenant_id', $tenantId)->with(['part', 'location']);
        if (!empty($args['id'])) {
            $builder->where('id', $args['id']);
        } elseif (!empty($args['sku'])) {
            $builder->whereHas('part', fn ($q) => $q->where('sku', $args['sku']));
        }

        $item = $builder->first();
        if (!$item) {
            return $this->inventoryNotFoundSuggestion($tenantId, $args['sku'] ?? null);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['SKU', $item->part?->sku ?? '-'],
            ['Part', $item->part?->name ?? '-'],
            ['Location', $item->location?->name ?? '-'],
            ['Qty', $item->quantity ?? 0],
            ['Min', $item->min_quantity ?? '-'],
            ['Max', $item->max_quantity ?? '-'],
        ];

        return $this->formatList('Inventory Detail', $headers, $rows);
    }

    private function getFuelLog(int $tenantId, array $args): array
    {
        $log = FuelLog::where('tenant_id', $tenantId)
            ->with(['asset', 'site'])
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$log) {
            return ['message' => 'Fuel log not found.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Asset', $this->assetLabel($log->asset)],
            ['Quantity', $log->quantity ?? '-'],
            ['Fuel Type', $log->fuel_type ?? '-'],
            ['Total Cost', $log->total_cost ?? '-'],
            ['Logged At', $log->logged_at?->format('Y-m-d H:i') ?? '-'],
        ];

        return $this->formatList('Fuel Log Detail', $headers, $rows);
    }

    private function getTyre(int $tenantId, array $args): array
    {
        $tyre = Tyre::where('tenant_id', $tenantId)
            ->with('asset')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$tyre) {
            return ['message' => 'Tyre record not found.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Asset', $this->assetLabel($tyre->asset)],
            ['Position', $tyre->position ?? '-'],
            ['Brand', $tyre->brand ?? '-'],
            ['Status', $tyre->status ?? '-'],
            ['Tread', $tyre->tread_depth ?? '-'],
        ];

        return $this->formatList('Tyre Detail', $headers, $rows);
    }

    private function getTelemetry(int $tenantId, array $args): array
    {
        $entry = Telemetry::where('tenant_id', $tenantId)
            ->with('asset')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$entry) {
            return ['message' => 'Telemetry record not found.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Asset', $this->assetLabel($entry->asset)],
            ['Timestamp', $entry->timestamp?->format('Y-m-d H:i') ?? '-'],
            ['Usage Hours', $entry->usage_hours ?? '-'],
        ];

        return $this->formatList('Telemetry Detail', $headers, $rows);
    }

    private function getAlert(int $tenantId, array $args): array
    {
        $alert = Alarm::where('tenant_id', $tenantId)
            ->with('asset')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$alert) {
            return ['message' => 'Alert not found.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Asset', $this->assetLabel($alert->asset)],
            ['Severity', $alert->severity ?? '-'],
            ['Status', $alert->status ?? '-'],
            ['Message', $alert->message ?? '-'],
            ['Triggered', $alert->triggered_at?->format('Y-m-d H:i') ?? '-'],
        ];

        return $this->formatList('Alert Detail', $headers, $rows);
    }

    private function getReport(int $tenantId, array $args): array
    {
        $report = AiReport::where('tenant_id', $tenantId)
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->first();

        if (!$report) {
            return ['message' => 'Report not found. Try /list module=reports.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Type', $report->report_type ?? '-'],
            ['Period', trim(($report->period_start?->format('Y-m-d') ?? '') . ' - ' . ($report->period_end?->format('Y-m-d') ?? ''))],
            ['Content', Str::limit(strip_tags($report->content ?? ''), 220)],
        ];

        return $this->formatList('Report Detail', $headers, $rows);
    }

    private function getTenant(array $args): array
    {
        $tenant = Tenant::query()
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['name']), fn ($q) => $q->where('name', $args['name']))
            ->first();

        if (!$tenant) {
            return ['message' => 'Tenant not found. Try /list module=tenants.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Name', $tenant->name ?? '-'],
            ['Email', $tenant->email ?? '-'],
            ['Status', $tenant->status ?? '-'],
            ['Subdomain', $tenant->subdomain ?? '-'],
        ];

        return $this->formatList('Tenant Detail', $headers, $rows);
    }

    private function getUser(int $tenantId, array $args): array
    {
        $user = User::where('tenant_id', $tenantId)
            ->with('roles')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['email']), fn ($q) => $q->where('email', $args['email']))
            ->first();

        if (!$user) {
            return $this->userNotFoundSuggestion($tenantId, $args['email'] ?? null);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Name', $user->name ?? '-'],
            ['Email', $user->email ?? '-'],
            ['Role', $user->roles->pluck('name')->implode(', ') ?: '-'],
        ];

        return $this->formatList('User Detail', $headers, $rows);
    }

    private function getPurchaseRequest(int $tenantId, array $args): array
    {
        $request = PurchaseRequest::where('tenant_id', $tenantId)
            ->with('site')
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['code']), fn ($q) => $q->where('request_code', $args['code']))
            ->first();

        if (!$request) {
            return $this->purchaseRequestNotFoundSuggestion($tenantId, $args['code'] ?? null);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Code', $request->request_code ?? '-'],
            ['Title', $request->title ?? '-'],
            ['Site', $request->site?->name ?? '-'],
            ['Status', $request->status ?? '-'],
            ['Priority', $request->priority ?? '-'],
        ];

        return $this->formatList('Purchase Request Detail', $headers, $rows);
    }

    private function getPurchaseOrder(int $tenantId, array $args): array
    {
        $order = PurchaseOrder::where('tenant_id', $tenantId)
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['po']), fn ($q) => $q->where('po_number', $args['po']))
            ->first();

        if (!$order) {
            return ['message' => 'Purchase order not found. Try /list module=purchase_orders.'];
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['PO', $order->po_number ?? '-'],
            ['Vendor', $order->vendor_name ?? '-'],
            ['Status', $order->status ?? '-'],
            ['Total', $order->total_estimated_cost ?? '-'],
            ['Terms', $order->terms ?? '-'],
            ['Delivery', $order->delivery_location ?? '-'],
        ];

        return $this->formatList('Purchase Order Detail', $headers, $rows);
    }
    private function getVendor(int $tenantId, array $args): array
    {
        $vendor = Vendor::where('tenant_id', $tenantId)
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['name']), fn ($q) => $q->where('name', $args['name']))
            ->first();

        if (!$vendor) {
            return $this->vendorNotFoundSuggestion($tenantId, $args['name'] ?? null);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['Name', $vendor->name ?? '-'],
            ['Contact', $vendor->contact_name ?? '-'],
            ['Phone', $vendor->phone ?? '-'],
            ['Email', $vendor->email ?? '-'],
        ];

        return $this->formatList('Vendor Detail', $headers, $rows);
    }

    private function getPart(int $tenantId, array $args): array
    {
        $part = Part::where('tenant_id', $tenantId)
            ->when(!empty($args['id']), fn ($q) => $q->where('id', $args['id']))
            ->when(!empty($args['sku']), fn ($q) => $q->where('sku', $args['sku']))
            ->first();

        if (!$part) {
            return $this->partNotFoundSuggestion($tenantId, $args['sku'] ?? null);
        }

        $headers = ['Field', 'Value'];
        $rows = [
            ['SKU', $part->sku ?? '-'],
            ['Name', $part->name ?? '-'],
            ['Category', $part->category ?? '-'],
            ['Unit', $part->unit ?? '-'],
        ];

        return $this->formatList('Part Detail', $headers, $rows);
    }

    private function formatList(string $title, array $headers, array $rows): array
    {
        if (!$rows) {
            return [
                'title' => $title,
                'headers' => $headers,
                'rows' => [],
                'message' => 'No records found.',
            ];
        }

        $lines = array_map(function ($row) {
            return implode(' | ', array_map(fn ($value) => (string) $value, $row));
        }, array_slice($rows, 0, 8));

        return [
            'title' => $title,
            'headers' => $headers,
            'rows' => $rows,
            'message' => $title . ":\n" . implode("\n", $lines),
        ];
    }

    private function assetLabel(?Asset $asset, ?string $fallbackTag = null): string
    {
        $tag = $asset?->asset_tag ?? $fallbackTag ?? '-';
        $name = $asset?->name ?? null;
        if ($name) {
            return "{$tag} - {$name}";
        }

        return (string) $tag;
    }

    private function normalizeModule(string $module): string
    {
        $module = strtolower(trim($module));

        return match ($module) {
            'workorders', 'work_orders', 'work-orders', 'wo' => 'workorders',
            'purchase-requests', 'purchase_request', 'purchase_requests', 'pr' => 'purchase_requests',
            'purchase-orders', 'purchase_order', 'purchase_orders', 'po' => 'purchase_orders',
            'vendors', 'vendor' => 'vendors',
            'inventory', 'stock' => 'inventory',
            'tyres', 'tires' => 'tyres',
            'alerts', 'alarms' => 'alerts',
            'kpis', 'kpi' => 'kpi',
            'reports', 'report' => 'reports',
            'users', 'user' => 'users',
            'tenants', 'tenant' => 'tenants',
            'parts', 'part' => 'parts',
            'supply', 'supply_chain' => 'supply',
            default => $module,
        };
    }

    private function limit(array $args): int
    {
        $limit = (int) ($args['limit'] ?? 10);
        return max(1, min(200, $limit));
    }

    private function findAsset(int $tenantId, array $args): ?Asset
    {
        if (!empty($args['id'])) {
            return Asset::where('tenant_id', $tenantId)->where('id', $args['id'])->first();
        }
        $tag = $args['tag'] ?? $args['asset_tag'] ?? null;
        if ($tag) {
            return Asset::where('tenant_id', $tenantId)->where('asset_tag', $tag)->first();
        }
        return null;
    }

    private function applyDateRange($builder, string $field, ?string $from, ?string $to): void
    {
        if ($from) {
            $builder->where($field, '>=', Carbon::parse($from));
        }
        if ($to) {
            $builder->where($field, '<=', Carbon::parse($to));
        }
    }

    private function assetNotFoundSuggestion(int $tenantId, ?string $needle): array
    {
        $lines = $this->suggestAssets($tenantId, $needle);
        return $this->notFoundWithSuggestions('Asset not found.', $lines, 'Try /list module=assets.');
    }

    private function locationNotFoundSuggestion(int $tenantId, ?string $needle): array
    {
        $lines = $this->suggestAssetLocations($tenantId, $needle);
        return $this->notFoundWithSuggestions('Location not found.', $lines, 'Try /list module=assets.');
    }

    private function siteNotFoundSuggestion(int $tenantId, ?string $needle): array
    {
        $lines = $this->suggestSites($tenantId, $needle);
        return $this->notFoundWithSuggestions('Site not found.', $lines, 'Try /list module=assets.');
    }

    private function inventoryLocationNotFoundSuggestion(int $tenantId, ?string $needle): array
    {
        $lines = $this->suggestInventoryLocations($tenantId, $needle);
        return $this->notFoundWithSuggestions('Location not found.', $lines, 'Try /list module=inventory.');
    }

    private function workOrderNotFoundSuggestion(int $tenantId, ?string $reference): array
    {
        $lines = $this->suggestWorkOrders($tenantId, $reference);
        return $this->notFoundWithSuggestions('Work order not found.', $lines, 'Try /list module=workorders.');
    }

    private function inventoryNotFoundSuggestion(int $tenantId, ?string $sku): array
    {
        $lines = $this->suggestInventorySkus($tenantId, $sku);
        return $this->notFoundWithSuggestions('Inventory item not found.', $lines, 'Try /list module=inventory.');
    }

    private function purchaseRequestNotFoundSuggestion(int $tenantId, ?string $code): array
    {
        $lines = $this->suggestPurchaseRequests($tenantId, $code);
        return $this->notFoundWithSuggestions('Purchase request not found.', $lines, 'Try /list module=purchase_requests.');
    }

    private function vendorNotFoundSuggestion(int $tenantId, ?string $name): array
    {
        $lines = $this->suggestVendors($tenantId, $name);
        return $this->notFoundWithSuggestions('Vendor not found.', $lines, 'Try /list module=vendors.');
    }

    private function partNotFoundSuggestion(int $tenantId, ?string $sku): array
    {
        $lines = $this->suggestParts($tenantId, $sku);
        return $this->notFoundWithSuggestions('Part not found.', $lines, 'Try /list module=parts.');
    }

    private function userNotFoundSuggestion(int $tenantId, ?string $email): array
    {
        $lines = $this->suggestUsers($tenantId, $email);
        return $this->notFoundWithSuggestions('User not found.', $lines, 'Try /list module=users.');
    }

    private function notFoundWithSuggestions(string $base, array $lines, string $fallback): array
    {
        if ($lines) {
            return ['message' => $base . " Did you mean:\n" . implode("\n", $lines)];
        }

        return ['message' => $base . ' ' . $fallback];
    }

    private function suggestAssets(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return Asset::where('tenant_id', $tenantId)
            ->where(function ($builder) use ($needle) {
                $builder->where('asset_tag', 'like', "%{$needle}%")
                    ->orWhere('name', 'like', "%{$needle}%");
            })
            ->limit(5)
            ->get()
            ->map(fn ($asset) => ($asset->asset_tag ?? '-') . ' - ' . ($asset->name ?? '-'))
            ->all();
    }

    private function suggestAssetLocations(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return Asset::where('tenant_id', $tenantId)
            ->whereNotNull('location')
            ->where('location', '!=', '')
            ->where('location', 'like', "%{$needle}%")
            ->distinct()
            ->limit(5)
            ->pluck('location')
            ->map(fn ($location) => (string) $location)
            ->all();
    }

    private function suggestSites(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return Site::where('tenant_id', $tenantId)
            ->where('name', 'like', "%{$needle}%")
            ->limit(5)
            ->pluck('name')
            ->map(fn ($name) => (string) $name)
            ->all();
    }

    private function suggestInventoryLocations(int $tenantId, ?string $needle): array
    {
        $needle = trim((string) $needle, "\"' ");
        if ($needle === '') {
            return [];
        }

        return InventoryLocation::where('tenant_id', $tenantId)
            ->where('name', 'like', "%{$needle}%")
            ->limit(5)
            ->pluck('name')
            ->map(fn ($name) => (string) $name)
            ->all();
    }

    private function suggestWorkOrders(int $tenantId, ?string $reference): array
    {
        $reference = trim((string) $reference, "\"' ");
        if ($reference === '') {
            return [];
        }

        return WorkOrder::where('tenant_id', $tenantId)
            ->where(function ($builder) use ($reference) {
                $builder->where('reference_code', 'like', "%{$reference}%")
                    ->orWhere('description', 'like', "%{$reference}%");
            })
            ->limit(5)
            ->get()
            ->map(function ($order) {
                $ref = $order->reference_code ?? ('WO-' . $order->id);
                $summary = Str::limit((string) ($order->description ?? ''), 32, '...');
                $tail = $summary ? " - {$summary}" : '';
                return $ref . $tail;
            })
            ->all();
    }

    private function suggestInventorySkus(int $tenantId, ?string $sku): array
    {
        $sku = trim((string) $sku, "\"' ");
        if ($sku === '') {
            return [];
        }

        return Part::where('tenant_id', $tenantId)
            ->where('sku', 'like', "%{$sku}%")
            ->limit(5)
            ->get()
            ->map(fn ($part) => ($part->sku ?? '-') . ' - ' . ($part->name ?? '-'))
            ->all();
    }

    private function suggestPurchaseRequests(int $tenantId, ?string $code): array
    {
        $code = trim((string) $code, "\"' ");
        if ($code === '') {
            return [];
        }

        return PurchaseRequest::where('tenant_id', $tenantId)
            ->where('request_code', 'like', "%{$code}%")
            ->limit(5)
            ->get()
            ->map(fn ($request) => ($request->request_code ?? '-') . ' - ' . ($request->title ?? '-'))
            ->all();
    }

    private function suggestVendors(int $tenantId, ?string $name): array
    {
        $name = trim((string) $name, "\"' ");
        if ($name === '') {
            return [];
        }

        return Vendor::where('tenant_id', $tenantId)
            ->where('name', 'like', "%{$name}%")
            ->limit(5)
            ->get()
            ->map(fn ($vendor) => ($vendor->name ?? '-') . ' - ' . ($vendor->contact_name ?? '-'))
            ->all();
    }

    private function suggestParts(int $tenantId, ?string $sku): array
    {
        $sku = trim((string) $sku, "\"' ");
        if ($sku === '') {
            return [];
        }

        return Part::where('tenant_id', $tenantId)
            ->where('sku', 'like', "%{$sku}%")
            ->limit(5)
            ->get()
            ->map(fn ($part) => ($part->sku ?? '-') . ' - ' . ($part->name ?? '-'))
            ->all();
    }

    private function suggestUsers(int $tenantId, ?string $email): array
    {
        $email = trim((string) $email, "\"' ");
        if ($email === '') {
            return [];
        }

        return User::where('tenant_id', $tenantId)
            ->where('email', 'like', "%{$email}%")
            ->limit(5)
            ->get()
            ->map(fn ($user) => ($user->email ?? '-') . ' - ' . ($user->name ?? '-'))
            ->all();
    }
}
