<?php

namespace App\Services\Notify;

use App\Models\AiReport;
use App\Models\Alert;
use App\Models\House;
use App\Services\Ai\PoultryAiService;
use Carbon\Carbon;
use Illuminate\Support\Str;

class PoultryTelegramCommandService
{
    public function __construct(private PoultryAiService $aiService)
    {
    }

    public function handle(?string $text, string $chatId): ?string
    {
        if (!$text) {
            return null;
        }

        $text = trim($text);
        if (!Str::startsWith(Str::lower($text), ['/status', '/alerts', '/report'])) {
            return null;
        }

        $house = House::where('telegram_chat_id', $chatId)->first();
        if (!$house) {
            return null;
        }

        if (Str::startsWith(Str::lower($text), '/status')) {
            return $this->statusReply($house);
        }

        if (Str::startsWith(Str::lower($text), '/alerts')) {
            return $this->alertsReply($house);
        }

        if (Str::startsWith(Str::lower($text), '/report')) {
            return $this->reportReply($house);
        }

        return null;
    }

    private function statusReply(House $house): string
    {
        $now = Carbon::now('UTC');
        $metrics = $this->aiService->collectMetrics($house, $now->copy()->subDay(), $now);

        return implode("\n", [
            'PoultryOps Status - ' . $house->name,
            'Feed (kg): ' . ($metrics['feed_kg'] ?? 'n/a'),
            'Water (L): ' . ($metrics['water_l'] ?? 'n/a'),
            'Weight gain (g/day): ' . ($metrics['weight']['gain_per_day_g'] ?? 'n/a'),
            'FCR est: ' . ($metrics['fcr_estimate'] ?? 'n/a'),
            'Mortality/1000: ' . ($metrics['mortality_per_1000'] ?? 'n/a'),
            'Behavior abnormal avg: ' . ($metrics['behavior_abnormal_avg'] ?? 'n/a'),
        ]);
    }

    private function alertsReply(House $house): string
    {
        $alerts = Alert::where('house_id', $house->id)
            ->where('status', 'open')
            ->orderByDesc('created_at')
            ->limit(5)
            ->get();

        if ($alerts->isEmpty()) {
            return 'PoultryOps Alerts - ' . $house->name . "\nNo open alerts.";
        }

        $lines = ['PoultryOps Alerts - ' . $house->name];
        foreach ($alerts as $alert) {
            $lines[] = strtoupper($alert->severity) . ': ' . $alert->title;
        }

        return implode("\n", $lines);
    }

    private function reportReply(House $house): string
    {
        $report = AiReport::where('house_id', $house->id)
            ->where('report_type', 'poultry_ops')
            ->orderByDesc('created_at')
            ->first();

        if (!$report) {
            return 'PoultryOps Report - ' . $house->name . "\nNo reports yet.";
        }

        return implode("\n", [
            'PoultryOps Report - ' . $house->name,
            $report->report_text ?? $report->content ?? '',
        ]);
    }
}
