<?php

namespace App\Services;

use App\Models\AgentNotification;
use App\Models\AssetDocument;
use App\Models\ComplianceRecord;
use App\Models\NotificationOutbox;
use App\Models\Tenant;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class OpsNotifierAgent
{
    public function __construct()
    {
    }

    public function runComplianceScanAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->runComplianceScan((int) $tenant->id);
            }
        });
    }

    public function runComplianceScan(int $tenantId): void
    {
        $records = ComplianceRecord::where('tenant_id', $tenantId)->get();
        if ($records->isEmpty()) {
            $records = $this->fallbackAssetDocuments($tenantId);
        }

        if ($records->isEmpty()) {
            return;
        }

        $findings = $this->buildComplianceFindings($records, $tenantId);
        if (!$findings) {
            return;
        }

        $alarmItems = array_values(array_filter($findings, fn ($item) => $item['severity'] === 'alarm'));
        $warnItems = array_values(array_filter($findings, fn ($item) => $item['severity'] === 'warn'));
        $infoItems = array_values(array_filter($findings, fn ($item) => $item['severity'] === 'info'));

        if ($alarmItems) {
            $this->queueComplianceMessage($tenantId, $alarmItems, 'alarm');
        }

        if (config('ops_notifier.send_warn_to_telegram', true) && $warnItems) {
            $this->queueComplianceMessage($tenantId, $warnItems, 'warn');
        }

        if (config('ops_notifier.send_info_to_telegram', false) && $infoItems) {
            $this->queueComplianceMessage($tenantId, $infoItems, 'info');
        }
    }

    public function queueFindingAlert(int $tenantId, string $summary, array $findings = []): void
    {
        $dedupeKey = $this->dedupeKey($tenantId, 'ops_finding', md5($summary));
        if ($this->isDuplicate($dedupeKey, (int) config('ops_notifier.dedupe_issue_hours', 6))) {
            return;
        }

        $notification = AgentNotification::create([
            'tenant_id' => $tenantId,
            'agent_key' => 'ops_notifier',
            'agent_name' => 'Ops Notifier Agent',
            'title' => 'Ops Notifier',
            'summary' => $summary,
            'severity' => 'alarm',
            'payload' => [
                'findings' => $findings,
                'prompt' => config('ai_agents.prompts.ops_notifier'),
            ],
            'recipients' => ['management'],
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        $this->queueOutbox($tenantId, $summary, 'alarm', ['management'], $dedupeKey);

        if ($notification) {
            $this->touchDedupe($dedupeKey, (int) config('ops_notifier.dedupe_issue_hours', 6));
        }
    }

    private function queueComplianceMessage(int $tenantId, array $items, string $severity): void
    {
        $maxItems = 7;
        $items = array_slice($items, 0, $maxItems);
        $lines = array_map(fn ($item) => '- ' . $item['line'], $items);

        $message = "Compliance " . strtoupper($severity) . "\n" . implode("\n", $lines);
        $dedupeKey = $this->dedupeKey($tenantId, 'compliance', md5($message));

        if ($this->isDuplicate($dedupeKey, (int) config('ops_notifier.dedupe_issue_hours', 6))) {
            return;
        }

        AgentNotification::create([
            'tenant_id' => $tenantId,
            'agent_key' => 'ops_notifier',
            'agent_name' => 'Ops Notifier Agent',
            'title' => 'Compliance ' . strtoupper($severity),
            'summary' => $message,
            'severity' => $severity,
            'payload' => [
                'items' => $items,
                'prompt' => config('ai_agents.prompts.ops_notifier'),
            ],
            'recipients' => $this->recipientsForSeverity($severity),
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        $this->queueOutbox(
            $tenantId,
            $message,
            $severity,
            $this->recipientsForSeverity($severity),
            $dedupeKey
        );

        $this->touchDedupe($dedupeKey, (int) config('ops_notifier.dedupe_issue_hours', 6));
    }

    private function buildComplianceFindings(Collection $records, int $tenantId): array
    {
        $findings = [];
        $infoDays = (int) config('ops_notifier.info_days', 60);
        $warnDays = (int) config('ops_notifier.warn_days', 30);
        $alarmDays = (int) config('ops_notifier.alarm_days', 7);
        $criticalTypes = array_map('strtolower', config('ops_notifier.critical_types', []));

        foreach ($records as $record) {
            $expiry = $record->expiry_date;
            if (!$expiry) {
                continue;
            }

            $daysTo = now()->diffInDays($expiry, false);
            $type = $this->complianceType($record);
            $isCritical = $this->isCriticalAsset($record);
            $isCriticalType = in_array(strtolower($type), $criticalTypes, true);
            $severity = null;

            if ($expiry->isPast()) {
                $severity = 'alarm';
            } elseif ($daysTo <= $alarmDays && ($isCritical || $isCriticalType)) {
                $severity = 'alarm';
            } elseif ($daysTo <= $warnDays) {
                $severity = 'warn';
            } elseif ($infoDays > 0 && $daysTo <= $infoDays) {
                $severity = 'info';
            }

            if (!$severity) {
                continue;
            }

            $statusSuffix = '';
            $dedupeHours = (int) config('ops_notifier.dedupe_hours', 24);
            if ($record->last_notified_at && $record->last_notified_at->gt(now()->subHours($dedupeHours))) {
                $statusSuffix = ' (still pending)';
            } elseif ($record->exists) {
                $record->update(['last_notified_at' => now()]);
            }

            $assetLabel = $this->assetLabel($record);
            $daysText = $daysTo < 0
                ? abs($daysTo) . ' days overdue'
                : $daysTo . ' days remaining';

            $impact = $this->impactText($type);
            $action = $this->actionText($type);
            $owner = data_get($record->meta_json, 'owner')
                ?: config('ops_notifier.default_owner', 'Fleet admin / Accounts');

            $line = sprintf(
                '%s - %s expires %s (%s)%s | Risk: %s | Action: %s | Owner: %s',
                $assetLabel,
                $type,
                $expiry->format('Y-m-d'),
                $daysText,
                $statusSuffix,
                $impact,
                $action,
                $owner
            );

            $findings[] = [
                'severity' => $severity,
                'line' => $line,
                'asset_id' => $record->asset_id,
                'type' => $type,
            ];
        }

        return $findings;
    }

    private function recipientsForSeverity(string $severity): array
    {
        return match ($severity) {
            'alarm' => ['maintenance', 'management'],
            'warn' => ['maintenance'],
            default => ['maintenance'],
        };
    }

    private function queueOutbox(
        int $tenantId,
        string $message,
        string $severity,
        array $recipients,
        ?string $dedupeKey
    ): void {
        NotificationOutbox::create([
            'tenant_id' => $tenantId,
            'channel' => 'telegram',
            'severity' => $severity,
            'title' => 'Ops Notifier',
            'message' => $message,
            'recipients' => $recipients,
            'status' => 'queued',
            'dedupe_key' => $dedupeKey,
            'available_at' => now(),
        ]);
    }

    private function dedupeKey(int $tenantId, string $type, string $hash): string
    {
        return $tenantId . ':' . $type . ':' . $hash;
    }

    private function isDuplicate(string $key, int $hours): bool
    {
        return Cache::has('ops_notifier:' . $key);
    }

    private function touchDedupe(string $key, int $hours): void
    {
        Cache::put('ops_notifier:' . $key, true, now()->addHours($hours));
    }

    private function complianceType($record): string
    {
        return $record->compliance_type ?? 'Compliance';
    }

    private function assetLabel($record): string
    {
        $assetTag = $record->asset?->asset_tag ?? $record->vehicle?->license_plate ?? 'Asset';
        $name = $record->asset?->name ?? $record->vehicle?->name;
        return $name ? "{$assetTag} - {$name}" : $assetTag;
    }

    private function impactText(string $type): string
    {
        $type = strtolower($type);
        if (str_contains($type, 'zinara') || str_contains($type, 'license') || str_contains($type, 'cof')) {
            return 'legal / operational';
        }

        if (str_contains($type, 'insurance')) {
            return 'insurance exposure';
        }

        return 'operational risk';
    }

    private function actionText(string $type): string
    {
        $type = strtolower($type);
        if (str_contains($type, 'zinara')) {
            return 'renew ZINARA today and upload receipt';
        }
        if (str_contains($type, 'insurance')) {
            return 'renew insurance and upload certificate';
        }
        if (str_contains($type, 'cof') || str_contains($type, 'road')) {
            return 'book inspection and upload certificate';
        }

        return 'update compliance record with renewal proof';
    }

    private function fallbackAssetDocuments(int $tenantId): Collection
    {
        return AssetDocument::where('tenant_id', $tenantId)
            ->whereNotNull('expiry_date')
            ->get()
            ->map(function ($doc) {
                $record = new ComplianceRecord();
                $record->asset_id = $doc->asset_id;
                $record->compliance_type = $doc->type ?? 'Compliance';
                $record->expiry_date = $doc->expiry_date;
                $record->status = $doc->expiry_date && $doc->expiry_date->isPast() ? 'expired' : 'active';
                $record->setRelation('asset', $doc->asset);
                return $record;
            });
    }

    private function highestSeverity(array $findings): string
    {
        $rank = ['info' => 1, 'warn' => 2, 'alarm' => 3];
        $highest = 'info';
        foreach ($findings as $finding) {
            $severity = $finding['severity'] ?? 'info';
            if (($rank[$severity] ?? 1) > ($rank[$highest] ?? 1)) {
                $highest = $severity;
            }
        }

        return $highest;
    }

    private function isCriticalAsset(ComplianceRecord $record): bool
    {
        $category = strtolower((string) ($record->asset?->category ?? ''));
        $tag = strtolower((string) ($record->asset?->asset_tag ?? $record->vehicle?->license_plate ?? ''));

        $criticalCategories = array_map('strtolower', config('ops_notifier.critical_categories', []));
        foreach ($criticalCategories as $critical) {
            if ($critical !== '' && str_contains($category, $critical)) {
                return true;
            }
        }

        $criticalTags = array_map('strtolower', config('ops_notifier.critical_tags', []));
        foreach ($criticalTags as $critical) {
            if ($critical !== '' && str_contains($tag, $critical)) {
                return true;
            }
        }

        return false;
    }
}
