<?php

namespace App\Services;

use App\Models\PurchaseRequestItem;
use App\Models\PurchaseRequest;
use App\Models\Site;
use Illuminate\Support\Carbon;

class PurchaseRequestEstimatorService
{
    public function backfillForSite(int $tenantId, string $siteName, float $fxRate = 13500, float $contingency = 0.12): array
    {
        $siteId = Site::where('tenant_id', $tenantId)->where('name', $siteName)->value('id');
        if (!$siteId) {
            return ['updated' => 0, 'errors' => 1, 'message' => 'Site not found'];
        }

        $requests = PurchaseRequest::where('tenant_id', $tenantId)->where('site_id', $siteId)->pluck('id');

        $items = PurchaseRequestItem::where('tenant_id', $tenantId)
            ->whereIn('purchase_request_id', $requests)
            ->get();

        $updated = 0;

        foreach ($items as $item) {
            $dirty = false;
            $quantity = $item->quantity;
            if ($quantity === null || $quantity <= 0) {
                $quantity = 1;
                $item->quantity = $quantity;
                $dirty = true;
            }

            if (!$item->selected_supplier && $item->supplier_name) {
                $item->selected_supplier = $item->supplier_name;
                $dirty = true;
            }

            if (!$item->purpose_cost_center) {
                $item->purpose_cost_center = $siteName . ' Operations';
                $dirty = true;
            }

            if (!$item->date_initiated) {
                $item->date_initiated = Carbon::now()->toDateString();
                $dirty = true;
            }

            $usd = $item->quote_amount_usd;
            if ($usd === null) {
                $usd = $this->estimateUsd($item->item_name, $quantity);
                $item->quote_amount_usd = $usd;
                $dirty = true;
            }

            if ($item->quote_amount_zwl === null && $usd !== null) {
                $zwl = $usd * $fxRate * (1 + $contingency);
                $item->quote_amount_zwl = round($zwl, 2);
                $dirty = true;

                $note = sprintf('Estimated ZWL incl %.0f%% contingency @ %s ZWL/USD.', $contingency * 100, number_format($fxRate, 0));
                $item->comments = trim(($item->comments ? $item->comments . ' ' : '') . $note);
            }

            if ($dirty) {
                $item->save();
                $updated++;
            }
        }

        $requests = PurchaseRequest::where('tenant_id', $tenantId)->where('site_id', $siteId)->get();
        foreach ($requests as $request) {
            $totalUsd = $request->items()->sum('quote_amount_usd');
            if ($totalUsd > 0) {
                $request->update(['total_estimated_cost' => $totalUsd]);
            }
        }

        return ['updated' => $updated, 'errors' => 0];
    }

    private function estimateUsd(?string $itemName, float $quantity): ?float
    {
        if (!$itemName) {
            return null;
        }

        $name = strtolower($itemName);
        $base = 150;

        if (str_contains($name, 'tyre') || str_contains($name, 'tire')) {
            $base = 800;
        } elseif (str_contains($name, 'filter')) {
            $base = 120;
        } elseif (str_contains($name, 'oil')) {
            $base = 80;
        } elseif (str_contains($name, 'belt')) {
            $base = 900;
        } elseif (str_contains($name, 'bearing')) {
            $base = 180;
        } elseif (str_contains($name, 'hydraulic')) {
            $base = 650;
        } elseif (str_contains($name, 'hose')) {
            $base = 95;
        } elseif (str_contains($name, 'pump')) {
            $base = 1200;
        } elseif (str_contains($name, 'grease')) {
            $base = 60;
        } elseif (str_contains($name, 'plate') || str_contains($name, 'steel')) {
            $base = 300;
        }

        return round($base * max(1, $quantity), 2);
    }
}
