<?php

namespace App\Services;

use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestItem;
use App\Models\Site;
use App\Support\XlsxReader;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use SimpleXMLElement;
use ZipArchive;

class PurchaseRequestImportService
{
    public function import(string $path, int $tenantId, string $siteName, ?string $sheetName = null): array
    {
        if (!is_file($path)) {
            return ['errors' => 1, 'message' => 'Purchase request file not found.'];
        }

        $siteId = $this->resolveSite($tenantId, $siteName);
        $sheetNames = $sheetName ? [$sheetName] : $this->sheetNames($path);

        $summary = [
            'requests_created' => 0,
            'items_created' => 0,
            'items_skipped' => 0,
            'errors' => 0,
        ];

        foreach ($sheetNames as $sheet) {
            $rows = XlsxReader::readSheetRows($path, $sheet);
            if (!$rows) {
                continue;
            }

            $request = PurchaseRequest::create([
                'tenant_id' => $tenantId,
                'site_id' => $siteId,
                'request_code' => $this->nextRequestCode($tenantId, 'PR-GOR'),
                'title' => trim($siteName . ' Procurement') . ($sheet ? ' - ' . trim($sheet) : ''),
                'status' => 'pending',
                'priority' => 'medium',
                'currency' => 'USD',
                'notes' => 'Imported from ' . basename($path) . ($sheet ? ' (' . trim($sheet) . ')' : ''),
                'source_file' => basename($path),
            ]);

            $summary['requests_created']++;

            $lineNumber = 1;
            $totalUsd = 0.0;

            foreach ($rows as $row) {
                $normalized = $this->normalizeRow($row);
                $itemName = $this->stringValue($normalized['item_ordered'] ?? null);
                $quantity = $this->numericValue($normalized['quantity'] ?? null);
                $purpose = $this->stringValue($normalized['purpose_cost_centre'] ?? null);
                $dateInitiated = $this->dateValue($normalized['date_initiated'] ?? null);
                $supplier = $this->stringValue($normalized['suppliers'] ?? null);
                $quoteNumber = $this->stringValue($normalized['quote_number'] ?? null);
                $quoteUsd = $this->numericValue($normalized['quote_amount_usd'] ?? null);
                $quoteZwl = $this->numericValue($normalized['quote_amount_zwl'] ?? null);
                $selectedSupplier = $this->stringValue($normalized['selected_supplier'] ?? null);
                $comments = $this->stringValue($normalized['comments'] ?? null);

                if (!$itemName && $quantity === null && !$purpose && !$quoteNumber && $quoteUsd === null && $quoteZwl === null) {
                    $summary['items_skipped']++;
                    continue;
                }

                PurchaseRequestItem::create([
                    'tenant_id' => $tenantId,
                    'purchase_request_id' => $request->id,
                    'line_number' => $lineNumber++,
                    'item_name' => $itemName,
                    'quantity' => $quantity,
                    'unit' => null,
                    'purpose_cost_center' => $purpose,
                    'date_initiated' => $dateInitiated,
                    'supplier_name' => $supplier,
                    'quote_number' => $quoteNumber,
                    'quote_amount_usd' => $quoteUsd,
                    'quote_amount_zwl' => $quoteZwl,
                    'selected_supplier' => $selectedSupplier,
                    'comments' => $comments,
                ]);

                if ($quoteUsd !== null) {
                    $totalUsd += $quoteUsd;
                }

                $summary['items_created']++;
            }

            $request->update([
                'total_estimated_cost' => $totalUsd > 0 ? $totalUsd : null,
            ]);
        }

        return $summary;
    }

    private function sheetNames(string $path): array
    {
        $zip = new ZipArchive();
        if ($zip->open($path) !== true) {
            return [null];
        }

        $workbookXml = $zip->getFromName('xl/workbook.xml');
        $zip->close();

        if (!$workbookXml) {
            return [null];
        }

        $workbook = new SimpleXMLElement($workbookXml);
        $names = [];
        foreach ($workbook->sheets->sheet as $sheet) {
            $name = (string) $sheet['name'];
            if ($name !== '') {
                $names[] = $name;
            }
        }

        return $names ?: [null];
    }

    private function normalizeRow(array $row): array
    {
        $normalized = [];
        foreach ($row as $header => $value) {
            $normalized[$this->normalizeHeader((string) $header)] = $value;
        }

        return $normalized;
    }

    private function normalizeHeader(string $header): string
    {
        $header = strtolower(trim($header));
        $header = preg_replace('/[^a-z0-9]+/', '_', $header);
        return trim($header, '_');
    }

    private function resolveSite(int $tenantId, string $siteName): int
    {
        $site = Site::firstOrCreate(
            ['tenant_id' => $tenantId, 'name' => $siteName],
            [
                'code' => strtoupper(Str::substr(Str::slug($siteName, ''), 0, 6)) ?: 'SITE',
                'type' => 'site',
                'address' => null,
                'active' => true,
            ]
        );

        return $site->id;
    }

    private function nextRequestCode(int $tenantId, string $prefix): string
    {
        $latest = PurchaseRequest::where('tenant_id', $tenantId)
            ->where('request_code', 'like', $prefix . '-%')
            ->orderByDesc('id')
            ->first();

        $sequence = 1;
        if ($latest) {
            $parts = explode('-', $latest->request_code);
            $last = (int) end($parts);
            $sequence = $last + 1;
        }

        return sprintf('%s-%03d', $prefix, $sequence);
    }

    private function stringValue(mixed $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        return $value === '' ? null : $value;
    }

    private function numericValue(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return (float) $value;
        }

        $cleaned = preg_replace('/[^0-9.]/', '', (string) $value);
        if ($cleaned === '') {
            return null;
        }

        return (float) $cleaned;
    }

    private function dateValue(mixed $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }

        if (is_numeric($value)) {
            return Carbon::create(1899, 12, 30)->addDays((int) $value)->toDateString();
        }

        try {
            return Carbon::parse($value)->toDateString();
        } catch (\Throwable $e) {
            return null;
        }
    }
}
