<?php

namespace App\Services;

use App\Models\AssetDocument;
use App\Models\DowntimeLog;
use App\Models\InventoryItem;
use App\Models\Maintenance;
use App\Domain\Telegram\TelegramNotifier;
use App\Models\TelegramGroup;
use App\Models\Tenant;
use App\Models\WorkOrder;

class TelegramDigestService
{
    public function __construct(private TelegramNotifier $notifier)
    {
    }

    public function sendDailyMaintenanceDigest(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $message = $this->buildMaintenanceDigest((int) $tenant->id);
                if (!$message) {
                    continue;
                }

                $groups = array_merge(
                    config('telegram_events.groups.maintenance', []),
                    config('telegram_events.groups.management', [])
                );

                $this->sendToGroups((int) $tenant->id, $groups, $message);
            }
        });
    }

    private function buildMaintenanceDigest(int $tenantId): ?string
    {
        $yesterdayStart = now()->subDay()->startOfDay();
        $yesterdayEnd = now()->subDay()->endOfDay();

        $opened = WorkOrder::where('tenant_id', $tenantId)
            ->whereBetween('created_at', [$yesterdayStart, $yesterdayEnd])
            ->count();

        $closed = WorkOrder::where('tenant_id', $tenantId)
            ->whereBetween('completed_at', [$yesterdayStart, $yesterdayEnd])
            ->count();

        $downtimeMinutes = DowntimeLog::where('tenant_id', $tenantId)
            ->whereBetween('ended_at', [$yesterdayStart, $yesterdayEnd])
            ->sum('duration_minutes');

        $pmDue = Maintenance::where('tenant_id', $tenantId)
            ->whereNull('completed_date')
            ->where('status', 'pending')
            ->whereDate('scheduled_date', now()->toDateString())
            ->with('asset')
            ->limit(5)
            ->get()
            ->map(fn ($item) => $this->assetLabel($item->asset) . ' (' . ($item->scheduled_date?->format('Y-m-d') ?? '-') . ')')
            ->all();

        $lowStock = InventoryItem::where('tenant_id', $tenantId)
            ->with('part')
            ->get()
            ->filter(function ($item) {
                $reorder = $item->reorder_point ?? $item->min_quantity;
                if ($reorder === null) {
                    return false;
                }
                return (float) $item->quantity <= (float) $reorder;
            })
            ->take(5)
            ->map(fn ($item) => ($item->part?->sku ?? '-') . ' (' . ($item->quantity ?? 0) . ')')
            ->all();

        $compliance = AssetDocument::where('tenant_id', $tenantId)
            ->whereNotNull('expiry_date')
            ->where('expiry_date', '<=', now()->addDays((int) config('telegram_events.thresholds.compliance_due_days', 30)))
            ->with('asset')
            ->limit(5)
            ->get()
            ->map(fn ($doc) => $this->assetLabel($doc->asset) . ' ' . ($doc->type ?? 'doc') . ' (' . ($doc->expiry_date?->format('Y-m-d') ?? '-') . ')')
            ->all();

        $downtimeHours = round(((float) $downtimeMinutes) / 60, 1);

        $lines = [
            'DAILY MAINT DIGEST - ' . now()->subDay()->toDateString(),
            "Work orders opened: {$opened} | closed: {$closed}",
            'Downtime: ' . $downtimeHours . 'h',
        ];

        if ($pmDue) {
            $lines[] = 'PM due today: ' . implode('; ', $pmDue);
        }
        if ($lowStock) {
            $lines[] = 'Low stock: ' . implode('; ', $lowStock);
        }
        if ($compliance) {
            $lines[] = 'Compliance due: ' . implode('; ', $compliance);
        }

        return implode("\n", $lines);
    }

    private function sendToGroups(int $tenantId, array $groupNames, string $message): void
    {
        if (!$groupNames) {
            return;
        }

        $groups = TelegramGroup::where('tenant_id', $tenantId)
            ->whereIn('name', $groupNames)
            ->where('enabled', true)
            ->get();

        foreach ($groups as $group) {
            $this->notifier->notifyChat($group->chat_id, $message);
        }
    }

    private function assetLabel(?\App\Models\Asset $asset, ?string $fallbackTag = null): string
    {
        $tag = $asset?->asset_tag ?? $fallbackTag ?? '-';
        $name = $asset?->name ?? null;
        if ($name) {
            return "{$tag} - {$name}";
        }

        return (string) $tag;
    }
}
