<?php

namespace App\Services;

use App\Domain\Telegram\TelegramNotifier;
use App\Models\NotificationRoute;
use App\Models\TelegramEvent;
use App\Models\TelegramEventMessage;
use App\Models\TelegramGroup;
use App\Models\User;

class TelegramEventDispatcher
{
    public function __construct(private TelegramNotifier $notifier)
    {
    }

    public function dispatch(TelegramEvent $event, array $groups = []): void
    {
        $recipients = $this->resolveRecipients($event, $groups);

        foreach ($recipients['groups'] as $group) {
            $messageId = $this->notifier->notifyChat($group->chat_id, $event->message ?? '');
            TelegramEventMessage::create([
                'tenant_id' => $event->tenant_id,
                'telegram_event_id' => $event->id,
                'chat_id' => $group->chat_id,
                'message_id' => $messageId,
            ]);
        }

        foreach ($recipients['users'] as $user) {
            $this->notifier->notifyUser($user->id, $event->message ?? '');
        }
    }

    private function resolveRecipients(TelegramEvent $event, array $overrideGroups = []): array
    {
        $groups = collect();
        $users = collect();

        $route = NotificationRoute::where('tenant_id', $event->tenant_id)
            ->where('event_type', $event->event_type)
            ->first();

        $recipients = $route?->recipients ?? [];

        if ($overrideGroups) {
            $recipients = $overrideGroups;
        }

        if ($recipients) {
            foreach ($recipients as $recipient) {
                if (is_string($recipient)) {
                    $mapped = config('telegram_events.groups.' . $recipient, [$recipient]);
                    $groups = $groups->merge($this->groupsByName($event->tenant_id, $mapped, $event->site_id));
                    continue;
                }

                $type = $recipient['type'] ?? null;
                $value = $recipient['value'] ?? null;
                if (!$type || !$value) {
                    continue;
                }
                if ($type === 'group') {
                    $mapped = config('telegram_events.groups.' . $value, [$value]);
                    $groups = $groups->merge($this->groupsByName($event->tenant_id, $mapped, $event->site_id));
                } elseif ($type === 'role') {
                    $users = $users->merge($this->usersByRole($event->tenant_id, [$value]));
                } elseif ($type === 'user') {
                    $user = User::where('tenant_id', $event->tenant_id)->where('id', $value)->first();
                    if ($user) {
                        $users->push($user);
                    }
                }
            }
        }

        if ($groups->isEmpty()) {
            $groupNames = config('telegram_events.event_groups.' . $event->event_type, []);
            foreach ($groupNames as $name) {
                $mapped = config('telegram_events.groups.' . $name, [$name]);
                $groups = $groups->merge($this->groupsByName($event->tenant_id, $mapped, $event->site_id));
            }
        }

        return [
            'groups' => $groups->unique('id')->values(),
            'users' => $users->unique('id')->values(),
        ];
    }

    private function groupsByName(int $tenantId, array $names, ?int $siteId): array
    {
        $query = TelegramGroup::where('tenant_id', $tenantId)
            ->whereIn('name', $names)
            ->where('enabled', true);

        if ($siteId) {
            $query->where(function ($builder) use ($siteId) {
                $builder->whereNull('site_id')->orWhere('site_id', $siteId);
            });
        }

        return $query->get()->all();
    }

    private function usersByRole(int $tenantId, array $roles)
    {
        return User::where('tenant_id', $tenantId)
            ->whereHas('roles', function ($q) use ($roles) {
                $first = true;
                foreach ($roles as $role) {
                    if ($first) {
                        $q->where('name', 'like', '%' . $role . '%');
                        $first = false;
                        continue;
                    }
                    $q->orWhere('name', 'like', '%' . $role . '%');
                }
            })
            ->get()
            ->all();
    }
}
