<?php

namespace App\Services;

use App\Models\Alarm;
use App\Models\AssetDocument;
use App\Models\FuelLog;
use App\Models\InventoryItem;
use App\Models\Maintenance;
use App\Models\PurchaseOrder;
use App\Models\PurchaseRequest;
use App\Models\Tenant;
use App\Models\Telemetry;
use App\Models\WorkOrder;

class TelegramEventEngine
{
    public function __construct(private TelegramEventService $events)
    {
    }

    public function scanAll(): void
    {
        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $this->scanTenant((int) $tenant->id);
            }
        });
    }

    public function scanTenant(int $tenantId): void
    {
        $this->scanMaintenance($tenantId);
        $this->scanWorkOrders($tenantId);
        $this->scanInventory($tenantId);
        $this->scanProcurement($tenantId);
        $this->scanCompliance($tenantId);
        $this->scanAlarms($tenantId);
        $this->scanTelemetry($tenantId);
        $this->scanFuelVariance($tenantId);
    }

    private function scanMaintenance(int $tenantId): void
    {
        $dueSoonHours = (int) config('telegram_events.thresholds.pm_due_hours', 24);
        $dueSoonDate = now()->addHours($dueSoonHours);

        $items = Maintenance::where('tenant_id', $tenantId)
            ->whereNull('completed_date')
            ->where('status', 'pending')
            ->whereNotNull('scheduled_date')
            ->where('scheduled_date', '<=', $dueSoonDate)
            ->with('asset')
            ->limit(100)
            ->get();

        foreach ($items as $item) {
            $asset = $item->asset;
            if (!$asset) {
                continue;
            }

            $scheduled = $item->scheduled_date;
            $isOverdue = $scheduled && $scheduled->isPast();
            $hoursDiff = $scheduled ? now()->diffInHours($scheduled, false) : null;
            $dueText = $hoursDiff === null
                ? 'due soon'
                : ($hoursDiff < 0 ? abs($hoursDiff) . 'h overdue' : 'due in ~' . $hoursDiff . 'h');

            $eventType = $isOverdue ? 'pm_overdue' : 'pm_due';
            $severity = $isOverdue ? 'alarm' : 'warn';

            $autoWorkOrder = null;
            if ($isOverdue) {
                $autoWorkOrder = $this->autoCreatePmWorkOrder($tenantId, $item, $asset);
            }

            $this->events->createEvent($tenantId, $eventType, $severity, [
                'maintenance_id' => $item->id,
                'asset_id' => $asset->id,
                'asset_tag' => $asset->asset_tag,
                'asset_name' => $asset->name,
                'site_id' => $asset->site_id,
                'work_order_id' => $autoWorkOrder?->id,
                'details' => [
                    'Scheduled: ' . ($scheduled?->format('Y-m-d') ?? '-'),
                    'Status: ' . ($item->status ?? 'pending'),
                    'Window: ' . $dueText,
                    $autoWorkOrder
                        ? 'WO: ' . ($autoWorkOrder->reference_code ?? 'WO-' . $autoWorkOrder->id) . ' (pending approval)'
                        : 'Reply: CREATE WO <event_code> or SNOOZE <event_code> 24H',
                ],
            ], [
                'title' => $isOverdue ? 'PM OVERDUE' : 'PM DUE SOON',
            ]);
        }
    }

    private function scanWorkOrders(int $tenantId): void
    {
        $overdue = WorkOrder::where('tenant_id', $tenantId)
            ->whereIn('status', ['open', 'scheduled', 'in_progress', 'pending_approval'])
            ->whereNotNull('due_at')
            ->where('due_at', '<', now())
            ->with('asset')
            ->limit(100)
            ->get();

        foreach ($overdue as $order) {
            $assetTag = $order->asset?->asset_tag ?? '-';
            $assetName = $order->asset?->name ?? null;
            $this->events->createEvent($tenantId, 'wo_overdue', 'warn', [
                'work_order_id' => $order->id,
                'asset_id' => $order->asset_id,
                'asset_tag' => $assetTag,
                'asset_name' => $assetName,
                'site_id' => $order->asset?->site_id,
                'details' => [
                    'WO: ' . ($order->reference_code ?? 'WO-' . $order->id),
                    'Due: ' . ($order->due_at?->format('Y-m-d H:i') ?? '-'),
                    'Priority: ' . ($order->priority ?? '-'),
                    'Reply: ASSIGN ' . ($order->reference_code ?? '') . ' @Name',
                ],
            ], [
                'title' => 'WORK ORDER OVERDUE',
            ]);
        }

        $breakdowns = WorkOrder::where('tenant_id', $tenantId)
            ->whereIn('status', ['open', 'scheduled', 'in_progress'])
            ->where(function ($builder) {
                $builder->where('type', 'like', '%breakdown%')
                    ->orWhere('priority', 'high');
            })
            ->with('asset')
            ->limit(100)
            ->get();

        foreach ($breakdowns as $order) {
            $assetTag = $order->asset?->asset_tag ?? '-';
            $assetName = $order->asset?->name ?? null;
            $this->events->createEvent($tenantId, 'breakdown', 'alarm', [
                'work_order_id' => $order->id,
                'asset_id' => $order->asset_id,
                'asset_tag' => $assetTag,
                'asset_name' => $assetName,
                'site_id' => $order->asset?->site_id,
                'details' => [
                    'WO: ' . ($order->reference_code ?? 'WO-' . $order->id),
                    'Priority: ' . ($order->priority ?? 'high'),
                    'Status: ' . ($order->status ?? '-'),
                    'Reply: ACK <event_code> or DISPATCH <event_code> @Tech',
                ],
            ], [
                'title' => 'BREAKDOWN',
            ]);
        }
    }

    private function scanInventory(int $tenantId): void
    {
        $lowMultiplier = (float) config('telegram_events.thresholds.stock_low_multiplier', 1.0);
        $criticalQty = (float) config('telegram_events.thresholds.stock_critical_qty', 0);

        $items = InventoryItem::where('tenant_id', $tenantId)
            ->with('part')
            ->limit(200)
            ->get();

        foreach ($items as $item) {
            $reorderPoint = $item->reorder_point ?? $item->min_quantity;
            if ($reorderPoint === null) {
                continue;
            }

            $lowThreshold = (float) $reorderPoint * $lowMultiplier;
            $quantity = (float) ($item->quantity ?? 0);
            $sku = $item->part?->sku ?? 'SKU';
            $partName = $item->part?->name ?? '-';

            if ($quantity <= $criticalQty) {
                $this->events->createEvent($tenantId, 'inventory_stockout', 'alarm', [
                    'inventory_item_id' => $item->id,
                    'details' => [
                        "{$sku} ({$partName})",
                        'On hand: ' . $quantity,
                        'Reorder point: ' . $reorderPoint,
                        'Reply: CREATE PR <event_code>',
                    ],
                ], [
                    'title' => 'STOCKOUT',
                ]);
                continue;
            }

            if ($quantity <= $lowThreshold) {
                $this->events->createEvent($tenantId, 'inventory_low', 'warn', [
                    'inventory_item_id' => $item->id,
                    'details' => [
                        "{$sku} ({$partName})",
                        'On hand: ' . $quantity,
                        'Reorder point: ' . $reorderPoint,
                        'Reply: CREATE PR <event_code> or VIEW USAGE <event_code>',
                    ],
                ], [
                    'title' => 'LOW STOCK',
                ]);
            }
        }
    }

    private function scanProcurement(int $tenantId): void
    {
        $requests = PurchaseRequest::where('tenant_id', $tenantId)
            ->where('status', 'submitted')
            ->limit(100)
            ->get();

        foreach ($requests as $request) {
            $this->events->createEvent($tenantId, 'procurement_pr_pending', 'warn', [
                'purchase_request_id' => $request->id,
                'details' => [
                    'PR: ' . ($request->request_code ?? '-'),
                    'Department: ' . ($request->department ?? '-'),
                    'Needed by: ' . ($request->needed_by?->format('Y-m-d') ?? '-'),
                    'Reply: APPROVE ' . ($request->request_code ?? '') . ' or REJECT ' . ($request->request_code ?? ''),
                ],
            ], [
                'title' => 'PR APPROVAL PENDING',
            ]);
        }

        $overdueDays = (int) config('telegram_events.thresholds.po_overdue_days', 3);
        $cutoff = now()->subDays($overdueDays);

        $orders = PurchaseOrder::where('tenant_id', $tenantId)
            ->whereNotIn('status', ['received', 'closed'])
            ->where(function ($builder) use ($cutoff) {
                $builder->whereNotNull('delivery_due_at')
                    ->where('delivery_due_at', '<', $cutoff);
            })
            ->limit(100)
            ->get();

        foreach ($orders as $order) {
            $this->events->createEvent($tenantId, 'procurement_po_overdue', 'warn', [
                'purchase_order_id' => $order->id,
                'details' => [
                    'PO: ' . ($order->po_number ?? '-'),
                    'Vendor: ' . ($order->vendor_name ?? '-'),
                    'Due: ' . ($order->delivery_due_at?->format('Y-m-d') ?? '-'),
                    'Reply: SEND ' . ($order->po_number ?? '') . ' or UPDATE PO',
                ],
            ], [
                'title' => 'PO DELIVERY OVERDUE',
            ]);
        }
    }

    private function scanCompliance(int $tenantId): void
    {
        $dueDays = (int) config('telegram_events.thresholds.compliance_due_days', 30);
        $cutoff = now()->addDays($dueDays);

        $docs = AssetDocument::where('tenant_id', $tenantId)
            ->whereNotNull('expiry_date')
            ->where('expiry_date', '<=', $cutoff)
            ->with('asset')
            ->limit(100)
            ->get();

        foreach ($docs as $doc) {
            $asset = $doc->asset;
            if (!$asset) {
                continue;
            }

            $days = $doc->expiry_date ? now()->diffInDays($doc->expiry_date, false) : null;
            $dueText = $days === null ? 'due soon' : ($days < 0 ? abs($days) . 'd overdue' : $days . 'd remaining');
            $severity = ($days !== null && $days < 0) ? 'alarm' : 'warn';
            $title = ($days !== null && $days < 0) ? 'COMPLIANCE OVERDUE' : 'COMPLIANCE DUE';

            $this->events->createEvent($tenantId, 'compliance_due', $severity, [
                'document_id' => $doc->id,
                'asset_id' => $asset->id,
                'asset_tag' => $asset->asset_tag,
                'asset_name' => $asset->name,
                'site_id' => $asset->site_id,
                'details' => [
                    'Document: ' . ($doc->type ?? 'document'),
                    'Expiry: ' . ($doc->expiry_date?->format('Y-m-d') ?? '-'),
                    'Window: ' . $dueText,
                    'Reply: CREATE TASK or UPLOAD RENEWAL',
                ],
            ], [
                'title' => $title,
            ]);
        }
    }

    private function scanAlarms(int $tenantId): void
    {
        $alarms = Alarm::where('tenant_id', $tenantId)
            ->where('status', 'open')
            ->with('asset')
            ->limit(100)
            ->get();

        foreach ($alarms as $alarm) {
            $assetTag = $alarm->asset?->asset_tag ?? '-';
            $assetName = $alarm->asset?->name ?? null;
            $message = $alarm->message ?? 'Alarm triggered';
            $eventType = $this->isGeneratorAsset($alarm->asset?->asset_tag, $alarm->asset?->category)
                ? 'generator_alarm'
                : 'breakdown';

            $this->events->createEvent($tenantId, $eventType, 'alarm', [
                'alarm_id' => $alarm->id,
                'asset_id' => $alarm->asset_id,
                'asset_tag' => $assetTag,
                'asset_name' => $assetName,
                'site_id' => $alarm->asset?->site_id,
                'details' => [
                    'Alarm: ' . $message,
                    'Reply: ACK <event_code> or CREATE WO <event_code>',
                ],
            ], [
                'title' => strtoupper(str_replace('_', ' ', $eventType)),
            ]);
        }
    }

    private function scanTelemetry(int $tenantId): void
    {
        $entries = Telemetry::where('tenant_id', $tenantId)
            ->whereNotNull('data')
            ->orderByDesc('timestamp')
            ->limit(50)
            ->with('asset')
            ->get();

        foreach ($entries as $entry) {
            $data = $entry->data ?? [];
            if (!is_array($data) || empty($data)) {
                continue;
            }

            $assetTag = $entry->asset?->asset_tag ?? '-';
            $assetName = $entry->asset?->name ?? null;
            $alarmText = $data['alarm'] ?? $data['fault_code'] ?? null;
            $fuelLevel = isset($data['fuel_level']) ? (float) $data['fuel_level'] : null;
            $lowFuelThreshold = (float) config('telegram_events.thresholds.generator_low_fuel_percent', 20);

            if ($alarmText) {
                $this->events->createEvent($tenantId, 'breakdown', 'alarm', [
                    'asset_id' => $entry->asset_id,
                    'asset_tag' => $assetTag,
                    'asset_name' => $assetName,
                    'site_id' => $entry->asset?->site_id,
                    'details' => [
                        'Fault: ' . $alarmText,
                        'Reply: ACK <event_code> or CREATE WO <event_code>',
                    ],
                ], [
                    'title' => 'SENSOR ALARM',
                ]);
            }

            if ($fuelLevel !== null && $this->isGeneratorAsset($entry->asset?->asset_tag, $entry->asset?->category)) {
                if ($fuelLevel <= $lowFuelThreshold) {
                    $this->events->createEvent($tenantId, 'generator_alarm', 'alarm', [
                        'asset_id' => $entry->asset_id,
                        'asset_tag' => $assetTag,
                        'asset_name' => $assetName,
                        'site_id' => $entry->asset?->site_id,
                        'details' => [
                            'Fuel level: ' . $fuelLevel . '%',
                            'Reply: ACK <event_code> or REQUEST FUEL',
                        ],
                    ], [
                        'title' => 'GENERATOR LOW FUEL',
                    ]);
                }
            }
        }
    }

    private function scanFuelVariance(int $tenantId): void
    {
        $threshold = (float) config('telegram_events.thresholds.fuel_variance_threshold', 10);
        if ($threshold <= 0) {
            return;
        }

        $logs = FuelLog::where('tenant_id', $tenantId)
            ->whereNotNull('variance')
            ->where('logged_at', '>=', now()->subDay())
            ->with('asset')
            ->limit(50)
            ->get();

        foreach ($logs as $log) {
            $variance = (float) ($log->variance ?? 0);
            if (abs($variance) < $threshold) {
                continue;
            }

            $assetTag = $log->asset?->asset_tag ?? '-';
            $assetName = $log->asset?->name ?? null;
            $this->events->createEvent($tenantId, 'fuel_variance', 'warn', [
                'asset_id' => $log->asset_id,
                'asset_tag' => $assetTag,
                'asset_name' => $assetName,
                'site_id' => $log->site_id,
                'details' => [
                    'Fuel variance: ' . $variance,
                    'Expected: ' . ($log->expected_quantity ?? '-'),
                    'Actual: ' . ($log->quantity ?? '-'),
                ],
            ], [
                'title' => 'FUEL VARIANCE',
            ]);
        }
    }

    private function isGeneratorAsset(?string $tag, ?string $category): bool
    {
        $tag = strtoupper((string) $tag);
        $category = strtoupper((string) $category);

        if ($tag !== '' && str_contains($tag, 'GEN')) {
            return true;
        }

        return $category !== '' && str_contains($category, 'GEN');
    }

    private function autoCreatePmWorkOrder(int $tenantId, Maintenance $item, $asset): ?WorkOrder
    {
        $name = $item->name ?? 'PM';
        $existing = WorkOrder::where('tenant_id', $tenantId)
            ->where('asset_id', $asset->id)
            ->where('auto_generated', true)
            ->whereIn('status', ['pending_approval', 'open', 'scheduled', 'in_progress'])
            ->where('description', 'like', '%' . $name . '%')
            ->first();

        if ($existing) {
            return $existing;
        }

        return WorkOrder::create([
            'tenant_id' => $tenantId,
            'asset_id' => $asset->id,
            'reference_code' => strtoupper('WO-' . \Illuminate\Support\Str::random(6)),
            'source' => 'system',
            'status' => 'pending_approval',
            'priority' => 'medium',
            'type' => 'preventive',
            'description' => 'PM: ' . $name,
            'reported_by' => null,
            'auto_generated' => true,
            'requires_approval' => true,
            'approval_status' => 'pending',
            'last_reminded_at' => now(),
        ]);
    }
}
