<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\TelegramEvent;
use App\Models\WorkOrder;
use Illuminate\Support\Str;

class TelegramEventService
{
    public function __construct(private TelegramEventDispatcher $dispatcher)
    {
    }

    public function createEvent(
        int $tenantId,
        string $eventType,
        string $severity,
        array $payload,
        array $options = []
    ): TelegramEvent {
        $eventCode = $this->nextEventCode($tenantId);
        $payload['event_code'] = $payload['event_code'] ?? $eventCode;
        $dedupeKey = $options['dedupe_key'] ?? $this->dedupeKey($eventType, $payload);

        if ($dedupeKey) {
            $existing = TelegramEvent::where('tenant_id', $tenantId)
                ->where('dedupe_key', $dedupeKey)
                ->whereIn('status', ['open', 'acknowledged', 'snoozed'])
                ->first();
            if ($existing) {
                return $existing;
            }
        }

        $event = TelegramEvent::create([
            'tenant_id' => $tenantId,
            'site_id' => $payload['site_id'] ?? null,
            'asset_id' => $payload['asset_id'] ?? null,
            'work_order_id' => $payload['work_order_id'] ?? null,
            'event_code' => $eventCode,
            'event_type' => $eventType,
            'severity' => $severity,
            'title' => $options['title'] ?? null,
            'message' => $options['message'] ?? $this->formatMessage($eventType, $severity, $payload),
            'payload' => $payload,
            'status' => 'open',
            'dedupe_key' => $dedupeKey,
        ]);

        $this->dispatcher->dispatch($event);

        return $event;
    }

    public function ackEvent(int $tenantId, string $eventCode, int $userId): ?TelegramEvent
    {
        $event = TelegramEvent::where('tenant_id', $tenantId)
            ->where('event_code', $eventCode)
            ->first();

        if (!$event) {
            return null;
        }

        $event->update([
            'status' => 'acknowledged',
            'acked_by' => $userId,
            'acked_at' => now(),
        ]);

        return $event;
    }

    public function snoozeEvent(int $tenantId, string $eventCode, int $userId, int $minutes, ?string $reason): ?TelegramEvent
    {
        $event = TelegramEvent::where('tenant_id', $tenantId)
            ->where('event_code', $eventCode)
            ->first();

        if (!$event) {
            return null;
        }

        $event->update([
            'status' => 'snoozed',
            'acked_by' => $userId,
            'acked_at' => now(),
            'snoozed_until' => now()->addMinutes($minutes),
            'payload' => array_merge($event->payload ?? [], ['snooze_reason' => $reason]),
        ]);

        return $event;
    }

    public function createWorkOrderFromEvent(int $tenantId, string $eventCode, int $userId): ?WorkOrder
    {
        $event = TelegramEvent::where('tenant_id', $tenantId)
            ->where('event_code', $eventCode)
            ->first();
        if (!$event) {
            return null;
        }

        $assetId = $event->asset_id ?? ($event->payload['asset_id'] ?? null);
        $asset = $assetId ? Asset::where('tenant_id', $tenantId)->where('id', $assetId)->first() : null;
        if (!$asset) {
            return null;
        }

        $order = WorkOrder::create([
            'tenant_id' => $tenantId,
            'asset_id' => $asset->id,
            'reference_code' => strtoupper('WO-' . Str::random(6)),
            'source' => 'telegram',
            'status' => 'pending_approval',
            'priority' => $event->payload['priority'] ?? 'medium',
            'description' => $event->message ?? $event->title ?? $event->event_type,
            'reported_by' => $userId,
            'requires_approval' => true,
            'approval_status' => 'pending',
            'last_reminded_at' => now(),
        ]);

        $event->update([
            'work_order_id' => $order->id,
            'status' => 'closed',
        ]);

        return $order;
    }

    private function dedupeKey(string $eventType, array $payload): ?string
    {
        $parts = [$eventType];
        foreach ([
            'pm_schedule_id',
            'maintenance_id',
            'alarm_id',
            'asset_id',
            'work_order_id',
            'inventory_item_id',
            'document_id',
            'purchase_request_id',
            'purchase_order_id',
        ] as $key) {
            if (!empty($payload[$key])) {
                $parts[] = $key . ':' . $payload[$key];
            }
        }

        return count($parts) > 1 ? implode('|', $parts) : null;
    }

    private function nextEventCode(int $tenantId): string
    {
        $last = TelegramEvent::where('tenant_id', $tenantId)
            ->where('event_code', 'like', 'EVT-%')
            ->orderByDesc('id')
            ->value('event_code');

        $number = 1;
        if ($last && preg_match('/EVT-(\\d+)/', $last, $matches)) {
            $number = (int) $matches[1] + 1;
        }

        return sprintf('EVT-%05d', $number);
    }

    private function formatMessage(string $eventType, string $severity, array $payload): string
    {
        $prefix = match (strtolower($severity)) {
            'alarm' => '[ALARM]',
            'warn' => '[WARN]',
            default => '[INFO]',
        };

        $assetTag = $payload['asset_tag'] ?? null;
        $assetName = $payload['asset_name'] ?? null;
        $title = $payload['title'] ?? strtoupper(str_replace('_', ' ', $eventType));
        $eventCode = $payload['event_code'] ?? '';

        $assetLabel = $assetTag ? ($assetName ? "{$assetTag} - {$assetName}" : $assetTag) : null;
        $line = $assetLabel ? "{$title}: {$assetLabel}" : $title;
        $details = $payload['details'] ?? null;
        $detailsText = '';
        if (is_array($details)) {
            $detailsText = implode("\n", array_filter($details));
        } elseif (is_string($details) && $details !== '') {
            $detailsText = $details;
        }

        if ($detailsText !== '' && $eventCode !== '') {
            $detailsText = str_replace('<event_code>', $eventCode, $detailsText);
        }

        $message = "{$prefix} {$line}";
        if ($detailsText !== '') {
            $message .= "\n" . $detailsText;
        }
        $message .= "\nEvent {$eventCode}";

        return trim($message);
    }
}
