<?php

namespace App\Services;

use App\Models\Tenant;
use App\Models\PurchaseRequest;
use App\Models\WorkOrder;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use RuntimeException;

class TelegramExportService
{
    public function exportTable(string $title, array $headers, array $rows, ?Tenant $tenant, string $format): array
    {
        $format = strtolower($format);
        if (!in_array($format, ['pdf', 'xlsx'], true)) {
            throw new RuntimeException('Unsupported export format.');
        }

        $directory = storage_path('app/telegram-exports');
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $filename = Str::slug($title) . '_' . now()->format('Ymd_His') . '.' . $format;
        $path = $directory . DIRECTORY_SEPARATOR . $filename;

        if ($format === 'pdf') {
            $this->writePdf($path, $title, $headers, $rows, $tenant);
        } else {
            $this->writeXlsx($path, $title, $headers, $rows);
        }

        return [
            'path' => $path,
            'filename' => $filename,
        ];
    }

    public function exportPurchaseRequest(PurchaseRequest $request, Tenant $tenant): array
    {
        $directory = storage_path('app/telegram-exports');
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $request->loadMissing(['requestedBy', 'items.part', 'approvals.approver']);

        $filename = 'purchase_request_' . $request->request_code . '_' . now()->format('Ymd_His') . '.pdf';
        $path = $directory . DIRECTORY_SEPARATOR . $filename;

        $logo = $this->logoData();
        $status = strtolower((string) ($request->status ?? 'draft'));
        $watermarkText = match ($status) {
            'approved' => 'APPROVED',
            'rejected' => 'REJECTED',
            'submitted' => 'SUBMITTED',
            default => 'DRAFT',
        };
        $meta = [
            'Request Code' => $request->request_code ?? '-',
            'Originator' => $request->requestedBy?->name ?? '-',
            'Created' => $request->created_at?->format('Y-m-d H:i') ?? '-',
            'Submitted' => $request->submitted_at?->format('Y-m-d H:i') ?? '-',
            'Approved' => $request->approved_at?->format('Y-m-d H:i') ?? '-',
            'Generated' => now()->format('M d, Y H:i'),
            'Approval Status' => $request->status ?? 'draft',
            'Department' => $request->department ?? '-',
            'Needed By' => $request->needed_by?->format('Y-m-d') ?? '-',
            'Priority' => $request->priority ?? 'medium',
            'Total Estimated' => $request->total_estimated_cost !== null
                ? '$' . number_format((float) $request->total_estimated_cost, 2)
                : '-',
        ];

        $items = $request->items()->with('part')->get();
        $headers = ['Line', 'SKU', 'Description', 'Qty', 'Unit', 'Est Unit Cost', 'Preferred Vendor'];
        $rows = $items->map(function ($item) {
            $estUnit = $item->est_unit_cost ?? $item->quote_amount_usd ?? null;
            return [
                $item->line_number ?? '-',
                $item->sku ?? $item->part?->sku ?? '-',
                $item->description ?? $item->item_name ?? '-',
                $item->quantity ?? 0,
                $item->unit ?? '-',
                $estUnit !== null ? $estUnit : '-',
                $item->preferred_vendor ?? '-',
            ];
        })->all();

        $approvals = $request->approvals()->with('approver')->orderBy('step')->get()->map(function ($approval) {
            return [
                'step' => $approval->step,
                'approver' => $approval->approver?->name ?? '-',
                'status' => $approval->status ?? '-',
                'decided_at' => $approval->decided_at?->format('Y-m-d H:i') ?? '-',
                'notes' => $approval->notes ?? '-',
            ];
        })->all();

        $pdf = Pdf::loadView('exports.purchase_request', [
            'title' => 'Purchase Request ' . $request->request_code,
            'tenant' => $tenant->name ?? 'Tenant',
            'meta' => $meta,
            'headers' => $headers,
            'rows' => $rows,
            'approvals' => $approvals,
            'logo' => $logo,
            'watermarkText' => $watermarkText,
            'request' => $request,
        ])->setPaper('a4', 'portrait');

        file_put_contents($path, $pdf->output());

        return [
            'path' => $path,
            'filename' => $filename,
        ];
    }

    public function exportWorkOrder(WorkOrder $order, Tenant $tenant): array
    {
        $directory = storage_path('app/telegram-exports');
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $reference = $order->reference_code ?? ('WO-' . $order->id);
        $filename = 'work_order_' . $reference . '_' . now()->format('Ymd_His') . '.pdf';
        $path = $directory . DIRECTORY_SEPARATOR . $filename;

        $order->loadMissing(['asset', 'reportedBy', 'assignedTo', 'parts.part', 'notes.createdBy']);

        $logo = $this->logoData();
        $priority = strtolower((string) ($order->priority ?? 'medium'));
        $watermarkText = match ($priority) {
            'critical' => 'PRIORITY: CRITICAL',
            'high' => 'PRIORITY: HIGH',
            'medium' => 'PRIORITY: MEDIUM',
            'low' => 'PRIORITY: LOW',
            default => 'PRIORITY',
        };
        $watermarkColor = match ($priority) {
            'critical' => 'rgba(220, 38, 38, 0.08)',
            'high' => 'rgba(249, 115, 22, 0.08)',
            'medium' => 'rgba(245, 158, 11, 0.08)',
            'low' => 'rgba(107, 114, 128, 0.08)',
            default => 'rgba(17, 24, 39, 0.08)',
        };
        [$description, $actionPlan] = $this->splitActionPlan($order->description ?? '');
        $meta = [
            'Status' => $order->status ?? '-',
            'Priority' => $order->priority ?? 'medium',
            'Type' => $order->type ?? 'preventive',
            'Due' => $order->due_at?->format('Y-m-d') ?? '-',
            'Planned Start' => $order->planned_start?->format('Y-m-d H:i') ?? '-',
            'Approval' => $order->approval_status ?? '-',
            'Estimated Cost' => $order->estimated_cost !== null ? '$' . number_format((float) $order->estimated_cost, 2) : '-',
            'Created' => $order->created_at?->format('Y-m-d H:i') ?? '-',
        ];

        $parts = $order->parts->map(function ($part) {
            return [
                $part->part?->sku ?? '-',
                $part->part?->name ?? '-',
                $part->quantity ?? 0,
                $part->unit_cost !== null ? '$' . number_format((float) $part->unit_cost, 2) : '-',
                $part->total_cost !== null ? '$' . number_format((float) $part->total_cost, 2) : '-',
            ];
        })->all();

        $notes = $order->notes->sortBy('created_at')->map(function ($note) {
            return [
                $note->created_at?->format('Y-m-d H:i') ?? '-',
                $note->type ?? 'note',
                $note->createdBy?->name ?? '-',
                $note->note ?? '-',
            ];
        })->all();

        $pdf = Pdf::loadView('exports.work_order', [
            'title' => 'Work Order ' . $reference,
            'tenant' => $tenant->name ?? 'Tenant',
            'meta' => $meta,
            'order' => $order,
            'parts' => $parts,
            'notes' => $notes,
            'logo' => $logo,
            'watermarkText' => $watermarkText,
            'watermarkColor' => $watermarkColor,
            'descriptionText' => $description,
            'actionPlan' => $actionPlan,
        ])->setPaper('a4', 'portrait');

        file_put_contents($path, $pdf->output());

        return [
            'path' => $path,
            'filename' => $filename,
        ];
    }

    private function splitActionPlan(string $description): array
    {
        $description = trim($description);
        if ($description === '') {
            return ['', []];
        }

        $parts = preg_split('/Action Plan:\\s*/i', $description, 2);
        if (count($parts) === 1) {
            return [$description, []];
        }

        $summary = trim($parts[0]);
        $planText = trim($parts[1]);
        if ($planText === '') {
            return [$summary, []];
        }

        $lines = preg_split('/\\r?\\n/', $planText);
        $actions = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }
            $line = ltrim($line, "-• \t");
            if ($line !== '') {
                $actions[] = $line;
            }
        }

        return [$summary, $actions];
    }

    private function writePdf(string $path, string $title, array $headers, array $rows, ?Tenant $tenant): void
    {
        $logo = $this->logoData();
        $meta = [
            'Generated' => now()->format('M d, Y H:i'),
            'Records' => count($rows),
        ];

        $pdf = Pdf::loadView('exports.template', [
            'title' => $title,
            'subtitle' => 'Telegram export',
            'tenant' => $tenant?->name ?? 'Tenant',
            'meta' => $meta,
            'headers' => $headers,
            'rows' => $rows,
            'logo' => $logo,
        ])->setPaper('a4', 'portrait');

        file_put_contents($path, $pdf->output());
    }

    private function writeXlsx(string $path, string $title, array $headers, array $rows): void
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle(Str::limit($title, 30, ''));

        $this->addBranding($sheet);

        $rowIndex = 4;
        $sheet->setCellValue("A{$rowIndex}", $title);
        $endColumn = $this->columnLetter(count($headers));
        $sheet->mergeCells("A{$rowIndex}:{$endColumn}{$rowIndex}");
        $sheet->getStyle("A{$rowIndex}")->getFont()->setBold(true)->setSize(14);
        $rowIndex += 2;

        $sheet->fromArray($headers, null, "A{$rowIndex}");
        $sheet->getStyle("A{$rowIndex}:{$endColumn}{$rowIndex}")->getFont()->setBold(true);
        $rowIndex++;

        foreach ($rows as $dataRow) {
            $sheet->fromArray($dataRow, null, "A{$rowIndex}");
            $rowIndex++;
        }

        foreach (range('A', $endColumn) as $column) {
            $sheet->getColumnDimension($column)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        $writer->save($path);
    }

    private function addBranding($sheet): void
    {
        $logoPath = $this->logoPath();
        if (!$logoPath) {
            return;
        }

        $logo = new Drawing();
        $logo->setName('Logo');
        $logo->setPath($logoPath);
        $logo->setHeight(40);
        $logo->setCoordinates('A1');
        $logo->setWorksheet($sheet);
    }

    private function logoPath(): ?string
    {
        $path = public_path('ui-assets/images/fourways-logo.png');
        return is_file($path) ? $path : null;
    }

    private function logoData(): ?string
    {
        $path = $this->logoPath();
        if (!$path) {
            return null;
        }

        return 'data:image/png;base64,' . base64_encode(file_get_contents($path));
    }

    private function columnLetter(int $count): string
    {
        $index = max(1, $count);
        $letter = '';
        while ($index > 0) {
            $index--;
            $letter = chr(65 + ($index % 26)) . $letter;
            $index = intdiv($index, 26);
        }
        return $letter;
    }
}
