<?php

namespace App\Services;

use App\Domain\Telegram\TelegramNotifier;
use App\Models\PurchaseRequest;
use App\Models\Tenant;
use App\Models\User;
use App\Models\WorkOrder;

class TelegramReminderService
{
    public function __construct(private TelegramNotifier $notifier)
    {
    }

    public function run(): void
    {
        if (!$this->withinReminderWindow()) {
            return;
        }

        Tenant::query()->select('id')->chunk(100, function ($tenants) {
            foreach ($tenants as $tenant) {
                $tenantId = (int) $tenant->id;
                $this->remindPurchaseRequests($tenantId);
                $this->remindWorkOrders($tenantId);
            }
        });
    }

    private function remindPurchaseRequests(int $tenantId): void
    {
        $requests = PurchaseRequest::where('tenant_id', $tenantId)
            ->where('status', 'submitted')
            ->with('approvals')
            ->limit(200)
            ->get();

        foreach ($requests as $request) {
            if (!$this->dueForReminder($request->last_reminded_at, $request->updated_at, (int) config('telegram.reminders.purchase_requests.minutes', 240))) {
                continue;
            }

            $pending = $request->approvals
                ->where('status', 'pending')
                ->sortBy('step')
                ->first();
            if (!$pending) {
                continue;
            }

            $stepLabel = ((int) $pending->step === 1) ? 'Engineering Manager' : 'Finance Manager';
            $message = "Reminder: PR {$request->request_code} awaiting {$stepLabel} approval. " .
                "Use the buttons below to approve or reject.";
            $buttons = $this->purchaseRequestActionMarkup($request->request_code);

            if ($pending->approver_id) {
                $this->notifier->notifyUser($pending->approver_id, $message, $buttons);
            }

            if ($request->requested_by) {
                $this->notifier->notifyUser($request->requested_by, $message, $buttons);
            }

            $engineering = $this->firstUserWithRole($tenantId, ['engineering manager', 'engineering', 'eng manager']);
            if ($engineering) {
                $this->notifier->notifyUser($engineering->id, $message, $buttons);
            }

            $request->update(['last_reminded_at' => now()]);
        }
    }

    private function remindWorkOrders(int $tenantId): void
    {
        $pendingApprovals = WorkOrder::where('tenant_id', $tenantId)
            ->where('approval_status', 'pending')
            ->limit(200)
            ->get();

        foreach ($pendingApprovals as $order) {
            if (!$this->dueForReminder($order->last_reminded_at, $order->updated_at, (int) config('telegram.reminders.work_orders.minutes', 180))) {
                continue;
            }

            $reference = $order->reference_code ?? ('WO-' . $order->id);
            $message = "Reminder: WO {$reference} awaiting Engineering approval. " .
                "Reply APPROVE {$reference} or REJECT {$reference} reason=\"...\".";

            if ($order->reported_by) {
                $this->notifier->notifyUser($order->reported_by, $message);
            }

            $engineering = $this->firstUserWithRole($tenantId, ['engineering manager', 'engineering', 'eng manager']);
            if ($engineering) {
                $this->notifier->notifyUser($engineering->id, $message);
            }

            $order->update(['last_reminded_at' => now()]);
        }

        $activeOrders = WorkOrder::where('tenant_id', $tenantId)
            ->whereIn('status', ['open', 'scheduled', 'in_progress'])
            ->limit(200)
            ->get();

        foreach ($activeOrders as $order) {
            if (!$this->dueForReminder($order->last_reminded_at, $order->updated_at, (int) config('telegram.reminders.work_orders.minutes', 180))) {
                continue;
            }

            $reference = $order->reference_code ?? ('WO-' . $order->id);
            $due = $order->due_at?->toDateString();
            $message = "Reminder: WO {$reference} is {$order->status}.";
            if ($due) {
                $message .= " Due {$due}.";
            }
            $message .= " Use the buttons below to take action.";

            $buttons = $this->workOrderActionMarkup($reference);
            if ($order->reported_by) {
                $this->notifier->notifyUser($order->reported_by, $message, $buttons);
            }

            if ($order->assigned_to) {
                $this->notifier->notifyUser($order->assigned_to, $message, $buttons);
            }

            if ($this->shouldEscalate($order)) {
                $escalationMessage = "Escalation: WO {$reference} is overdue and still {$order->status}.";
                foreach ($this->usersByRole($tenantId, config('telegram.reminders.work_orders.escalation_roles', ['engineering manager', 'managing director'])) as $user) {
                    $this->notifier->notifyUser($user->id, $escalationMessage, $buttons);
                }
            }

            $order->update(['last_reminded_at' => now()]);
        }
    }

    private function dueForReminder(?\DateTimeInterface $lastRemindedAt, ?\DateTimeInterface $lastUpdatedAt, int $minutes): bool
    {
        $last = $lastRemindedAt ?? $lastUpdatedAt;
        if (!$last) {
            return true;
        }

        return now()->diffInMinutes($last) >= $minutes;
    }

    private function withinReminderWindow(): bool
    {
        $hour = (int) now()->format('G');
        return $hour >= 8 && $hour <= 20;
    }

    private function shouldEscalate(WorkOrder $order): bool
    {
        $hours = (int) config('telegram.reminders.work_orders.escalate_after_hours', 24);
        if (!$order->due_at) {
            return false;
        }

        return now()->diffInHours($order->due_at, false) <= -$hours;
    }

    private function workOrderActionMarkup(string $reference): array
    {
        return [
            'inline_keyboard' => [
                [
                    ['text' => 'Accept', 'callback_data' => 'accept ' . $reference],
                    ['text' => 'Schedule', 'callback_data' => 'schedule ' . $reference],
                ],
                [
                    ['text' => 'Create PR', 'callback_data' => 'pr ' . $reference],
                    ['text' => 'Close', 'callback_data' => 'close ' . $reference],
                ],
            ],
        ];
    }

    private function usersByRole(int $tenantId, array $roles)
    {
        return User::where('tenant_id', $tenantId)
            ->whereHas('roles', function ($q) use ($roles) {
                $first = true;
                foreach ($roles as $role) {
                    if ($first) {
                        $q->where('name', 'like', '%' . $role . '%');
                        $first = false;
                        continue;
                    }
                    $q->orWhere('name', 'like', '%' . $role . '%');
                }
            })
            ->get();
    }

    private function purchaseRequestActionMarkup(string $code): array
    {
        return [
            'inline_keyboard' => [
                [
                    ['text' => 'Approve', 'callback_data' => 'approve ' . $code],
                    ['text' => 'Reject', 'callback_data' => 'reject ' . $code],
                ],
            ],
        ];
    }

    private function firstUserWithRole(int $tenantId, array $roles): ?User
    {
        foreach ($roles as $role) {
            $user = User::where('tenant_id', $tenantId)
                ->whereHas('roles', fn ($q) => $q->where('name', 'like', '%' . $role . '%'))
                ->first();
            if ($user) {
                return $user;
            }
        }

        return null;
    }
}
